/* REXX */
info.PROGRAM_STR = 'NISTIME/2 for OS/2 Warp'
info.VERSION_STR = 'Version 1.3, 15-Oct-2002'
info.COPYRGT_STR = 'Copyright (c) 2002, Pieter Bras.  All rights reserved.'
/*
  description:  Get current time from one of a list of NIST time servers
        usage:  see the routine Usage (below)
     requires:  nistime.exe
                nist-srv.lst file (from NIST), or equivalent

       author:  Pieter Bras
       e-mail:  pbras@pobox.com
*/
/*
   You may redistribute this program provided this copyright notice is
   preserved in full.

Usage:

   nistime2 options listfile

   "listfile" is the name of a file in the current directory containing
   the various NIST time servers, in the order that they should be tried.

   If "listfile" is missing then the program looks for a file named
   my-nist-srv.lst (preferred) or nist-srv.lst in the current directory.

   If there is also no file named nist-srv.lst then only the NISTIME default
   NIST time server is tried.

Options (introduced with the '-' character):

   -? or  ?    display help, then exit
   -h or -H    same as -?

   -s0   don't display any messages, and pass -s0 to nistime.exe

   all other options are passed through to nistime.exe
*/
signal on halt
signal on novalue
signal on syntax

/* initialize OS/2 RexxUtil functions */
if LoadLib( 'RexxUtil', 'SysLoadFuncs') then
   call Fatal 'RexxUtil.dll not found'

/* define non-printing ascii codes */
HTAB  = '09'X
SPACE = '20'X

msglvl = 1
protocol = ''
n_opts = ''
parse arg commandline

/* get name of file containing list of NIST servers */
defaultlist = 'my-nist-srv.lst'
serverlist = checkopts( strip( commandline))
if serverlist = '' then
   serverlist = defaultlist
if \FileExists( serverlist) then do
   call message 'Cannot find file:' serverlist
   serverlist = defaultlist
   if \FileExists( serverlist) then
      serverlist = 'nist-srv.lst'
   end
if FileExists( serverlist) then
   say 'Using file:' serverlist

SrvAddr.0 = 0                       /* number of servers */
rv = BuildServerTable( serverlist)
if SrvAddr.0 = 0 then do            /* if table empty, set up default server */
   call message 'Server list empty or not found, using default server...'
   SrvAddr.1 = ''
   SrvName.1 = '(default server)'
   SrvNote.1 = ''
   SrvLoc.1  = ''
   SrvAddr.0 = 1
   end
rv = GetSetTime()
exit rv

/* Try all NIST servers until one of them returns the correct time. */
/* return codes from nistime.exe (ver 0.3a or later):
      0        success: clock not adjusted
      1        success: clock was adjusted
      100-199  unrecoverable error
      200-299  error, recoverable? with different choice of server
      300-399  error, recoverable? with current server/different protocol
      400-...  error, recoverable? with current server/same protocol
*/
GetSetTime:
   do i = 1 to SrvAddr.0
      if pos( '3', SrvNote.i) > 0 then
         iterate                       /* test server, do not use */
      call message ''
      call message 'Trying:' SrvName.i
      if SrvAddr.i = '' then
         SrvAddr.i = SrvName.i
      rv = nistime( SrvAddr.i)
      if rv < 200 then		      /* done: success or unrecoverable err */
         leave
      if rv < 400 then				      /* try another server */
         iterate
      call message 'Trying again...'	/* try current server one more time */
      rv = nistime( SrvAddr.i)
      if rv < 200 then		      /* done: success or unrecoverable err */
         leave
      end /* do i = ... */
   if rv >= 100 then do
      call message ''
      call message 'Unsuccessful: NISTIME error' rv
      end
   return rv

/* Call nistime.exe program to set clock. Returns nistime.exe return code. */
nistime:
   /* address CMD */ '@nistime.exe -s1' n_opts arg( 1)
   return RC

/* Build table of NIST time servers. Returns: 0=success, 1=failure */
BuildServerTable:
   parse arg srvlist
   if \FileExists( srvlist) then
      return 1
   if \FileOpen( srvlist) then
      return 1
   i = SrvAddr.0
/* scan until a line starting with '$' is found */
   do while lines( srvlist)
      if left( linein( srvlist), 1) = '$' then
         leave
      end /* do while */
/* parse the lines until another line starting with '$' is found */
   do while lines( srvlist)
      server = translate( linein( srvlist), SPACE, HTAB)
      if left( server, 1) = '$' then
         leave
      i = i + 1
      parse var server SrvName.i SrvAddr.i SrvNote.i SrvLoc.i .
      end /* do while */
   SrvAddr.0 = i
   rv = FileClose( srvlist)
   return 0

message:
   if msglvl = 1 then
      say arg( 1)
   return

/**************************************/
/* file utility routines */

/* rv = FileExists( fileName) */         /* TRUE if exists, FALSE otherwise */
FileExists:
   return (stream( arg( 1), 'C', 'QUERY EXISTS') \= '')

/* rv = FileOpen( fileName, fileMode) */     /* TRUE/FALSE if success/error */
FileOpen:
   return abbrev( 'READY:', stream( arg( 1), 'C', 'OPEN' arg( 2)))

/* rv = FileClose( fileName) */              /* TRUE/FALSE if success/error */
FileClose:
   return abbrev( 'READY:', stream( arg( 1), 'C', 'CLOSE'))

FileDelete:                                  /* TRUE/FALSE if success/error */
   return (SysFileDelete( arg( 1)) = 0)

/* Conditionally load a DLL and register its functions.
   returns: 1 if error, 0 otherwise
*/
LoadLib: procedure
   parse arg lib, proc
   if RxFuncQuery( proc) <> 0 then do
      call RxFuncAdd proc, lib, proc
      str = 'call' proc
      signal on syntax name LoadLibFail
      interpret str
      end
   return 0
      
LoadLibFail:
   call RxFuncDrop proc
   return 1

/**************************************/
/* the user interface */

checkopts:
   parse arg optlist
   if left( optlist, 1) = '?' then
      signal Usage
   do while left( optlist, 1) = '-'
      parse var optlist switch optlist
      if left( switch, 2) = '-?' | left( translate( switch), 2) = '-H' then
         signal Usage
      if left( switch, 3) = '-m0' then
         msglvl = 0
      if left( switch, 3) = '-D' then
         protocol = 'D'
      if left( switch, 3) = '-N' then
         protocol = 'N'
      n_opts = n_opts switch              /* pass to nistime.exe */
      end
   return strip( optlist)

Usage:
   say info.PROGRAM_STR '   ' info.VERSION_STR
   say info.COPYRGT_STR
   say ''
   say 'Usage:  nistime2 [options] [serverlist]'
   say ''
   say '   where "serverlist" (optional) is a preferred list of NIST servers;'
   say '   if not found, program will look for a file named "nist-srv.lst".'
   say ''
   say 'Options:'
   say ''
   say '   -? or  ?    display this help message'
   say '   -h or -H    same as -?'
   say "   -s0         don't display messages (also passed to nistime.exe)"
   say ''
   say '   all other options are passed through to nistime.exe'
   exit 0

/**************************************/
/* and just in case... */

Fatal:
   say 'ERROR --' arg( 1)
   exit 1

halt:
   say 'Program halted by operator'
   exit 1

novalue:
   say 'Uninitialized variable on line:' sigl
   say sourceline( sigl)
   exit 1

syntax:
   say 'Syntax error on line:' sigl  ' error code:' rc
   say sourceline( sigl)
   exit 1

/* <eof> */
