Unit ACLDialogs;
// A variety of useful message box functions, including displaying
// lists. They all allow specification of a caption.

 Interface

Uses
  Classes,
{$ifdef os2}
  ACLMessageForm
{$else}
  Controls,
  MessageFormWin
{$endif}
  ;

// Shows a message box
Procedure DoMessageDlg( Caption: String;
                        Message: string );

// Shows an error message box
Procedure DoErrorDlg( Caption: String;
                      Message: string );

// Shows a warning message box
Procedure DoWarningDlg( Caption: String;
                        Message: string );

// Shows a long message box
Procedure DoLongMessageDlg( Caption: String;
                            Message: PChar );

// Shows a dialog with OK and cancel buttons
// Returns true if OK pressed
Function DoConfirmDlg( Caption: string;
                       Message: string ): boolean;

// Returns true if confirmation given.
Function DoConfirmListDlg( Caption: string;
                           Message: string;
                           List: TStrings ): boolean;

// Shows a message containing a list
Procedure DoMessageListDlg( Caption: string;
                            Message: string;
                            List: TStrings );

// Shows an error message containing a list
Procedure DoErrorListDlg( Caption: string;
                          Message: string;
                          List: TStrings );

// Returns true if yes is clicked
Function DoYesNoListDlg( Caption: string;
                         Message: string;
                         List: TStrings ): boolean;

// Returns true if yes is clicked
Function DoYesNoDlg( Caption: string;
                     Message: string ): boolean;

// Shows a message containing a list
// with a help callback for items in the list
Procedure DoMessageListHelpDlg( Caption: string;
                                Message: string;
                                List: TStrings;
                                ListHelpCallback: TListHelpCallback );

{$ifdef os2}
Function DoInputQuery( Const ACaption: String;
                       Const APrompt: String;
                       Var Value: String ): Boolean;

{$endif}

Implementation

uses
  SysUtils,
  Forms, Dialogs, StdCtrls, Buttons,
  ACLStringUtility
{$ifdef os2}
  , ACLLanguageUnit
{$endif}
  ;

// -------------------------------------------------

Function DoACLDialog( const Caption: string;
                      const Message: string;
                      const List: TStrings;
                      const ShowCancel: boolean;
                      const IconType: TMessageIconType;
                      const ShowHelp: boolean;
                      const UseYesNo: boolean;
                      const ListHelpCallBack: TListHelpCallBack ): TModalResult;
var
  TheDialog: TMessageForm;
  PMessage: PChar;
Begin
  TheDialog := TMessageForm.Create( nil );
  TheDialog.Caption := Caption;
  PMessage := StrDupPas( Message );
  TheDialog.TheText := PMessage;

  TheDialog.ShowCancel := ShowCancel;
  TheDialog.IconType := IconType;
  TheDialog.ShowHelp := ShowHelp;
  TheDialog.UseYesNo := UseYesNo;
  TheDialog.ListHelpCallBack := ListHelpCallBack;

  if List <> nil then
  begin
    TheDialog.ShowList := true;
    TheDialog.ListBox.Items.Assign( List );
  end
  else
  begin
    TheDialog.ShowList := false;
  end;

  Result := TheDialog.ShowModal;

  TheDialog.Destroy;
  StrDispose( PMessage );
end;

Procedure DoMessageListDlg( Caption: string;
                            Message: string;
                            List: TStrings );
Begin
  DoACLDialog( Caption,
               Message,
               List,
               false, // show cancel
               mitInfo,
               false, // show help
               false, // use yes no
               nil
               );
End;

Procedure DoMessageListHelpDlg( Caption: string;
                                Message: string;
                                List: TStrings;
                                ListHelpCallback: TListHelpCallback );
Begin
  DoACLDialog( Caption,
               Message,
               List,
               false, // show cancel
               mitInfo,
               true, // show help
               false, // use yes no
               ListHelpCallBack
               );
End;

Function DoConfirmListDlg( Caption: string;
                           Message: string;
                           List: TStrings ): boolean;
Begin
  Result :=
    DoACLDialog( Caption,
                 Message,
                 List,
                 true, // show cancel
                 mitQuestion,
                 false, // show help
                 false, // use yes no
                 nil // no help callback
               ) = mrOK;
End;

Function DoConfirmDlg( Caption: string;
                       Message: string ): boolean;
Begin
  Result :=
    DoACLDialog( Caption,
                 Message,
                 nil, // no list
                 true, // show cancel
                 mitQuestion,
                 false, // show help
                 false, // use yes no
                 nil // no help callback
               ) = mrOK;
End;

Procedure DoLongMessageDlg( Caption: String;
                            Message: PChar );
Var
  TheDialog: TMessageForm;
Begin
  TheDialog := TMessageForm.Create( nil );
  TheDialog.Caption := Caption;
  TheDialog.TheText := Message;

  TheDialog.ShowList := false;
  TheDialog.ShowCancel := false;
  TheDialog.IconType := mitInfo;
  TheDialog.ShowHelp := false;
  TheDialog.UseYesNo := false;

  TheDialog.ShowModal;
  TheDialog.Destroy;
End;

Procedure DoMessageDlg( Caption: String;
                        Message: string );
Begin
  DoACLDialog( Caption,
               Message,
               nil, // no list
               false, // show cancel
               mitInfo,
               false, // show help
               false, // use yes no
               nil // no help callback
             );
End;

Procedure DoErrorListDlg( Caption: string;
                          Message: string;
                          List: TStrings );
Begin
  DoACLDialog( Caption,
               Message,
               List,
               false, // show cancel
               mitError,
               false, // show help
               false, // use yes no
               nil // no help callback
             );
End;

Procedure DoErrorDlg( Caption: String;
                      Message: string );
Begin
  DoACLDialog( Caption,
               Message,
               nil, // no list
               false, // show cancel
               mitError,
               false, // show help
               false, // use yes no
               nil // no help callback
             );
End;

Procedure DoWarningDlg( Caption: String;
                        Message: string );
begin
  DoACLDialog( Caption,
               Message,
               nil, // no list
               false, // show cancel
               mitWarning,
               false, // show help
               false, // use yes no
               nil // no help callback
             );
end;

Function DoYesNoListDlg( Caption: string;
                         Message: string;
                         List: TStrings ): boolean;
Begin
  Result :=
    DoACLDialog( Caption,
                 Message,
                 List,
                 true, // show "cancel" (=no)
                 mitQuestion,
                 false, // show help
                 true, // use yes no
                 nil // no help callback
               ) = mrYes;
End;

Function DoYesNoDlg( Caption: string;
                     Message: string ): boolean;
Begin
  Result :=
    DoACLDialog( Caption,
                 Message,
                 nil, // no list
                 true, // show "cancel" (=no)
                 mitQuestion,
                 false, // show help
                 true, // use yes no
                 nil // no help callback
               ) = mrYes;
End;

{$ifdef os2}

var
  OKButtonCaption: string;
  CancelButtonCaption: string;

Procedure OnLanguageEvent( Language: TLanguageFile;
                           const Apply: boolean );
begin
  Language.Prefix := 'TextInputForm.';

  Language.LL( Apply, OKButtonCaption, 'OKButtonCaption', '~OK' );
  Language.LL( Apply, CancelButtonCaption, 'CancelButtonCaption', '~Cancel' );
end;

// This is a copy of InputQuery from SPCC with the following changes:
// - slight layout mods
// - normal buttons instead of bitmap buttons
// - buttons are centered like my other popup dialogs
Function DoInputQuery( Const ACaption: String;
                       Const APrompt: String;
                       Var Value: String ): Boolean;
Var
  Dlg: TDialog;
  FEdit: TEdit;
  Btn: TButton;
Begin
  Dlg := TDialog.Create( Screen.ActiveForm );
  Dlg.Name := 'TextInputForm';

  Dlg.ClientWidth := 340;
  Dlg.ClientHeight := 120;
  Dlg.Caption := ACaption;

  Btn := InsertButton( Dlg, 75, 10, 90, 30, OKButtonCaption, '' );
  Btn.ModalResult := mrOK;
  Btn.Default := true;

  Btn := InsertButton( Dlg, 175, 10, 90, 30, CancelButtonCaption, '' );
  Btn.ModalResult := mrCancel;
  Btn.Cancel := true;

  InsertLabel( Dlg, 10, 90, 320, 20, APrompt );

  FEdit := TEdit.Create( Dlg );
  FEdit.SetWindowPos( 10, 60, 320, 20 );
  FEdit.Text := Value;
  FEdit.parent := Dlg;
  FEdit.Focus;

  Dlg.Execute;
  Case Dlg.ModalResult Of
     cmOk:
     Begin
       Value := FEdit.Text;
       Result := True;
     End;

     Else
     Begin
       Result := False;
     End;
  End; {Case}

  Dlg.Destroy;
End;

{$endif}

Initialization
{$ifdef os2}
   RegisterProcForLanguages( OnLanguageEvent );
{$endif}
End.
