unit ControlsUtility;

interface

uses
  Classes, StdCtrls, ExtCtrls,
{$ifdef win32}
  checklst, Controls, ComCtrls, Windows, Messages, Graphics;
{$else}
  Forms, CustomMemo, CustomCheckListBox;
{$endif}

const
{$ifdef win32}
  Mnem = '&';
{$else}
  Mnem = '~';
{$endif}

// Checklistbox utility functions
type
{$ifdef win32}
  TCheckListBoxType = TCheckListBox;
  TListBoxType = TCustomListBox;
  TMemoType = TMemo;
{$else}
  TCheckListBoxType = TCustomCheckListBox;
  TListBoxType = TListBox;
  TMemoType = TCustomMemo;
{$endif}

{$ifdef os2}
function GetNiceDefaultFont: TFont;
{$endif}

procedure StartTimer( Timer: TTimer );
procedure StopTimer( Timer: TTimer );

{$ifdef win32}
procedure SetFocusTo( Control: TWinControl );
{$else}
procedure SetFocusTo( Control: TControl );
{$endif}

procedure AddSelectedMemoText( Memo: TMemoType;
                               Text: string );

Procedure GetCheckedItems( CheckListBox: TCheckListBoxType;
                           Items: TStrings );
Procedure GetCheckedObjects( CheckListBox: TCheckListBoxType;
                             Objects: TList );
Procedure AddCheckedItems( CheckListBox: TCheckListBoxType;
                           Items: TStrings );
Procedure AddCheckedObjects( CheckListBox: TCheckListBoxType;
                             Objects: TList );
Procedure AddCheckListItemObject( CheckListBox: TCheckListBoxType;
                                  Text: String;
                                  TheObject: TObject;
                                  Checked: boolean );
Procedure SetAllCheckListItems( CheckListBox: TCheckListBoxType;
                                Value: boolean );
Function CheckListItemCount( CheckListBox: TCheckListBoxType ): integer;
Function CheckListObject( CheckListBox: TCheckListBoxType;
                          Index: integer ): TObject;
Function SelectedCheckListObject( CheckListBox: TCheckListBoxType ): TObject;
Function SelectedCheckListItem( CheckListBox: TCheckListBoxType ): string;
Function CheckedCount( CheckListBox: TCheckListBoxType ): integer;

// Listbox utility functions
Function SelectedObject( ListBox: TListBoxType ): TObject;
Procedure SetSelectedByObject( ListBox: TListBoxType;
                               SelectObject: TObject );
Function SelectedItem( ListBox: TListBoxType ): string;
Procedure GetSelectedItems( ListBox: TListBoxType;
                            Dest: TStrings );

{$ifdef win32}
Procedure AddBoldedLine( RichEdit: TRichEdit;
                         BoldPart: string;
                         NormalPart: string );

Procedure GetExpandedNodes( TreeView: TTreeView;
                            ExpandedNodeList: TStrings );

{$endif}


implementation

{$ifdef os2}
function GetNiceDefaultFont: TFont;
begin
  // try warpsans.9
  Result := Screen.GetFontFromPointSize( 'WarpSans', 9 );
  if Result <> nil then
    // ok use warpsans
    exit;

  // try Helv.8
  Result := Screen.GetFontFromPointSize( 'Helv', 8 );
  if Result <> nil then
    // ok use helv
    exit;

  // Ok, system default
  Result := Screen.DefaultFont;
end;
{$endif}

procedure StartTimer( Timer: TTimer );
begin
{$ifdef win32}
  Timer.Enabled := true;
{$else}
  Timer.Start;
{$endif}
end;

procedure StopTimer( Timer: TTimer );
begin
{$ifdef win32}
  Timer.Enabled := false;
{$else}
  Timer.Stop;
{$endif}
end;

procedure AddSelectedMemoText( Memo: TMemoType;
                               Text: string );
{$ifdef win32}
var
  StartPos: integer;
{$endif}
begin
{$ifdef win32}
  StartPos := Length( Memo.Lines.Text );
  Memo.Lines.Add( Text );
  Memo.SelStart := StartPos;
  Memo.SelLength := Length( Memo.Lines.Text ) - StartPos;
{$else}
  Memo.AddSelectedLine( Text );
{$endif}
end;

{$ifdef win32}
procedure SetFocusTo( Control: TWinControl );
begin
   Control.SetFocus;
end;
{$else}
procedure SetFocusTo( Control: TControl );
begin
   Control.Focus;
end;
{$endif}

{$ifdef win32}
Procedure AddCheckedItems( CheckListBox: TCheckListBoxType;
                           Items: TStrings );
var
  i: integer;
begin
  for i:= 0 to CheckListBox.Items.Count - 1 do
    if CheckListBox.Checked[ i ] then
      Items.AddObject( CheckListBox.Items[ i ],
                       CheckListBox.Items.Objects[ i ] );
end;

Procedure AddCheckedObjects( CheckListBox: TCheckListBoxType;
                             Objects: TList );
var
  i: integer;
begin
  for i:= 0 to CheckListBox.Items.Count - 1 do
    if CheckListBox.Checked[ i ] then
      Objects.Add( CheckListBox.Items.Objects[ i ] );
end;

Procedure GetCheckedItems( CheckListBox: TCheckListBoxType;
                           Items: TStrings );
begin
  Items.Clear;
  AddCheckedItems( CheckListBox, Items );
end;

Procedure GetCheckedObjects( CheckListBox: TCheckListBoxType;
                             Objects: TList );
begin
  Objects.Clear;
  AddCheckedObjects( CheckListBox, Objects );
end;

Procedure AddCheckListItemObject( CheckListBox: TCheckListBoxType;
                                  Text: String;
                                  TheObject: TObject;
                                  Checked: boolean );
var
  AddPosition: integer;
begin
  AddPosition:= CheckListBox.Items.AddObject( Text, TheObject );
  CheckListBox.Checked[ AddPosition ]:= Checked;
end;

Procedure SetAllCheckListItems( CheckListBox: TCheckListBoxType;
                                Value: boolean );
var
  i: integer;
begin
{$ifdef os2}
  CheckListBox.BeginUpdate;
{$endif}
  for i:= 0 to CheckListBox.Items.Count - 1 do
    CheckListBox.Checked[ i ] := Value;
{$ifdef os2}
  CheckListBox.EndUpdate;
{$endif}
end;

Function CheckListItemCount( CheckListBox: TCheckListBoxType ): integer;
begin
  Result:= CheckListBox.Items.Count;
end;

Function CheckListObject( CheckListBox: TCheckListBoxType;
                          Index: integer ): TObject;
begin
  Result:= CheckListBox.Items.Objects[ Index ];
end;

Function SelectedCheckListObject( CheckListBox: TCheckListBoxType ): TObject;
begin
  if CheckListBox.ItemIndex <> -1 then
    Result:= CheckListBox.Items.Objects[ CheckListBox.ItemIndex ]
  else
    Result:= nil;
end;

Function SelectedCheckListItem( CheckListBox: TCheckListBoxType ): string;
begin
  if CheckListBox.ItemIndex <> -1 then
    Result:= CheckListBox.Items[ CheckListBox.ItemIndex ]
  else
    Result:= '';
end;

Function CheckedCount( CheckListBox: TCheckListBoxType ): integer;
var
  i: integer;
begin
  Result:= 0;
  for i:= 0 to CheckListBox.Items.Count - 1 do
    if CheckListBox.Checked[ i ] then
      inc( Result );
end;

{$else}
Procedure AddCheckedItems( CheckListBox: TCustomCheckListBox;
                           Items: TStrings );
begin
  CheckListBox.AddCheckedItems( Items );
end;

Procedure GetCheckedObjects( CheckListBox: TCheckListBoxType;
                             Objects: TList );
var
  i: integer;
begin
  for i:= 0 to CheckListBox.Items.Count - 1 do
    if CheckListBox.Checked[ i ] then
      Objects.Add( CheckListBox.Objects[ i ] );
end;

Procedure GetCheckedItems( CheckListBox: TCustomCheckListBox;
                           Items: TStrings );
begin
  Items.Clear;
  AddCheckedItems( CheckListBox, Items );
end;

Procedure AddCheckedObjects( CheckListBox: TCheckListBoxType;
                             Objects: TList );
begin
  Objects.Clear;
  AddCheckedObjects( CheckListBox, Objects );
end;

Procedure AddCheckListItemObject( CheckListBox: TCustomCheckListBox;
                                  Text: String;
                                  TheObject: TObject;
                                  Checked: boolean );
begin
  CheckListBox.AddItemObject( Text, TheObject, Checked );
end;

Procedure SetAllCheckListItems( CheckListBox: TCustomCheckListBox;
                                Value: boolean );
var
  i: integer;
begin
  CheckListBox.BeginUpdate;
  for i:= 0 to CheckListBox.Items.Count - 1 do
    CheckListBox.Checked[ i ] := Value;
  CheckListBox.EndUpdate;
end;

Function CheckListItemCount( CheckListBox: TCustomCheckListBox ): integer;
begin
  Result:= CheckListBox.ItemCount;
end;

Function CheckListObject( CheckListBox: TCustomCheckListBox;
                          Index: integer ): TObject;
begin
  Result:= CheckListBox.Objects[ Index ];
end;

Function SelectedCheckListObject( CheckListBox: TCustomCheckListBox ): TObject;
begin
  Result:= CheckListBox.SelectedObject;
end;

Function SelectedCheckListItem( CheckListBox: TCustomCheckListBox ): string;
begin
  Result:= CheckListBox.SelectedString;
end;

Function CheckedCount( CheckListBox: TCustomCheckListBox ): integer;
begin
  Result:= CheckListBox.CheckedCount;
end;

{$endif}


Function SelectedObject( ListBox: TListBoxType ): TObject;
begin
  if ( ListBox.ItemIndex >= 0 )
     and ( ListBox.ItemIndex < ListBox.Items.Count ) then
    Result:= ListBox.Items.Objects[ ListBox.ItemIndex ]
  else
    Result:= nil;
end;

Procedure SetSelectedByObject( ListBox: TListBoxType;
                               SelectObject: TObject );
var
  Index: integer;
begin
  Index := ListBox.Items.IndexOfObject( SelectObject );
  ListBox.ItemIndex := Index;
end;

Function SelectedItem( ListBox: TListBoxType ): string;
begin
  if ( ListBox.ItemIndex >= 0 )
     and ( ListBox.ItemIndex < ListBox.Items.Count ) then
    Result:= ListBox.Items[ ListBox.ItemIndex ]
  else
    Result:= '';
end;

Procedure GetSelectedItems( ListBox: TListBoxType;
                            Dest: TStrings );
var
  i: integer;
begin
  for i:= 0 to ListBox.Items.Count - 1 do
    if ListBox.Selected[ i ] then
      Dest.AddObject( ListBox.Items[ i ],
                      ListBox.Items.Objects[ i ] );
end;

{$ifdef win32}
const
  EndLineStr = #13 +#10;
  
Procedure AddBoldedLine( RichEdit: TRichEdit;
                         BoldPart: string;
                         NormalPart: string );
var
  LineStart: integer;
  Dummy: integer;
begin
  with RichEdit do
  begin
    SendMessage( Handle,
                 EM_GETSEL,
                 Longint( Addr( LineStart ) ),
                 Longint( Addr( Dummy ) ) );

    SendMessage( Handle,
                 EM_REPLACESEL,
                 0,
                 Longint(PChar( BoldPart)));

    SelStart:= LineStart;
    SelLength:= Length( BoldPart );
    SelAttributes.Style:= [ fsBold ];

    SelStart:= LineStart + Length( BoldPart );
    SelLength:= 0;
    SendMessage( Handle,
                 EM_REPLACESEL,
                 0,
                 Longint(PChar( NormalPart)));
    SelStart:= LineStart + Length( BoldPart );
    SelLength:= Length( NormalPart );
    SelAttributes.Style:= [];
    SelStart:= LineStart + Length( BoldPart )
               + Length( NormalPart );
    SelLength:= 0;
    SendMessage( Handle,
                 EM_REPLACESEL,
                 0,
                 Longint(PChar( EndLineStr )));

  end;

end;

Procedure GetExpandedNodesCumulative( Node: TTreeNode;
                                      ExpandedNodeList: TStrings;
                                      const Path: string );
var
  SubNode: TTreeNode;
begin
  if Node.Expanded then
  begin
    ExpandedNodeList.Add( Path + Node.Text );
    
    SubNode := Node.getFirstChild;
    while SubNode <> nil do
    begin
      GetExpandedNodesCumulative( SubNode,
                                  ExpandedNodeList,
                                  Path + Node.Text + '\' );
      SubNode := SubNode.getNextSibling;
    end;
  end;
end;

Procedure GetExpandedNodes( TreeView: TTreeView;
                            ExpandedNodeList: TStrings );
begin
  ExpandedNodeList.Clear;
  if TreeView.Items.GetFirstNode = nil then
    exit;
  GetExpandedNodesCumulative( TreeView.Items.GetFirstNode,
                              ExpandedNodeList,
                              '' );
end;

{$endif}

end.
