Unit CustomFontDialog;

Interface

Uses
  Classes, Forms, Dialogs, StdCtrls, Buttons, Graphics;

Type

    TCustomFontDialog=Class(TDialog)
      Private
         FNameList: TListBox;
         FNameLabel: TLabel;
         FSizeCombo: TComboBox;
         FSizeLabel: TLabel;
         FExampleText: TEdit;
         FItalicCheck:TCheckBox;
         FStyleGroup: TGroupBox;
         FBoldCheck:TCheckBox;
         FOutlineCheck:TCheckBox;
         FUnderscoreCheck:TCheckBox;
         FStrikeOutCheck:TCheckBox;
         FOKButton:TButton;
         FCancelButton:TButton;
         FEditFont: TFont;

         Function GetFaceName: String;
         Procedure SetFaceName( s: String );

         Function GetPointSize: LongInt;
         Procedure SetPointSize( Value:LongInt );

         Function GetAttributes: TFontAttributes;
         Procedure SetAttributes( NewValue: TFontAttributes );

         Procedure InsertSizes;
         Procedure OnSelectFace( Sender: TObject; Index: LongInt );
         Procedure OnSetSize( Sender: TObject );
         Procedure OnStyleChanged( Sender:TObject );

         Procedure OnSizeClick( Sender: TObject; Index: longint );

         Function GetEditFont: TFont;
         Procedure SetEditFont( NewFont:TFont );

         Function FindFace( FaceName: string ): TFont;

      Protected
         Procedure SetupComponent; Override;
         Procedure SetupShow; Override;
         Procedure LayoutControls;
         Procedure Resize; override;
         Procedure SelectFont;
      public
         Property FaceName: String read GetFaceName write SetFaceName;
         Property PointSize: longint read GetPointSize write SetPointSize;
         Property Attributes: TFontAttributes read GetAttributes write SetAttributes;
         Property EditFont: TFont Read GetEditFont Write SetEditFont;
    End;

Exports
  TCustomFontDialog, 'User', 'CustomFontDialog.bmp';

Implementation

uses
  SysUtils, PmWin,
  ACLStringUtility;

// Returns true if s ends with endstr (case insensitive)
Function StringAtEnd( const endStr: string; const s: string ): boolean;
Var
  i, j: integer;
Begin
  Result := false;
  if Length( s ) < length( endStr ) then
    exit;
  j := Length( s );
  for i := length( endstr ) downto 1 do
  begin
    if UpCase( s[ j ] ) <> UpCase( endStr[ i ] ) then
      exit;
    dec( j );
  end;
  Result := true;
End;

// Returns S minus count characters from the right
Function StringLeftWithout( const S:string; const count:integer ):string;
Begin
  Result := copy( S, 1, length( S )-count );
End;

Function StripBoldItalic( Const FaceName: string ): String;
Begin
  Result := Trim( FaceName );
  if StringAtEnd( 'Italic', Result ) then
    Result := StringLeftWithout( Result, Length( 'Italic' ) );
  Result := Trim( Result );
  if StringAtEnd( 'Bold', Result ) then
    Result := StringLeftWithout( Result, Length( 'Bold' ) );
  Result := Trim( Result );
end;

Procedure TCustomFontDialog.SetupComponent;
Var
     FontIndex: LongInt;
     aFont: TFont;
Begin
     Inherited SetupComponent;

     BorderStyle := bsSizeable;
     BorderIcons := [ biSystemMenu, biMaximize ];

     Caption := LoadNLSStr( SSelectAFont );
     Width := 480;
     Height := 350;
     MinTrackHeight := 350;
     MinTrackWidth := 250;

     // Give controls names for purposes of language support...

     FNameList := TListBox.Create( self );
     FNameList.Parent := self;
     FNameList.Sorted := true;
     FNameList.Duplicates := false;
     FNameList.OnItemFocus := OnSelectFace;
     FNameList.Name := 'NameList';

     FNameLabel := TLabel.Create( self );
     FNameLabel.Parent := self;
     FNameLabel.Caption := LoadNLSStr( SName )+':';
     FNameLabel.Name := 'NameLabel';

     FSizeCombo := TComboBox.Create( self );
     FSizeCombo.Style := csSimple;
     FSizeCombo.Parent := self;
     FSizeCombo.OnChange := OnSetSize;
     FSizeCombo.OnItemFocus := OnSizeClick;
     FSizeCombo.Name := 'SizeCombo';

     FSizeLabel := TLabel.Create( self );
     FSizeLabel.Parent := self;
     FSizeLabel.Caption := LoadNLSStr( SSize )+':';
     FSizeLabel.Name := 'SizeLabel';

     FExampleText := TEdit.Create( self );
     FExampleText.Parent := self;
     FExampleText.Text := 'abc ABC def DEF';
     FExampleText.Autosize := false;
     FExampleText.Name := 'ExampleText';

     FStyleGroup := TGroupBox.Create( self );
     FStyleGroup.Parent := self;
     FStyleGroup.Caption := LoadNLSStr( SStyle );
     FStyleGroup.Name := 'StyleGroup';

     FItalicCheck := TCheckBox.Create( self );
     FItalicCheck.Parent := FStyleGroup;
     FItalicCheck.OnClick := OnStyleChanged;
     FItalicCheck.Caption := LoadNLSStr( SItalic );
     FItalicCheck.Name := 'ItalicCheck';

     FBoldCheck := TCheckBox.Create( self );
     FBoldCheck.Parent := FStyleGroup;
     FBoldCheck.OnClick := OnStyleChanged;
     FBoldCheck.Caption := LoadNLSStr( SBold );
     FBoldCheck.Name := 'BoldCheck';

     FOutlineCheck := TCheckBox.Create( self );
     FOutlineCheck.Parent := FStyleGroup;
     FOutlineCheck.OnClick := OnStyleChanged;
     FOutlineCheck.Caption := LoadNLSStr( SOutline );
     FOutlineCheck.Name := 'OutlineCheck';

     FStrikeOutCheck := TCheckBox.Create( self );
     FStrikeOutCheck.Parent := FStyleGroup;
     FStrikeOutCheck.OnClick := OnStyleChanged;
     FStrikeOutCheck.Caption := LoadNLSStr( SStrikeOut );
     FStrikeOutCheck.Name := 'StrikeOutCheck';

     FUnderscoreCheck := TCheckBox.Create( self );
     FUnderscoreCheck.Parent := FStyleGroup;
     FUnderscoreCheck.OnClick := OnStyleChanged;
     FUnderscoreCheck.Caption := LoadNLSStr( SUnderscore );
     FUnderscoreCheck.Name := 'UnderscoreCheck';

     For FontIndex := 0 To Screen.FontCount - 1 Do
     Begin
          aFont := Screen.Fonts[ FontIndex ];
          FNameList.Items.Add( StripBoldItalic( aFont.FaceName ) );
     End;

     FOKButton := InsertButton( Self, 150, 10, 90, 30,
                                LoadNLSStr( SOkButton ),
                                '' );
     FOKButton.Name := 'OKButton';
     FOKButton.Default := true;
     FOKButton.ModalResult := mrOK;

     FCancelButton := InsertButton( Self, 250, 10, 90, 30,
                                    LoadNLSStr( SCancelButton ),
                                    '' );
     FCancelButton.Name := 'CancelButton';
     FCancelButton.Cancel := true;
     FCancelButton.ModalResult := mrCancel;

     LayoutControls;

     SetEditFont( Screen.DefaultFont );
End;

Function TCustomFontDialog.FindFace( FaceName: string ): TFont;
Var
     FontIndex: LongInt;
     aFont: TFont;
begin
     For FontIndex := 0 To Screen.FontCount - 1 Do
     Begin
          aFont := Screen.Fonts[ FontIndex ];
          if CompareText( aFont.FaceName, FaceName ) = 0 then
          begin
            Result := aFont;
            exit;
          end;
     End;
     Result := nil;
end;

Procedure TCustomFontDialog.LayoutControls;
Var
     W: longint;
     H: longint;
     ExampleH: longint;
     VSplit: longint;
     GrpH: longint;
     TopH: longint;
Begin
     W := ClientWidth;
     H := ClientHeight;

     // Example is minimum 40 pixels, or 20% of height
     ExampleH := 40;
     if ( H div 5 ) > ExampleH then
       ExampleH := H div 5;

     // Base of name/size/style (45 allows for buttons/spacing)
     VSplit := 45 + ExampleH;
     TopH := H - VSplit - 25;

     GrpH := TopH + 20;
     //                             Left     Bottom      Width    Height
     FNameLabel      .SetWindowPos( 5,       H - 25,     W - 200, 20      );
     FSizeLabel      .SetWindowPos( W - 190, H - 25,     60,      20      );

     FNameList       .SetWindowPos( 5,       VSplit,     W - 200, TopH    );
     FSizeCombo      .SetWindowPos( W - 190, VSplit,     60,      TopH    );
     FStyleGroup     .SetWindowPos( W - 125, VSplit,     120,     GrpH    );

     FExampleText    .SetWindowPos( 5,       40,         W - 10,  ExampleH );

     FOKButton       .SetWindowPos( W - 170, 5,          80,      30      );
     FCancelButton   .SetWindowPos( W - 85,  5,          80,      30      );


     FItalicCheck    .SetWindowPos( 5,       GrpH - 40,  100,     20      );
     FBoldCheck      .SetWindowPos( 5,       GrpH - 60,  100,     20      );

     FOutlineCheck   .SetWindowPos( 5,       GrpH - 90,  100,     20      );
     FStrikeOutCheck .SetWindowPos( 5,       GrpH - 110, 100,     20      );
     FUnderscoreCheck.SetWindowPos( 5,       GrpH - 130, 100,     20      );

End;


Procedure TCustomFontDialog.SetupShow;
Begin
     Inherited SetupShow;

     FNameList.Focus;
     FOKButton.Default := True;
End;

Procedure TCustomFontDialog.Resize;
begin
  inherited Resize;
  LayoutControls;
end;

Function GetFontNameWithStyle( Const FontName: String;
                               Const Attrs: TFontAttributes ): String;
Begin
     Result := FontName;
     If faItalic in Attrs Then
        Result:= Result + '.Italic';
     If faBold in Attrs Then
        Result := Result + '.Bold';
     If faOutline in Attrs Then
        Result := Result + '.Outline';
     If faStrikeOut in Attrs Then
        Result := Result + '.Strikeout';
     If faUnderScore in Attrs Then
        Result := Result + '.Underscore';
End;

Function TCustomFontDialog.GetFaceName: string;
begin
  if FNameList.ItemIndex = -1 then
    Result := ''
  else
    Result := FNameList.Items[ FNameList.ItemIndex ];
end;

Procedure TCustomFontDialog.SetFaceName( s: string );
begin
  FNameList.ItemIndex := FNameList.Items.IndexOf( s );
end;

Function TCustomFontDialog.GetAttributes: TFontAttributes;
Begin
     Result :=[];
     If FItalicCheck.Checked Then
       Include( Result, faItalic );
     If FBoldCheck.Checked Then
       Include( Result, faBold );
     If FOutlineCheck.Checked Then
       Include( Result, faOutline );
     If FStrikeOutCheck.Checked Then
       Include( Result, faStrikeOut );
     If FUnderscoreCheck.Checked Then
       Include( Result, faUnderScore );
End;

Procedure TCustomFontDialog.SetAttributes( NewValue: TFontAttributes );
Begin
     FBoldCheck.Checked := faBold in NewValue;
     FItalicCheck.Checked := faItalic in NewValue;
     FOutlineCheck.Checked := faOutline in NewValue;
     FStrikeoutCheck.Checked := faStrikeout in NewValue;
     FUnderscoreCheck.Checked := faUnderscore in NewValue;
End;

Function TCustomFontDialog.GetPointSize: LongInt;
Var
  S: String;
  C: Integer;
Begin
  S := FSizeCombo.Text;
  Val( S, Result, C );
  If C <> 0 Then
    // invalid conversion
    Result := 0;
End;

Procedure TCustomFontDialog.SetPointSize( Value:LongInt );
Begin
  If Value = 0 Then
    Value := 8;
  FSizeCombo.Text := IntToStr( Value );
End;


Function TCustomFontDialog.GetEditFont: TFont;
var
     Attrs: TFontAttributes;
     StyleFaceName: string;
     BaseFont: TFont;
Begin
     StyleFaceName := Facename;
     Attrs := GetAttributes;

     // Before we go using attributes, we'll see if there's a bold/italic face
     if faBold in Attrs then
       StyleFaceName := StyleFaceName + ' Bold';
     if faItalic in Attrs then
       StyleFaceName := StyleFaceName + ' Italic';

     if FindFace( StyleFaceName ) <> nil then
     begin
       // yes, there is
       Exclude( Attrs, faBold );
       Exclude( Attrs, faItalic );
     end
     else
     begin
       // Nope.
       StyleFaceName := FaceName;
     end;

     BaseFont := Screen.GetFontFromPointSize( StyleFaceName,
                                              PointSize );
     if BaseFont = nil then
       BaseFont := Screen.DefaultFont;

     FEditFont := Screen.CreateCompatibleFont( BaseFont );
     FEditFont.Attributes := GetAttributes;

     Result := FEditFont;
End;

Procedure TCustomFontDialog.SelectFont;
begin
  FExampleText.Font := GetEditFont;
end;

Procedure TCustomFontDialog.SetEditFont( NewFont:TFont );
Begin
  If NewFont = Nil Then
    NewFont := Screen.DefaultFont;

  FaceName := NewFont.FaceName;
  PointSize := NewFont.PointSize;
  Attributes := NewFont.Attributes;
End;

const
  StandardOutlineSizes: array[ 0 .. 11 ] of longint =
  (
    4, 5, 6, 8, 10, 12,
    15, 18, 24, 36, 48, 72
  );

function LongintListCompare( Item1: pointer;
                             Item2: pointer ): longint;
begin
  if item1 < item2 then
    result := -1
  else if item1 > item2 then
    result := 1
  else
    result := 0;
end;

Procedure TCustomFontDialog.InsertSizes;
var
     Face: string;
     TheFont: TFont;
     FontIndex: longint;
     SizeIndex: longint;
     OldSize: longint;
     SizeString: string;

     LimitedSizes: boolean;
     Size: longint;
     NearestSize: longint;
     NearestSizeIndex: longint;

     Sizes: TList;

     procedure AddSize( const size: longint );
     begin
       if Sizes.IndexOf( pointer( size ) ) = -1 then
         Sizes.Add( pointer( size ) );
     end;
Begin
     Sizes := TList.Create;

     Face := FaceName;
     try
       OldSize := StrToInt( FSizeCombo.Caption );
     except
       OldSize := 8;
     end;

     FSizeCombo.BeginUpdate;
     FSizeCombo.Clear;

     LimitedSizes := true;

     For FontIndex := 0 To Screen.FontCount - 1 Do
     Begin
       TheFont := Screen.Fonts[ FontIndex ];
       If TheFont.FaceName = Face Then
       Begin
          // this is a font for the current face.
          if TheFont.FontType = ftBitmap then
          begin
            // just insert the specified point size
            AddSize( TheFont.NominalPointSize );
          end
          else
          begin
            // an outline font...
            LimitedSizes := false;
            for SizeIndex := Low( StandardOutlineSizes ) to High( StandardOutlineSizes ) do
            begin
              AddSize( StandardOutlineSizes[ SizeIndex ] );
            end;
          end;
       end;
     End;

     // sort from small to large
     Sizes.Sort( LongintListCompare );

     // add to combobox
     For SizeIndex := 0 to Sizes.Count - 1 do
     begin
       SizeString := IntToStr( longint( Sizes[ SizeIndex ] ) );
       FSizeCombo.Items.Add( SizeString );
     end;

     if LimitedSizes then
     begin
       // Find nearest match for old size
       if Sizes.Count > 0 then
       begin
         NearestSizeIndex := 0;
         NearestSize := longint( Sizes[ 0 ] );
         for SizeIndex := 1 to Sizes.Count - 1 do
         begin
           Size := longint( Sizes[ SizeIndex ] );
           if Abs( Size - OldSize ) < Abs( NearestSize - OldSize ) then
           begin
             // closer,
             NearestSizeIndex := SizeIndex;
             NearestSize := Size;
           end;
         end;
       end
       else
       begin
         NearestSizeIndex := -1;
       end;

       FSizeCombo.ItemIndex := NearestSizeIndex;
     end
     else
     begin
       FSizeCombo.Text := IntToStr( OldSize );

       // if there's an exact match, select it
       NearestSizeIndex := FSizeCombo.Items.IndexOf( IntToStr( OldSize ) );

       FSizeCombo.ItemIndex := NearestSizeIndex

     end;

     FSizeCombo.EndUpdate;

     Sizes.Destroy;
End;


{$HINTS OFF}
Procedure TCustomFontDialog.OnSelectFace( Sender: TObject; Index: LongInt );
Begin
     InsertSizes;
     SelectFont;
End;

Procedure TCustomFontDialog.OnSetSize( Sender: TObject );
Begin
     SelectFont;
End;

Procedure TCustomFontDialog.OnSizeClick( Sender: TObject; Index: longint );
Begin
  // make a single click select
  FSizeCombo.Text := FSizeCombo.Items[ Index ];
End;

Procedure TCustomFontDialog.OnStyleChanged( Sender:TObject );
Begin
     SelectFont;
End;

Initialization
  {Register classes}
  RegisterClasses([TCustomFontDialog]);
End.

