Unit BookmarksFormUnit;

Interface

Uses
  Classes, Forms, Graphics, TabCtrls, StdCtrls, Buttons,
  ACLLanguageUnit,
  Outline2, CustomListBox,
  BookmarkUnit;

Type
  TBookmarkCallback = procedure( Bookmark: TBookmark ) of object;

  TBookmarksForm = Class (TForm)
    BookmarksListBox: TListBox;
    GotoButton: TButton;
    DeleteButton: TButton;
    RenameButton: TButton;
    CloseButton: TButton;
    Procedure BookmarksListBoxOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure CloseButtonOnClick (Sender: TObject);
    Procedure BookmarksListBoxOnDblClick (Sender: TObject);
    Procedure BookmarksFormOnCreate (Sender: TObject);
    Procedure BookmarksListBoxOnItemFocus (Sender: TObject; Index: LongInt);
    Procedure DeleteButtonOnClick (Sender: TObject);
    Procedure RenameButtonOnClick (Sender: TObject);
    Procedure GotoButtonOnClick (Sender: TObject);
    Procedure BookmarksFormOnShow (Sender: TObject);
  protected
    function GetSelectedBookmark: TBookmark;
    procedure UpdateControls;
    procedure GotoSelectedBookmark;

  Protected
    Procedure OnLanguageEvent( Language: TLanguageFile;
                               const Apply: boolean );
    DeleteBookmarkTitle: string;
    DeleteBookmarkA: string;
    DeleteBookmarkB: string;
    RenameBookmarkTitle: string;
    RenameBookmark: string;

  Public
    // Input
    OpenBookmarkCallback: TBookmarkCallback;
    BookmarksChangedCallback: TNotifyEvent;
    // Input/Output parameter
    BookmarkList: TList;
    procedure RefreshList;
  End;

Var
  BookmarksForm: TBookmarksForm;

Implementation

uses
  PMWin,
  Dialogs,
  ControlsUtility, ACLDialogs, ACLStringUtility;

Procedure TBookmarksForm.OnLanguageEvent( Language: TLanguageFile;
                                          const Apply: boolean );
begin
  Language.LoadComponentLanguage( self, Apply );

  Language.LL( Apply, DeleteBookmarkTitle, 'DeleteBookmarkTitle', 'Delete Bookmark' );
  Language.LL( Apply, DeleteBookmarkA, 'DeleteBookmarkA', 'Delete the bookmark named ' );
  Language.LL( Apply, DeleteBookmarkB, 'DeleteBookmarkB', '?' );
  Language.LL( Apply, RenameBookmarkTitle, 'RenameBookmarkTitle', 'Rename Bookmark' );
  Language.LL( Apply, RenameBookmark, 'RenameBookmarkTitle', 'Enter the new name of the bookmark' );
end;

Procedure TBookmarksForm.BookmarksListBoxOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  if KeyCode = kb_VK + VK_NEWLINE then
    GotoSelectedBookmark;
End;

Procedure TBookmarksForm.CloseButtonOnClick (Sender: TObject);
Begin
  Close;
End;

Procedure TBookmarksForm.BookmarksListBoxOnDblClick (Sender: TObject);
Begin
  GotoSelectedBookmark;
End;

Procedure TBookmarksForm.BookmarksFormOnCreate (Sender: TObject);
Begin
  RegisterForLanguages( OnLanguageEvent );

  BookmarksListBox.XStretch := xsFrame;
  BookmarksListBox.YStretch := ysFrame;
End;

Procedure TBookmarksForm.BookmarksListBoxOnItemFocus (Sender: TObject;
  Index: LongInt);
Begin
  UpdateControls;
End;

Procedure TBookmarksForm.DeleteButtonOnClick (Sender: TObject);
Var
  Bookmark: TBookmark;
  BookmarkIndex: longint;
Begin
  Bookmark := GetSelectedBookmark;
  if Bookmark = nil then
    exit;

  if DoConfirmDlg( DeleteBookmarkTitle,
                   DeleteBookmarkA
                   + StrDoubleQuote( Bookmark.Name )
                   + DeleteBookmarkB ) then
  begin
    BookmarkIndex := BookmarkList.IndexOf( Bookmark );
    BookmarksListBox.Items.Delete( BookmarkIndex );
    BookmarkList.Delete( BookmarkIndex );

    if BookmarkIndex > BookmarkList.Count - 1 then
      BookmarkIndex := BookmarkList.Count - 1;

    BookmarksListBox.ItemIndex := BookmarkIndex;

    Bookmark.Destroy;
    BookmarksChangedCallback( self );

    UpdateControls;
  end;
End;

Procedure TBookmarksForm.RenameButtonOnClick (Sender: TObject);
Var
  Bookmark: TBookmark;
Begin
  Bookmark := GetSelectedBookmark;
  if Bookmark = nil then
    exit;
  if DoInputQuery( RenameBookmarkTitle,
                   RenameBookmark,
                   Bookmark.Name ) then
  begin
    BookmarksChangedCallback( self );

    // redisplay name in list
    BookmarksListBox.Items[ BookmarksListBox.ItemIndex ] := Bookmark.Name;

  end;
End;

function TBookmarksForm.GetSelectedBookmark: TBookmark;

begin
  if SelectedObject( BookmarksListBox ) = nil then
    result := nil
  else
    result := SelectedObject( BookmarksListBox ) as TBookmark;

end;

Procedure TBookmarksForm.GotoButtonOnClick (Sender: TObject);
Begin
  GotoSelectedBookmark;
End;

Procedure TBookmarksForm.GotoSelectedBookmark;
Begin
  if Assigned( OpenBookmarkCallback ) then
    if GetSelectedBookmark <> nil then
      OpenBookmarkCallback( GetSelectedBookmark );
End;

Procedure TBookmarksForm.BookmarksFormOnShow (Sender: TObject);
begin
  RefreshList;
  BookmarksListBox.Focus;
end;

procedure TBookmarksForm.UpdateControls;
var
  Selected: Boolean;
begin
  Selected := GetSelectedBookmark <> nil;
  RenameButton.Enabled := Selected;
  DeleteButton.Enabled := Selected;
  GotoButton.Enabled := Selected;
end;

procedure TBookmarksForm.RefreshList;
var
  i: integer;
  Bookmark: TBookmark;
Begin
  BookmarksListBox.Items.BeginUpdate;

  BookmarksListBox.Clear;

  if not Assigned( BookmarkList ) then
    exit;

  for i := 0 to BookmarkList.Count - 1 do
  begin
    Bookmark := BookmarkList[ i ];
    BookmarksListBox.Items.AddObject( Bookmark.Name,
                                      Bookmark );
  end;

  if BookmarksListBox.Items.Count > 0 then
    BookmarksListBox.ItemIndex := 0;

  BookmarksListBox.Items.EndUpdate;
  UpdateControls;
End;

Initialization
  RegisterClasses ([TBookmarksForm
   , TListBox, TButton]);
End.
