Unit DataTypes;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// Just defines various types useful in manipulating help files.

type
  uint32 = longword;
  uint16 = word;
  uint8 = byte;
  pUInt16 = ^ uint16;
  pUInt32 = ^ uint32;
  pUInt8 = ^ uint8;

  PCharArray = array[ 0..0 ] of PCHar;
  UInt32Array = array[ 0..0 ] of UInt32;
  UInt16Array = array[ 0..0 ] of UInt16;
  UInt8Array = array[ 0..0 ] of UInt8;

  PCharArrayPointer = ^ PCharArray;
  UInt32ArrayPointer = ^ UInt32Array;
  UInt16ArrayPointer = ^ UInt16Array;
  UInt8ArrayPointer = ^ UInt8Array;

  TBooleanArray = array[ 0..0 ] of boolean;
  BooleanArrayPointer = ^TBooleanArray;

// Operations on Int32 arrays, used for searching
// These could be optimised heavily if needed.
procedure AllocUInt32Array( Var pArray: UInt32ArrayPointer;
                            Size: longint );
procedure FreeUInt32Array( Var pArray: UInt32ArrayPointer;
                           Size: longint );

procedure FillUInt32Array( pArray: UInt32ArrayPointer;
                           Size: longint;
                           Value: UInt32 );

procedure AddUInt32Array( pSource: UInt32ArrayPointer;
                          pDest: UInt32ArrayPointer;
                          Size: longint );

// Dest = Dest + source * Multiplier
procedure AddMultConstUInt32Array( pSource: UInt32ArrayPointer;
                                   Multiplier: longint;
                                   pDest: UInt32ArrayPointer;
                                   Size: longint );

procedure AndUInt32Array( pSource: UInt32ArrayPointer;
                          pDest: UInt32ArrayPointer;
                          Size: longint );

// If both source and dest > 0 then
//   add source to dest
procedure AndAddUInt32Array( pSource: UInt32ArrayPointer;
                             pDest: UInt32ArrayPointer;
                             Size: longint );

// if Source > 0 then dest is set to 0
procedure AndNotUInt32Array( pSource: UInt32ArrayPointer;
                             pDest: UInt32ArrayPointer;
                             Size: longint );

// dest = dest or source;
// if source > 0  then set dest to  > 0
procedure OrUInt32Array( pSource: UInt32ArrayPointer;
                         pDest: UInt32ArrayPointer;
                         Size: longint );

// if source = 0 then dest set to >0
procedure NotOrUInt32Array( pSource: UInt32ArrayPointer;
                            pDest: UInt32ArrayPointer;
                            Size: longint );

procedure CopyUInt32Array( pSource: UInt32ArrayPointer;
                           pDest: UInt32ArrayPointer;
                           Size: longint );

procedure ClearUInt32Array( pArray: UInt32ArrayPointer;
                            Size: longint );
procedure SetUInt32Array( pArray: UInt32ArrayPointer;
                          Size: longint );

// returns the result of ORing every array element.
// Can be useful for debugging e.g. seeing at a glance
// if any element is non-zero
function OrAllUInt32Array( pArray: UInt32ArrayPointer;
                           Size: longint ): longint;

Implementation

uses
  ACLUtility;

procedure AllocUInt32Array( Var pArray: UInt32ArrayPointer;
                            Size: longint );
begin
  GetMem( pArray,
          Size
          * sizeof( UInt32 ) );
end;

procedure FreeUInt32Array( Var pArray: UInt32ArrayPointer;
                           Size: longint );
begin
  FreeMem( pArray,
           Size
           * sizeof( UInt32 ) );
end;

// This is a nice fast implementation of filling an
// array of dwords (Int32/longword)
procedure FillUInt32Array( pArray: UInt32ArrayPointer;
                           Size: longint;
                           Value: UInt32 );
begin
  assert( Size >= 0 );
  Asm
    Mov EAX, Value
    Mov EDI, pArray
    Mov ECX, Size
    CLD        // direction = up
    REP STOSD   // store double word, until ECX = 0
  End;
end;

procedure ClearUInt32Array( pArray: UInt32ArrayPointer;
                            Size: longint );
begin
  FillUInt32Array( pArray, Size, 0 );
end;

procedure SetUInt32Array( pArray: UInt32ArrayPointer;
                          Size: longint );
begin
  FillUInt32Array( pArray, Size, $ffffffff );
end;

procedure AddUInt32Array( pSource: UInt32ArrayPointer;
                          pDest: UInt32ArrayPointer;
                          Size: longint );
var
  i: longint;
begin
  for i := 0 to Size - 1 do
    inc( pDest^[ i ], pSource^[ i ] );
end;

procedure AddMultConstUInt32Array( pSource: UInt32ArrayPointer;
                                   Multiplier: longint;
                                   pDest: UInt32ArrayPointer;
                                   Size: longint );
var
  i: longint;
begin
  for i := 0 to Size - 1 do
    inc( pDest^[ i ], pSource^[ i ] * Multiplier );
end;

procedure OrUInt32Array( pSource: UInt32ArrayPointer;
                         pDest: UInt32ArrayPointer;
                         Size: longint );
var
  i: longint;
begin
  for i := 0 to Size - 1 do
    pDest^[ i ] := pDest^[ i ] or pSource^[ i ];
end;

procedure CopyUInt32Array( pSource: UInt32ArrayPointer;
                           pDest: UInt32ArrayPointer;
                           Size: longint );
begin
  MemCopy( pSource, pDest, Size * sizeof( uint32 ) );
end;

procedure NotOrUInt32Array( pSource: UInt32ArrayPointer;
                            pDest: UInt32ArrayPointer;
                            Size: longint );
var
  i: longint;
begin
  for i := 0 to Size - 1 do
    if pSource^[ i ] = 0 then
      pDest^[ i ] := 1;
end;

procedure AndUInt32Array( pSource: UInt32ArrayPointer;
                          pDest: UInt32ArrayPointer;
                          Size: longint );
var
  i: longint;
begin
  for i := 0 to Size - 1 do
    pDest^[ i ] := pDest^[ i ] and pSource^[ i ];
end;

procedure AndAddUInt32Array( pSource: UInt32ArrayPointer;
                             pDest: UInt32ArrayPointer;
                             Size: longint );
var
  i: longint;
begin
  for i := 0 to Size - 1 do
    if     ( pSource^[ i ] > 0 )
       and ( pDest^[ i ] > 0 ) then
      inc( pDest^[ i ], pSource^[ i ] )
    else
      pDest^[ i ] := 0;
end;

procedure AndNotUInt32Array( pSource: UInt32ArrayPointer;
                             pDest: UInt32ArrayPointer;
                             Size: longint );
var
  i: longint;
begin
  for i := 0 to Size - 1 do
    if pSource^[ i ] > 0 then
      pDest^[ i ] := 0;
end;

function OrAllUInt32Array( pArray: UInt32ArrayPointer;
                           Size: longint ): longint;
var
  i: longint;
begin
  Result := 0;
  for i := 0 to Size - 1 do
    Result := Result or pArray^[ i ];
end;

Initialization
End.
