Unit GlobalSearchForm;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// Global search uses a thread (THelperThread,
// from Erik Huelsmann) to load helpfiles, search them, and
// send the results to be displayed.

Uses
  PmWin,
  Classes, Forms, Graphics, Grids, Buttons, StdCtrls, ComCtrls,
  MultiColumnListBox, Outline2, GenericThread, ACLLanguageUnit,
  TextSearchQuery, HelpFile;

const
  // Custom window messages for this form
  // NOTE! Sibyl uses WM_USER+1 and +2!
  WM_OPENED     = WM_USER + 10;

Type
  TViewTopicCallback = procedure( Filename: string;
                                  TopicIndex: longint ) of object;
  // Returned by thread. (Note - helpfiles and topics are destroyed dynamically.)
  TSearchResult = class
    Filename: string;
    FileTitle: string;
    MatchingTopics: TList;
    constructor Create;
    destructor Destroy; override;
  end;

  TGlobalSearchForm = Class (TForm)
    SearchTextEdit: TEdit;
    SearchTextLabel: TLabel;
    ResultsLabel: TLabel;
    ResultsOutline: TOutline2;
    SearchButton: TButton;
    ViewTopicButton: TButton;
    CancelButton: TButton;
    ProgressBar: TProgressBar;
    ProgressLabel: TLabel;
    FileProgressLabel: TLabel;
    Procedure CancelButtonOnClick (Sender: TObject);
    Procedure GlobalSearchFormOnClose (Sender: TObject;
      Var Action: TCloseAction);
    Procedure ResultsOutlineOnItemDblClick (Node: TNode);
    Procedure ViewTopicButtonOnClick (Sender: TObject);
    Procedure GlobalSearchFormOnCreate (Sender: TObject);
    Procedure GlobalSearchFormOnCloseQuery (Sender: TObject;
      Var CanClose: Boolean);
    Procedure SearchButtonOnClick (Sender: TObject);
    Procedure GlobalSearchFormOnShow (Sender: TObject);
  Protected
    Procedure OnSearchProgress ( n, outof: integer;
                                 S: String );
    Procedure OnMatchFound( SearchResult: TSearchResult );
    Procedure OnThreadData( S: string; Data: TObject );
    Procedure OnSearchFinished( Result: TObject );
    Procedure SetProgressLabel( const S: String );
    Procedure SetFileProgressLabel( const S: String );

    // Handle our own WM_OPENED message
    Procedure WMOpened( Var Msg: TMessage ); Message WM_OPENED;
    Procedure ClearResults;
    Procedure ViewTopic;

    ThreadManager: TGenericThreadManager;

    // search thread context

    Function Search( Parameters: TObject ): TObject;
    Procedure OnFileProgress( n, outof: integer;
                              Message: string );

  protected
    Procedure OnLanguageEvent( Language: TLanguageFile;
                               const Apply: boolean );

    SearchCaption: string;
    StopCaption: string;
    NoResultsMsg: string;
    ScanBookshelfMsg: string;
    ScanHelpMsg: string;
    SearchingFilesMsg: string;
    SearchingFileMsg: string;
    OfMsg: string;
    DoneMsg: string;
    SearchErrorTitle: string;
    SearchError: string;

  public
    // set by caller
    ViewTopicCallback: TViewTopicCallback;
    Procedure DoSearch;
  End;

Var
  GlobalSearchForm: TGlobalSearchForm;

procedure EnsureGlobalSearchFormLoaded;

Implementation

uses
  PMWin,
  SysUtils, BseDos,
  Dialogs,
  ACLFileUtility, ACLProfile, ACLDialogs,
  HelpTopic, SearchUnit;

type
  // Used to store filenames in outline
  THelpFileInfo = class
    FileName: string;
  end;

constructor TSearchResult.Create;
begin
  inherited Create;
  MatchingTopics := TList.Create;
end;

destructor TSearchResult.Destroy;
begin
  MatchingTopics.Destroy;
  inherited Destroy;
end;

Procedure TGlobalSearchForm.OnLanguageEvent( Language: TLanguageFile;
                                             const Apply: boolean );
begin
  Language.LoadComponentLanguage( self, Apply );

  Language.LL( Apply, SearchCaption, 'SearchCaption', '~Search' );
  Language.LL( Apply, StopCaption, 'StopCaption', '~Stop' );
  Language.LL( Apply, NoResultsMsg, 'NoResultsMsg', '(No results found)' );
  Language.LL( Apply, ScanBookshelfMsg, 'ScanBookshelfMsg', 'Finding BOOKSHELF files...' );
  Language.LL( Apply, ScanHelpMsg, 'ScanHelpMsg', 'Finding HELP files...' );
  Language.LL( Apply, SearchingFilesMsg, 'SearchingFilesMsg', 'Searching files...' );
  Language.LL( Apply, SearchingFileMsg, 'SearchingFileMsg', 'Searching ' );
  Language.LL( Apply, OfMsg, 'OfMsg', ' of ' );
  Language.LL( Apply, DoneMsg, 'DoneMsg', 'Done' );
  Language.LL( Apply, SearchErrorTitle, 'SearchErrorTitle', 'Search' );
  Language.LL( Apply, SearchError, 'SearchError', 'Error in search syntax: ' );
end;

Procedure TGlobalSearchForm.CancelButtonOnClick (Sender: TObject);
Begin
  Close;
End;

Procedure TGlobalSearchForm.GlobalSearchFormOnClose (Sender: TObject;
  Var Action: TCloseAction);
Begin
  Action := caFreeHandle; // DON'T release the form! (Default action for non-modal forms)
End;

Procedure TGlobalSearchForm.ResultsOutlineOnItemDblClick (Node: TNode);
Begin
  ViewTopic;
End;

Procedure TGlobalSearchForm.ViewTopicButtonOnClick (Sender: TObject);
begin
  ViewTopic;
end;

Procedure TGlobalSearchForm.ViewTopic;
var
  Node: TNode;
  HelpFileInfo: THelpFileInfo;
  TopicIndex: longint;
Begin
  Node := ResultsOutline.SelectedNode;
  if Node = nil then
    exit;
  case Node.Level of
    0:
    begin
      // file node
      HelpFileInfo := Node.Data as THelpFileInfo;
      TopicIndex := -1;
    end;

    1:
    begin
      // topic node
      HelpFileInfo := Node.Parent.Data as THelpFileInfo;
      TopicIndex := longint( Node.Data );
    end;

    else
      assert( false, 'Invalid node level in ViewTopic!: ' + IntToStr( Node.Level ) );
  end;

  ViewTopicCallback( HelpFileInfo.FileName, TopicIndex );
End;

Procedure TGlobalSearchForm.GlobalSearchFormOnCreate (Sender: TObject);
Begin
  RegisterForLanguages( OnLanguageEvent );

  ThreadManager := TGenericThreadManager.Create( self );
  ThreadManager.OnProgressUpdate := OnSearchProgress;
  ThreadManager.OnDataFromThread := OnThreadData;
  ThreadManager.OnJobComplete := OnSearchFinished;
End;

Procedure TGlobalSearchForm.OnSearchFinished( Result: TObject );
Begin
  SearchButton.Caption := SearchCaption;
  ProgressBar.Hide;

  if ResultsOutline.ChildCount > 0 then
  begin
    ResultsOutline.SelectedNode:= ResultsOutline.Children[ 0 ];
    ResultsOutline.Focus;
    SetProgressLabel( '' );
  end
  else
  begin
    SetProgressLabel( NoResultsMsg );
  end;

  SetFileProgressLabel( '' );

End;

Procedure TGlobalSearchForm.GlobalSearchFormOnCloseQuery (Sender: TObject;
  Var CanClose: Boolean);
Begin
  if ThreadManager.IsRunning then
  begin
    ThreadManager.Stop;
  end;
End;

Procedure TGlobalSearchForm.SetProgressLabel( const S: String );
begin
  ProgressLabel.Text := S;
  ProgressLabel.Refresh;
end;

Procedure TGlobalSearchForm.SetFileProgressLabel( const S: String );
begin
  FileProgressLabel.Text := S;
  FileProgressLabel.Refresh;
end;

Procedure TGlobalSearchForm.OnSearchProgress ( n, outof: integer;
                                               S: String );
Begin
  ProgressBar.Position := n * 100 div outof;
  SetProgressLabel( S );
End;

Procedure TGlobalSearchForm.OnFileProgress( n, outof: integer;
                                            Message: string );
Begin
  ThreadManager.SendData( Message, nil );
end;

Function TGlobalSearchForm.Search( Parameters: TObject ): TObject;
var
  Files: TStringList;
  FileIndex: longint;
  Filename: string;
  HelpFile: THelpFile;
  SearchResult: TSearchResult;
  MatchingTopics: TList;

  Query: TTextSearchQuery;
Begin
//  StartProfile( GetLogFilesDir + 'searchprofile' );

  Query := Parameters as TTextSearchQuery;
  Files := TStringList.Create;
  MatchingTopics := TList.Create;

  // make sure we ignore duplicate files...
  Files.Sorted := true;
  Files.CaseSensitive := false;
  Files.Duplicates := dupIgnore;

  ThreadManager.UpdateProgress( 0, 100, ScanBookshelfMsg );
  GetFilesForPath( 'BOOKSHELF', '*.inf;*.hlp', Files );

  ThreadManager.UpdateProgress( 1, 100, ScanHelpMsg );
  GetFilesForPath( 'HELP', '*.inf;*.hlp', Files );

  ThreadManager.UpdateProgress( 2, 100, SearchingFilesMsg );

  for FileIndex:= 0 to Files.Count - 1 do
  begin
    if ThreadManager.StopRequested then
      break;
    Filename := Files[ FileIndex ];
    ThreadManager.UpdateProgress( 5 + FileIndex * 95 div Files.Count,
                                  100,
                                  SearchingFileMsg
                                  + Filename
                                  + ' ('
                                  + IntToStr( FileIndex + 1 )
                                  + OfMsg
                                  + IntToStr( Files.Count )
                                  + ')...' );

    try
      HelpFile := THelpFile.Create( FileName );

      MatchingTopics.Clear;
      SearchHelpFile( HelpFile,
                      Query,
                      MatchingTopics,
                      nil // don't care about words matched
                      );

      if MatchingTopics.Count > 0 then
      begin
        ProfileEvent( '  Sort results' );
        // Create a searchresult object to send back to main thread.
        SearchResult := TSearchResult.Create;
        SearchResult.Filename := HelpFile.Filename;
        SearchResult.FileTitle := HelpFile.Title;

        SearchResult.MatchingTopics.Assign( MatchingTopics );

        SearchResult.MatchingTopics.Sort( TopicRelevanceCompare );
        ProfileEvent( '  Display results' );

        ThreadManager.SendData( '', SearchResult );
      end;

      ProfileEvent( 'Unload helpfile' );
      HelpFile.Destroy;

    except
      on E: EHelpFileException do
      begin
        ; // ignore exceptions
      end;
    end;

  end;
  ThreadManager.UpdateProgress( 100, 100, DoneMsg );
  Query.Destroy;
  Files.Destroy;
  Result := nil;
End;

Procedure TGlobalSearchForm.ClearResults;
var
  FileIndex: longint;
begin
  for FileIndex := 0 to ResultsOutline.ChildCount - 1 do
    ResultsOutline.Children[ FileIndex ].Data.Free;
  ResultsOutline.Clear;
end;

Procedure TGlobalSearchForm.SearchButtonOnClick (Sender: TObject);
begin
  DoSearch;
end;

Procedure TGlobalSearchForm.DoSearch;
var
  SearchText: string;
  Query: TTextSearchQuery;
Begin
  if ThreadManager.IsRunning then
  begin
    ThreadManager.Stop;
    exit;
  end;

  SearchText := trim( SearchTextEdit.Text );
  if SearchText = '' then
    exit;

  try
    Query := TTextSearchQuery.Create( SearchText );
  except
    on e: ESearchSyntaxError do
    begin
      DoErrorDlg( SearchErrorTitle,
                  SearchError
                  + e.Message );
      exit;
    end;
  end;

  ClearResults;

  ThreadManager.StartJob( Search, Query );
  SearchButton.Caption := StopCaption;
  ProgressBar.Show;

End;

Procedure TGlobalSearchForm.GlobalSearchFormOnShow (Sender: TObject);
Begin
  // make search button default
  SearchButton.Focus;
  SearchTextEdit.Focus;
  SetProgressLabel( '' );
  SetFileProgressLabel( '' );
  PostMsg( Handle, WM_OPENED, 0, 0 );
  SearchButton.Caption := SearchCaption;
  ProgressBar.Hide;
End;

Procedure TGlobalSearchForm.OnThreadData( S: string; Data: TObject );
var
  SearchResult: TSearchResult;
begin
  if Data <> nil then
  begin
    SearchResult := Data as TSearchResult;
    OnMatchFound( SearchResult );
    SearchResult.Destroy;
    exit;
  end;

  // just an individual file progress update
  SetFileProgressLabel( S );

end;

Procedure TGlobalSearchForm.OnMatchFound( SearchResult: TSearchResult );
var
  Topic: TTopic;
  HelpFileInfo: THelpFileInfo;
  FileNode: TNode;
  TopicIndex: longint;
begin
  HelpFileInfo := THelpFileInfo.Create;
  HelpFileInfo.FileName := SearchResult.FileName;
  FileNode := ResultsOutline.AddChild( SearchResult.FileTitle
                                       + ' ('
                                       + SearchResult.FileName
                                       + ')',
                                       HelpFileInfo );
  for TopicIndex := 0 to SearchResult.MatchingTopics.Count - 1 do
  begin
    Topic := SearchResult.MatchingTopics[ TopicIndex ];
    FileNode.AddChild( Topic.Title,
                       TObject( Topic.Index ) );
  end;
end;

Procedure TGlobalSearchForm.WMOpened( Var Msg: TMessage );
begin
  SearchTextEdit.XStretch := xsFrame;
  ProgressLabel.XStretch := xsFrame;
  ResultsOutline.XStretch := xsFrame;
  ResultsOutline.YStretch := ysFrame;
  ProgressLabel.YAlign := yaTop;
  SearchTextEdit.YAlign := yaTop;
  SearchTextEdit.Focus;
end;

procedure EnsureGlobalSearchFormLoaded;
begin
  if GlobalSearchForm = nil then
    GlobalSearchForm := TGlobalSearchForm.Create( nil );
end;

Initialization
  RegisterClasses ([TGlobalSearchForm, TEdit, TLabel,
    TProgressBar, TButton, TOutline2]);
  RegisterUpdateProcForLanguages( EnsureGlobalSearchFormLoaded );
End.
