Unit HelpFile;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// Encapsulates the basic reading of a help file's structure.

uses
  classes, BseDos, os2def, SysUtils, Graphics,
  DataTypes,
  IPFFileFormatUnit, HelpTopic, HelpBitmap, ACLUtility,
  TextSearchQuery, SearchTable;

Type

  EHelpFileException = class( Exception )
  end;

type
  THelpFile = class
  protected
    _Filename : string;
    _FileSize : longint;
    _Handle: HFILE;

    _pSlotData: pUInt16;
    _SlotDataSize: longint;

    _Title: string;

    _Topics: TList; // of TTopics

    _Dictionary: TList; // pointers to strings.

    _Index: TStringList;

    _SearchTable: TSearchTable;

    _ReferencedFiles: TStringList;

    _FontTable: TList;

    _pHeader: TPHelpFileHeader;
    _pExtendedHeader: TPExtendedHelpFileHeader;
    _pContentsData: pointer;
    _pResourceData: pointer;
    _pSearchData: pointer;
    _pHighlightWords: UInt32ArrayPointer;
    _pSlotOffsets: Uint32ArrayPointer;
    _pDictionaryData: pointer;
    _pIndexData: pointer;
    _pFontTableData: pointer;

    procedure InitMembers;
    procedure Open;
    procedure Close;
    procedure ReadFileBlock( Var Dest: pointer;
                             const StartPosition: ULONG;
                             const Length: ULONG );

    procedure ReadHeader;
    procedure ReadContents;
    procedure ReadDictionary;
    procedure ReadSearchTable;

    procedure ReadIndex;

    procedure ReadReferencedFilesTable;
    procedure ReadFontTable;

    function GetTopic( Index: longint ): TTopic;
    function GetTopicCount: longint;

    function GetDictionaryCount: longint;
    function GetDictionaryWord( Index: longint ): string;
    function GetDictionaryWordPtr( Index: longint ): pstring;

    function GetIndexEntryPtr( Index: longint ): pstring;
    function GetHighlightWords: UInt32ArrayPointer;

    function GetSearchTable: TSearchTable;
  public
    constructor Create( const FileName: string );

    destructor Destroy; override;

    property Title: string read _Title;
    property Topics[ Index: longint ]: TTopic read GetTopic;
    property TopicList: TList read _Topics;
    property TopicCount: longint read GetTopicCount;
    property Index: TStringList read _Index;
    property IndexEntryPtr[ index: longint ]: pstring read GetIndexEntryPtr;
    property Filename: string read _FileName;

    procedure GetImages( ImageOffsets: TList;
                         Images: TImageList );

    property DictionaryCount: longint read GetDictionaryCount;
    property DictionaryWords[ Index: longint ]: string read GetDictionaryWord;
    property DictionaryWordPtrs[ Index: longint ]: pstring read GetDictionaryWordPtr;

    function IndexOfTopic( Topic: TTopic ): longint;

    property SearchTable: TSearchTable read GetSearchTable;

    function FindTopicByResourceID( ID: uint16 ): TTopic;
    procedure FindResourceIDsForTopic( Topic: TTopic;
                                       ResourceIDs: TList );

    property HighlightWords: UInt32ArrayPointer read GetHighlightWords;

    property FileSize: longint read _FileSize;
  end;

// Returns helpfile that the given topic is within
Function TopicFile( Topic: TTopic ): THelpFile;

function GetHelpFileTitle( const Filename: string ): string;

Implementation

uses
  Dialogs, Forms,
  BseErr,
  ACLFileUtility, ACLStringUtility, ACLFileIOUtility, ACLProfile,
  ACLDialogs, ACLLanguageUnit,
  HelpWindowDimensions;

// Load "missing" bitmap
{$R Images}

var
  FileErrorNotFound: string;
  FileErrorAccessDenied: string;
  FileErrorInUse: string;
  FileErrorInvalidHeader: string;

Procedure OnLanguageEvent( Language: TLanguageFile;
                           const Apply: boolean );
begin

  Language.Prefix := 'HelpFile.';
  Language.LL( Apply, FileErrorNotFound, 'FileErrorNotFound', 'File not found' );
  Language.LL( Apply, FileErrorAccessDenied, 'FileErrorAccessDenied', 'Access denied' );
  Language.LL( Apply, FileErrorInUse, 'FileErrorInUse', 'File in use by another program' );
  Language.LL( Apply,
               FileErrorInvalidHeader,
               'FileErrorInvalidHeader',
               'File doesn''t appear to be an OS/2 Help document (header ID not correct)' );
end;

Function TopicFile( Topic: TTopic ): THelpFile;
Begin
  Result := Topic.HelpFile as THelpFile;
end;

procedure THelpFile.InitMembers;
begin
  _SlotDataSize := 0;

  _pHeader := nil;
  _pExtendedHeader := nil;
  _pContentsData := nil;
  _pSlotOffsets := nil;
  _pResourceData := nil;
  _pSearchData := nil;
  _pDictionaryData := nil;
  _pIndexData := nil;
  _pFontTableData := nil;

  _pHighlightWords := nil;

  _Dictionary := TList.Create;
  _Topics := TList.Create;
  _Index := TStringList.Create;
  _ReferencedFiles := TStringList.Create;
  _FontTable := TList.Create;
end;

constructor THelpFile.Create( const FileName: string );
begin
  ProfileEvent( 'Helpfile Load: ' + FileName );

  _FileName := FileName;

  InitMembers;

  Open;

  // we always need these basics:
  try
    ReadHeader;
    ReadContents;
    ReadIndex;
    ReadDictionary;
    ReadFontTable;
    ReadReferencedFilesTable;
  except
    Close;
    raise;
  end;

  // the rest is loaded on demand
end;

destructor THelpFile.Destroy;
begin
  DeallocateMemory( _pHeader );
  DeallocateMemory( _pExtendedHeader );
  DeallocateMemory( _pContentsData );
  DeallocateMemory( _pSlotOffsets );
  DeallocateMemory( _pResourceData );
  DeallocateMemory( _pSearchData );
  DeallocateMemory( _pDictionaryData );
  DeallocateMemory( _pIndexData );
  DeallocateMemory( _pFontTableData );

  DeallocateMemory( _pHighlightWords );

  if Assigned( _Topics ) then
    DestroyListAndObjects( _Topics );

  _Index.Destroy;

  _Dictionary.Free;
  _SearchTable.Free;
  _ReferencedFiles.Free;
  _FontTable.Free;

  DosClose( _Handle );
end;

procedure THelpFile.Open;
var
  OpenAction: ULong;
  rc: APIRET;
  szName: Cstring;
  FileInfo: FILESTATUS3;
begin
  if not FileExists( _Filename ) then
    raise EHelpFileException.Create( FileErrorNotFound );

  szName:= _FileName;
  rc:= DosOpen( szName,
                _Handle,
                OpenAction,
                0, // file size - irrelevant, not creating,
                0, // attributes - ''
                OPEN_ACTION_OPEN_IF_EXISTS,
                OPEN_SHARE_DENYNONE + OPEN_ACCESS_READONLY,
                nil ); // no extended attributes
  if rc<> 0 then
  begin
    case rc of
      ERROR_FILE_NOT_FOUND: // crap, this doesn't actually occur!
        raise EHelpFileException.Create( FileErrorNotFound );

      ERROR_ACCESS_DENIED:
        raise EHelpFileException.Create( FileErrorAccessDenied );

      ERROR_SHARING_VIOLATION:
        raise EHelpFileException.Create( FileErrorInUse );

      else
        raise EHelpFileException.Create( SysErrorMessage( rc ) );
    end;
  end;

  DosQueryFileInfo( _Handle,
                    FIL_STANDARD,
                    FileInfo,
                    sizeof( FileInfo ) );
  _FileSize := FileInfo.cbFile; // file size
end;

procedure THelpFile.Close;
begin
  if _Handle <> 0 then
    DosClose( _Handle );
  _Handle := 0;
end;

procedure THelpFile.ReadFileBlock( Var Dest: pointer;
                                   const StartPosition: ULONG;
                                   const Length: ULONG );
begin
  ACLFileIOUtility.ReadFileBlock( _Handle, Dest, StartPosition, Length );
end;

// -------------------------------------------------------------------------

procedure THelpFile.ReadHeader;
begin
  ProfileEvent( 'Read header' );

  ReadFileBlock( _pHeader,
                 0,
                 sizeof( _pHeader^ ) );

  if _pHeader^.ID <> INF_HEADER_ID then
    raise EHelpFileException.Create( FileErrorInvalidHeader );

  _Title:= StrPas( _pHeader^.Title );

  if _pHeader^.extstart > 0 then
  begin
    // read extended header
    ReadFileBlock( _pExtendedHeader,
                   _pHeader^.extstart,
                   sizeof( _pExtendedHeader^ ) );
  end;
end;

procedure THelpFile.ReadContents;
var
  Topic: TTopic;
  EntryIndex: longint;
  pEntry: pTTOCEntryStart;
begin
  ProfileEvent( 'Read contents' );

  if _pHeader^.ntoc = 0 then
    exit; // explicit check required since ntoc is unsigned

  // Presize the topics list to save reallocation time
  _Topics.Capacity:= _pHeader^.ntoc;

  // read slots first so that Topics can refer to it.
  ReadFileBlock( _pSlotOffsets,
                 _pHeader^.slotsstart,
                 _pHeader^.nslots * sizeof( uint32 ) );

  ReadFileBlock( _pContentsData,
                 _pHeader^.tocstart,
                 _pHeader^.toclen );

  pEntry := _pContentsData;

  for EntryIndex:= 0 to _pHeader^.ntoc - 1 do
  begin
    Topic:= TTopic.Create( _Handle,
                           _pSlotOffsets,
                           _Dictionary,
                           pEntry,
                           _FontTable );

    Topic.HelpFile:= Self;
    Topic.Index:= EntryIndex;

    _Topics.Add( Topic );

    inc( pEntry, pEntry ^. Length );
  end;
end;

procedure THelpFile.ReadDictionary;
var
  i: longint;
  Len: uint8;
  p: pbyte;
begin
  ProfileEvent( 'Read dictionary' );

  if _pHeader^.ndict = 0 then
    exit; // explicit check required since ndict is unsigned

  ReadFileBlock( _pDictionaryData,
                 _pHeader^.dictstart,
                 _pHeader^.dictlen );

  P := _pDictionaryData;

  // Presize the dictionary to save reallocation
  _Dictionary.Capacity := _pHeader^.ndict;
  for i:= 0 to _pHeader^.ndict - 1 do
  begin
    // adjust length so we can use as a Pascal string
    // (file uses length including length byte,
    //  Pascal string have length excluding length byte)
    Len:= p^ - 1;
    p^ := Len;
    _Dictionary.Add( P );
    inc( P, Len + 1 );
  end;
end;

type
  TIndexEntryHeader = record
    TextLength: uint8;
    Flags: uint8;
    NumberOfRoots: uint8;
    TOCIndex: uint16;
  end;

procedure THelpFile.ReadIndex;
var
  IndexIndex: longint; // I can't resist :-)
  pEntryHeader: ^TIndexEntryHeader;
  EntryText: string;
  IndexTitleLen: longint;
  p: pointer;
begin
  ProfileEvent( 'Read index' );

  if _pHeader^.nindex = 0 then
    exit; // explicit check required since ndict is unsigned

  ReadFileBlock( _pIndexData,
                 _pHeader^.indexstart,
                 _pHeader^.indexlen );

  P := _pIndexData;

  for IndexIndex := 0 to _pHeader^.nindex - 1 do
  begin
    pEntryHeader:= p;
    IndexTitleLen:= pEntryHeader^.TextLength;
    inc( p, sizeof( TIndexEntryHeader ) );

    GetMemString( p, EntryText, IndexTitleLen );
    if ( pEntryHeader^.flags and 2 ) > 0 then
      EntryText:= '- ' + EntryText;
    if pEntryHeader^.TOCIndex < _Topics.Count then
      _Index.AddObject( EntryText, _Topics[ pEntryHeader^.TOCIndex ] )
    else
//      raise EHelpFileException.Create( 'Error reading help file index - out of range topic reference' );
      ; // pass! something special

    inc( p, IndexTitleLen
            + pEntryHeader^.NumberOfRoots
              * sizeof( uint32 ) ); // skip 'roots' for index search
  end;

end;

function THelpFile.GetSearchTable: TSearchTable;
begin
  if _SearchTable = nil then
    ReadSearchTable;
  Result := _SearchTable;
end;

procedure THelpFile.ReadSearchTable;
var
  SearchTableOffset: longint;
  SearchTableRecordLengthIs16Bit: boolean;
begin
  ProfileEvent( 'Read search table' );

  if _pHeader^.SearchLen = 0 then
  begin
    ProfileEvent( 'Read search table' );
    exit;
  end;

  SearchTableOffset := _pHeader^.SearchStart and $7fffffff;
  SearchTableRecordLengthIs16Bit := _pHeader^.SearchStart and $80000000 > 0;
  ReadFileBlock( _pSearchData,
                 SearchTableOffset,
                 _pHeader^.SearchLen );

  _SearchTable := TSearchTable.Create( _pSearchData,
                                       SearchTableRecordLengthIs16Bit,
                                       _Dictionary.Count,
                                       _Topics.Count );
end;

function THelpFile.GetHighlightWords: UInt32ArrayPointer;
begin
  if _pHighlightWords = nil then
    _pHighlightWords := AllocateMemory( _Dictionary.Count * sizeof( UInt32 ) );
  Result := _pHighlightWords;
end;

function THelpFile.FindTopicByResourceID( ID: uint16 ): TTopic;
var
  i: longint;
  pResourceIDs: UInt16ArrayPointer;
  pTopicIndices: UInt16ArrayPointer;
  FileResourceID: uint16;
  TopicIndex: uint16;
begin
  Result := nil;

  if _pHeader^.nres = 0 then
    // since nres is unsigned
    exit;

  if _pResourceData = nil then
    ReadFileBlock( _pResourceData,
                   _pHeader^.resstart,
                   _pHeader^.nres * sizeof( uint16 ) * 2 ); // list of IDs, list of topics

  pResourceIDs := _pResourceData;
  pTopicIndices := _pResourceData
                   + _pHeader^.nres * sizeof( uint16 );

  for i := 0 to _pHeader^.nres - 1 do
  begin
    FileResourceID := pResourceIDs^[ i ];
    if FileResourceID = ID then
    begin
      // found
      TopicIndex := pTopicIndices^[ i ];
      Result := _Topics[ TopicIndex ];
      exit;
    end;
  end;
end;

procedure THelpFile.FindResourceIDsForTopic( Topic: TTopic;
                                             ResourceIDs: TList );
var
  i: longint;
  pResourceIDs: UInt16ArrayPointer;
  pTopicIndices: UInt16ArrayPointer;
begin
  ResourceIDs.Clear;

  if _pHeader^.nres = 0 then
    // since nres is unsigned
    exit;

  if _pResourceData = nil then
    ReadFileBlock( _pResourceData,
                   _pHeader^.resstart,
                   _pHeader^.nres * sizeof( uint16 ) * 2 ); // list of IDs, list of topics

  pResourceIDs := _pResourceData;
  pTopicIndices := _pResourceData
                   + _pHeader^.nres * sizeof( uint16 );

  for i := 0 to _pHeader^.nres - 1 do
  begin
    if pTopicIndices^[ i ] = Topic.Index then
    begin
      // found
      ResourceIDs.Add( pointer( pResourceIDs^[ i ] ) );
    end;
  end;
end;

procedure THelpFile.ReadReferencedFilesTable;
var
  i: longint;
  p: pointer;
  pData: pointer;
  DatabaseName: string;
  pLength: pByte;
begin
  if _pExtendedHeader = nil then
    // no extended header -> no referenced files table
    exit;

  if _pExtendedHeader ^.Numdatabase = 0 then
    exit;

  pData := nil; // please allocate...
  ReadFileBlock( pData,
                 _pExtendedHeader^.DatabaseOffset,
                 _pExtendedHeader^.DatabaseSize );

  p := pData;
  for i:= 0 to _pExtendedHeader^.Numdatabase - 1 do
  begin
    pLength := p; // length byte, including itself
    GetMemString( p + 1, DatabaseName, ( pLength ^ ) - 1 );
    _ReferencedFiles.Add( DatabaseName );
    inc( p, pLength ^ ); // skip to next entry
  end;
  DeallocateMemory( pData );
end;

procedure THelpFile.ReadFontTable;
var
  i: longint;
  p: pointer;
  pFontSpec: pTHelpFontSpec;
begin
  if _pExtendedHeader = nil then
    // no extended header -> no font table
    exit;

  if _pExtendedHeader^.NumFontEntry = 0 then
    exit;

  ReadFileBlock( _pFontTableData,
                 _pExtendedHeader^.FontTableOffset,
                 _pExtendedHeader^.NumFontEntry * sizeof( THelpFontSpec ) );
  p := _pFontTableData;
  for i := 0 to _pExtendedHeader^.NumFontEntry - 1 do
  begin
    pFontSpec := p + i * sizeof( THelpFontSpec );
    _FontTable.Add( pFontSpec );
  end;
end;

procedure THelpFile.GetImages( ImageOffsets: TList;
                               Images: TImageList );
var
  ListIndex: longint;
  ImageOffset: longint;
  Bitmap: THelpBitmap;
begin
  Images.Clear;
  for ListIndex:= 0 to ImageOffsets.Count - 1 do
  begin
    ImageOffset := longint( ImageOffsets[ ListIndex ] );
    try
      Bitmap:= THelpBitmap.CreateFromHelpFile( _Handle,
                                               _pHeader^.imgstart
                                               + ImageOffset );
    except
      on e: EHelpBitmapException do
{        raise EHelpFileException.Create( 'Error loading help bitmap at'
                                         + IntToStr( ImageOffset )
                                         + ': '
                                         + e.Message );}
      begin
        Bitmap:= THelpBitmap.Create;
        Bitmap.LoadFromResourceName( 'MissingBitmap' );
      end;
    end;

    Images.Add( Bitmap, nil );
    Bitmap.Destroy;

  end;
end;

function THelpFile.GetTopic( Index: longint ): TTopic;
begin
  if    ( Index < 0 )
     or ( Index > _Topics.Count - 1 ) then
    Result := nil
  else
    Result:= _Topics[ Index ];
end;

function THelpFile.GetTopicCount: longint;
begin
  Result:= _Topics.Count;
end;

function THelpFile.IndexOfTopic( Topic: TTopic ): longint;
begin
  Result:= _Topics.IndexOf( Topic );
end;

function THelpFile.GetDictionaryCount: longint;
begin
  Result := _Dictionary.Count;
end;

function THelpFile.GetDictionaryWord( Index: longint ): string;
begin
  Result := pstring( _Dictionary[ Index ] )^;
end;

function THelpFile.GetDictionaryWordPtr( Index: longint ): pstring;
begin
  Result := pstring( _Dictionary[ Index ] );
end;

function THelpFile.GetIndexEntryPtr( Index: longint ): pstring;
begin
  Result := _Index.ValuePtrs[ Index ];
end;

function GetHelpFileTitle( const Filename: string ): string;
var
  OpenAction: ULong;
  rc: APIRET;
  szName: Cstring;

  Header: THelpFileHeader;
  Handle: HFILE;
  Ext: string;
begin
  Ext := ExtractFileExt( Filename );
  Result := '';

  if    StringsSame( Ext, '.inf' )
     or StringsSame( Ext, '.hlp' ) then
  begin
    szName := Filename;
    rc:= DosOpen( szName,
                  Handle,
                  OpenAction,
                  0, // file size - irrelevant, not creating,
                  0, // attributes - ''
                  OPEN_ACTION_OPEN_IF_EXISTS,
                  OPEN_SHARE_DENYNONE + OPEN_ACCESS_READONLY,
                  nil ); // no extended attributes
    if rc = 0 then
    begin
      FillChar( Header, sizeof( Header ), 0 );
      MyRead( Handle, Addr( Header ), sizeof( Header ) );
      if Header.ID = INF_HEADER_ID then
        Result := StrPas( Header.Title );
      DosClose( Handle );
    end;
  end;
end;

Initialization
  RegisterProcForLanguages( OnLanguageEvent );
End.

