Unit HelpTopic;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// This is it - the monster which decodes IPF data.
// It's created with a reference to the contents data defining it.
// It gets relevant pointers out of that. When GetText is called
// it decodes the data and spits out formatted text to suit
// RichTextView.

uses
  BseDos, OS2Def,
  Classes, Graphics,
  ACLString,
  RichTextDocumentUnit,
  DataTypes, HelpWindowDimensions, IPFFileFormatUnit;

const
  DefaultGroupIndex = 0;

Type
  THelpLink = class
    HelpFile: TObject;     // file this link is within

    // Even though it doesn't do anything,
    // we have to have a constructor to allow
    // virtual constructors to work
    constructor Create; virtual;
  end;

  THelpTopicSlot = class
    pData: pUInt8;
    Size: longint;
    pLocalDictionary: UInt16ArrayPointer;
    LocalDictSize: uint8;
    destructor Destroy; override;
  end;

  THelpLinkClass = class of THelpLink;

  TFootnoteHelpLink = class( THelpLink )
    TopicIndex: longint;
  end;

  TWindowedHelpLink = class( THelpLink )
    GroupIndex: longint;   // DefaultGroupIndex if not specified.
                           // Note: Overrides contents group index of topic
    Automatic: boolean;    // link should be automatically followed on topic display
    Split: boolean;        // link should open the window within the parent
    ViewPort: boolean;     // link should always open a new window
    Dependent: boolean;    // window opened by link should be closed
                           // when current topic is closed
    Rect: THelpWindowRect; // Display window with this rectangle.
                           // Note: overrides contents rect
    constructor Create; override;
    destructor Destroy; override;
  end;

  TInternalHelpLink = class( TWindowedHelpLink )
    TopicIndex: longint;
  end;

  THelpLinkByResourceID = class( TWindowedHelpLink )
    ResourceID: longint;
  end;

  SlotArray = array[ 0..0 ] of THelpTopicSlot;

  pSlotArray = ^SlotArray;

  TFontState = ( fsNormal, fsFixed, fsCustom );

  TIPFTextAlignment = ( itaLeft, itaRight, itaCenter, itaCenterOnePara );

  TParseState = record
    Alignment: TIPFTextAlignment;
    ForegroundColorTag: string;
    BackgroundColorTag: string;
    Spacing: boolean;
    FontState: TFontState;
    LinkIndex: longint;
  end;


  TTopic = class
  protected
    _FileHandle: HFILE;

    _pTOCEntry: pTTOCEntryStart;
    _pSlotOffsets: UInt32ArrayPointer;
    _Slots: TList;
    _pSlotNumbers: puint16;
    _NumSlots: longint;
    _Title: pstring;
    _GlobalDictionary: TList;

    _ShowInContents: boolean;
    _ContentsLevel: integer;
    _ContentsGroupIndex: longint;

    _FontTable: TList;

    procedure SetTitle( const NewValue: string );
    function GetTitle: string;
    function GetTitlePtr: pstring;

    // Returns the tag texts for the given bitmap ref
    function GetImageText( CurrentAlignment: TIPFTextAlignment;
                           BitmapOffset: longint;
                           BitmapFlags: longint;
                           ImageOffsets: TList ): string;

    Procedure ProcessLinkedImage( Var State: TParseState;
                                  Var pData: pByte;
                                  Var OutputString: string;
                                  Var DebugString: string;
                                  Var ImageOffsets: TList );
    procedure TranslateIPFEscapeCode( Var State: TParseState;
                                      Var pData: pUInt8;
                                      Var OutputString: string;
                                      Var DebugString: string;
                                      Var WordsOnLine: longint;
                                      Var ImageOffsets: TList );

    function CreateLink( Var LinkIndex: longint;
                         Var Link: THelpLink;
                         LinkClass: THelpLinkClass ): boolean;

    procedure EnsureSlotsLoaded;

  public
    constructor Create( FileHandle: HFILE;
                        pSlotOffsets: UInt32ArrayPointer;
                        Dictionary: TList;
                        pTOCEntry: pTTOCEntryStart;
                        FontTable: TList );

    destructor destroy; override;

    property Title: string read GetTitle write SetTitle;
    property TitlePtr: pstring read GetTitlePtr;
    procedure SetTitleFromMem( const p: pointer; const Len: byte );

    // Main function for retrieving text for topic.
    // HighlightWords: array indicating whether words
    //   should be highlighted
    // if nil then ignored.
    // ShowCodes: indicates debugging: hex output of escape
    //   codes will be included
    // ShowWordSeparators: | will be included after each dictionary
    //   word inserted
    // Text: The output is written to here.
    // ImageOffsets: For each image that occurs in the text,
    //   the help file offset will be written to this list.
    procedure GetText( HighLightWords: UInt32ArrayPointer;
                       ShowCodes: boolean;
                       ShowWordSeparators: boolean;
                       Text: TAString;
                       ImageOffsets: TList );

    // if StopAtFirstOccurrence true, returns 0 or 1
    // if false, returns count of occurrences of word
    function SearchForWord( DictIndex: integer;
                            StopAtFirstOccurrence: boolean ): longint;

    // searches for sequences out of those listed in WordSequence
    // Each element of WordSequence contains a pointer to an array
    // of flags for each dictionary word, indicating whether that word
    // is to be a possible match.
    function SearchForWordSequences( WordSequence: TList;
                                     StopAtFirstOccurrence: boolean ): longint;

    procedure GetContentsWindowRect( ContentsRect: THelpWindowRect );

  public

    Links: TList; // only valid after GetText

    property ShowInContents: boolean read _ShowInContents;
    property ContentsLevel: integer read _ContentsLevel;
    property ContentsGroupIndex: longint read _ContentsGroupIndex;

    function CountWord( DictIndex: integer ): longint;
    function ContainsWord( DictIndex: integer ): boolean;

    // Used externally
    HelpFile: TObject;
    Index: longint;

    SearchRelevance: longint;
  end;

// Compares two topics for purposes of sorting by
// search match relevance
function TopicRelevanceCompare( Item1, Item2: pointer ): longint;

// Compares two topics for purposes of sorting by
// title
function TopicTitleCompare( Item1, Item2: pointer ): longint;

Implementation

uses
  SysUtils,
  ACLUtility, ACLStringUtility, ACLProfile, ACLFileIOUtility,
  ACLLanguageUnit,
  SettingsUnit;

const
  IPFColors: array[ 0..15 ] of string =
  (
    '', // default
    '#0000ff', // blue
    '#ff0000', // red
    '#ff00ff', // pink (purple)
    '#00ff00', // green
    '#00ffff', // cyan
    '#ffff00', // yellow
    '#808080', // "neutral"
    '#404040', // dark gray
    '#000080', // dark blue
    '#800000', // dark red
    '#800080', // dark pink (purple)
    '#008000', // dark green
    '#008080', // cyan
    '#000000', // black
    '#c0c0c0'  // pale gray
  );

  // for ecHighlight1
  IPFHighlight1Tags : array [ 0..6 ] of string =
  (
    '</i></b></u></color>',  // normal
    '<i>',           // hp1 italitc
    '<b>',           // hp2 bold
    '<b><i>',        // hp3 bold italic
    '<u>',           // hp5 underline
    '<u><i>',        // hp6 underline italic
    '<u><b>'         // hp7 underline bold
  );

  // for ecHighlight2
  IPFHighlight2Tags : array [ 0..3 ] of string =
  (
    '</i></b></u></color>',  // normal
    '<color blue>',  // hp4 blue
    '<color red>',   // hp8 red
    '<color purple>' // hp9 purple
  );

var
  DefaultTitle: string;

Procedure OnLanguageEvent( Language: TLanguageFile;
                           const Apply: boolean );
begin
  Language.Prefix := 'HelpTopic.';
  Language.LL( Apply, DefaultTitle, 'DefaultTitle', '(No title)' );
end;

function GetBeginLink( LinkIndex: longint ): string;
begin
  Result := '<link '
            + IntToStr( LinkIndex )
            + '>'
end;

function GetEndLinkStyle( const State: TParseState ): string;
begin
  Result := '</link>'
            + State.ForegroundColorTag;
end;

// Even though it doesn't do anything,
// we have to have a constructor to allow
// virtual constructors to work
constructor THelpLink.Create;
begin
end;

constructor TWindowedHelpLink.Create;
begin
  GroupIndex := DefaultGroupIndex;
  Automatic := false;
  ViewPort := false;
  Dependent := false;

  Rect := THelpWindowRect.Create;
end;

destructor TWindowedHelpLink.Destroy;
begin
  Rect.Destroy;
end;

destructor THelpTopicSlot.Destroy;
begin
  DeallocateMemory( pData );
  DeallocateMemory( pLocalDictionary );
end;

constructor TTopic.Create( FileHandle: HFILE;
                           pSlotOffsets: UInt32ArrayPointer;
                           Dictionary: TList;
                           pTOCEntry: pTTOCEntryStart;
                           FontTable: TList );
var
  pExtendedInfo: pExtendedTOCEntry;
  titleLen: integer;
  XY: THelpXYPair;
  p: pbyte;

  Flags: byte;

begin
  _FileHandle := FileHandle;
  _pSlotOffsets := pSlotOffsets;

  _Title := nil;
  _GlobalDictionary := Dictionary;
  _ContentsGroupIndex := 0;

  _pTOCEntry := pTOCEntry;
  _NumSlots := pTOCEntry ^. numslots;

  Flags := _pTOCEntry ^. flags;
  p := pUInt8( _pTOCEntry ) + sizeof( TTOCEntryStart );

  if ( Flags and TOCEntryExtended ) > 0 then
  begin
    pExtendedInfo := pExtendedTOCEntry( p );
    inc( p, sizeof( TExtendedTOCEntry ) );

    if ( pExtendedInfo^.w1 and 1 ) > 0 then
      // skip position
      inc( p, sizeof( XY ) );

    if ( pExtendedInfo^.w1 and 2 ) > 0 then
      // skip size
      inc( p, sizeof( XY ) );

    if ( pExtendedInfo^.w1 and 8 ) > 0 then
      // skip window controls
      inc( p, 2 );

    if ( pExtendedInfo^.w1 and $40 ) > 0 then
      // skip something else, unknown... style? 2 bytes
      inc( p, 2 );

    if ( pExtendedInfo^.w2 and 4 ) > 0 then
    begin
      _ContentsGroupIndex := pUInt16( p )^;
      // read group
      inc( p, sizeof( uint16 ) );
    end;
  end;

  // skip slot numbers for now.
  _pSlotNumbers := puint16( p );
  inc( p, _NumSlots * sizeof( uint16 ) );

  titleLen := _pTOCEntry ^.length
              - ( longword( p ) - longword( _pTOCEntry ) );

  // Read title
  if TitleLen > 0 then
    SetTitleFromMem( p, TitleLen )
  else
    Title := DefaultTitle;

  _ContentsLevel := ( Flags and $f );
  _ShowInContents := Flags and TOCEntryHidden = 0;
  if _ContentsLevel = 0 then
    _ShowInContents := false; // hmmm....

  _FontTable := FontTable;
end;

destructor TTopic.Destroy;
begin
  DestroyListAndObjects( Links );
  FreePString( _Title );
  DestroyListAndObjects( _Slots );
end;

procedure TTopic.SetTitle( const NewValue: string );
begin
  FreePString( _Title );
  _Title := NewPString( NewValue );
end;

procedure TTopic.SetTitleFromMem( const p: pointer; const Len: byte );
begin
  FreePString( _Title );
  GetMem( _Title, Len + 1 );
  _Title^[ 0 ] := char( Len );
  MemCopy( p, _Title + 1, Len );
end;

function TTopic.GetTitle: string;
begin
  Result := _Title^;
end;

function TTopic.GetTitlePtr: pstring;
begin
  Result := _Title;
end;

// Replace < and > characters with doubles << and >>
// for compatibility with richtextview.
// This works in place, assuming that instances of > or < are
// actually rare. In practice, IPF normally would insert these
// two characters as distinct words, but I don't want to assume that.
procedure SubstituteAngleBrackets( Var s: string );
var
  i: integer;
begin
  i := 1;
  while i <= Length( S ) do
  begin
    case S[ i ] of
      '<':
      begin
        Insert( '<', s, i );
        inc( i );
      end;

      '>':
      begin
        Insert( '>', s, i );
        inc( i );
      end;
    end;
    inc( i );
  end;
end;

function TTopic.GetImageText( CurrentAlignment: TIPFTextAlignment;
                              BitmapOffset: longint;
                              BitmapFlags: longint;
                              ImageOffsets: TList ): string;
var
  BitmapIndex: longint;
  OriginalAlignTag: string;
  ImageTag: string;
  AlignTag: string;
begin
  BitmapIndex := ImageOffsets.IndexOf( pointer( BitmapOffset ) );
  if BitmapIndex = -1 then
    BitmapIndex := ImageOffsets.Add( pointer( BitmapOffset ) );

  ImageTag := '<image '
              + IntToStr( BitmapIndex )
              + '>';

  if ( BitmapFlags and $08 ) > 0 then
  begin
    // stretch to fit - not implemented
  end;

  // aligned
  case CurrentAlignment of
    itaLeft:
      OriginalAlignTag := '<align left>';
    itaRight:
      OriginalAlignTag := '<align right>';
    itaCenter,
    itaCenterOnePara:
      OriginalAlignTag := '<align center>';
  end;

  case BitmapFlags and 7 of
    1: // left
      AlignTag := '<align left>';
    2: // right
      AlignTag := '<align right>';
    4,5: // centre (4 is official, 5 seems to occur too)
      AlignTag := '<align center>';
  end;

  Result := AlignTag
            + ImageTag
            + OriginalAlignTag;

  if ( BitmapFlags and $10 ) = 0 then
  begin
    // NOT runin, new lines before and after
    Result := #10 + Result + #10;
  end;

end;

Procedure TTopic.ProcessLinkedImage( Var State: TParseState;
                                     Var pData: pByte;
                                     Var OutputString: string;
                                     Var DebugString: string;
                                     Var ImageOffsets: TList );
var
  EscapeLen: uint8;
  EscapeCode: uint8;
  SubEscapeCode: uint8;
  BitmapOffset: longword;
  BitmapFlags: uint8;
  Link: TInternalHelpLink;
  LinkTopicIndex: uint16;
begin
  LinkTopicIndex := -1;
  while true do
  begin
    EscapeLen := pData^;
    SubEscapeCode := ( pData + 2 )^;
    case SubEscapeCode of
      HPART_DEFINE:
      begin
        BitmapFlags := ( pData + 3 )^;
        BitmapOffset := pUInt32( pData + 4 )^;
      end;

      HPART_HDREF: // define whole bitmap topic link?
      begin
        LinkTopicIndex := pUInt16( pData + 3 )^;
      end
      else
      begin
        // ignore others for now
        DebugString := DebugString + ' ?';
      end;
    end;
    inc( pData, EscapeLen );

    // Now pData points at next code or item
    if pData^ <> IPF_ESC then
      // not an escape code, done
      break;
    EscapeCode := (pData + 2) ^;
    if EscapeCode <> ecLinkedImage then
      // not a hyperlink code, done
      break;
    // another linked image code is coming up.
    SubEscapeCode := ( pData + 3 )^;
    if SubEscapeCode = HPART_DEFINE then
      // started another linked image.
      break;
    inc( pData ); // move pointer to escape code len.
  end;

  OutputString := GetImageText( State.Alignment,
                                BitmapOffset,
                                BitmapFlags,
                                ImageOffsets );

  // Don't make it a link if we didn't find a
  // overall link code, i.e. degrade gracefully.
  if LinkTopicIndex > -1 then
  begin
    if CreateLink( State.LinkIndex, Link, TInternalHelpLink ) then
    begin
      Link.TopicIndex := LinkTopicIndex;
    end;

    OutputString := GetBeginLink( State.LinkIndex )
                    + OutputString
                    + GetEndLinkStyle( State );

    inc( State.LinkIndex );
  end;

end;

Procedure GetExtraLinkData( Link: TWindowedHelpLink;
                            pData: pUInt8 );
var
  LinkFlags1: uint8;
  LinkFlags2: uint8;
  LinkDataIndex: longint;
  pLinkXY: pHelpXYPair;
  pLinkData: pUInt8;
begin
  LinkFlags1 := ( pData + 0 ) ^;
  LinkFlags2 := ( pData + 1 ) ^;

  pLinkData := pData + 2;

  if ( LinkFlags1 and 1 ) > 0 then
  begin
    // position specified
    pLinkXY := pHelpXYPair( pLinkData );
    ReadHelpPosition( pLinkXY^, Link.Rect );
    inc( pLinkData, sizeof( THelpXYPair ) );
  end;

  if ( LinkFlags1 and 2 ) > 0 then
  begin
    // size specified
    pLinkXY := pHelpXYPair( pLinkData );
    ReadHelpSize( pLinkXY^, Link.Rect );
    inc( pLinkData, sizeof( THelpXYPair ) );
  end;

  if ( LinkFlags1 and 8 ) > 0 then
  begin
    // window controls specified - skip
    inc( pLinkData, 2 );
  end;

  if ( LinkFlags2 and 4 ) > 0 then
  begin
    // group specified
    Link.GroupIndex := pUInt16( pLinkData )^;
    inc( LinkDataIndex, sizeof( uint16 ) );
  end;

  if ( LinkFlags1 and 64 ) > 0 then
  begin
    Link.Automatic := true;
  end;

  if ( LinkFlags1 and 4 ) > 0 then
    Link.ViewPort := true;

  if ( LinkFlags2 and 2 ) > 0 then
    Link.Dependent := true;

  if ( LinkFlags1 and 128 ) > 0 then
    Link.Split := true;

  // cant be bothered with the others.
end;

// If the given link has already been decoded
// ie. the topic has been displayed before,
// then return the already decoded link & return false
// Otherwise, create a new link object & return true
function TTopic.CreateLink( Var LinkIndex: longint;
                            Var Link: THelpLink;
                            LinkClass: THelpLinkClass ): boolean;
begin
  if LinkIndex >= Links.Count then
  begin
    Link := LinkClass.Create;
    Link.HelpFile := HelpFile;
    Links.Add( Link );
    Result := true;
  end
  else
  begin
    Link := Links[ LinkIndex ];
    Result := false;
  end;
end;

const
  // size of the original View's default font
  AverageViewCharWidth = 8;

procedure GetMarginTag( const Margin: longint;
                        FontState: TFontState;
                        Var MarginString: string );
begin
  MarginString := '<leftmargin ';
  if FontState <> fsCustom then
    // for standard fonts, scale margins to match font
    MarginString := MarginString + IntToStr( Margin ) + '>'
  else
    // for custom fonts, since the IPF margins were always in
    // terms of the standard font size, set the margin to a width based on that.
    MarginString := MarginString + IntToStr( Margin * AverageViewCharWidth ) + ' pixels>';
end;

procedure TTopic.TranslateIPFEscapeCode( Var State: TParseState;
                                         Var pData: pUInt8;
                                         Var OutputString: string;
                                         Var DebugString: string;
                                         Var WordsOnLine: longint;
                                         Var ImageOffsets: TList );
var
  EscapeLen: uint8;
  EscapeCode: uint8;

  Link: TInternalHelpLink;
  FootnoteLink: TFootnoteHelpLink;
  LinkByResourceID: THelpLinkByResourceID;

  Margin: integer;
  EscCodeDataIndex: longint;

  BitmapOffset: longword;
  BitmapFlags: uint8;

  ColorCode: uint8;
  StyleCode: uint8;

  FontIndex: uint8;
  pFontSpec: pTHelpFontSpec;

  FaceName: string;
  PointSize: longint;
  QuotedFaceName: string;

  ExternalLinkFileIndex: uint8;
  ExternalLinkTopicID: string;
begin
  EscapeLen := pData^;
  EscapeCode := (pData + 1) ^;

  OutputString := '';

  DebugString := IntToHex( EscapeCode, 2 ) + ' ';
  for EscCodeDataIndex := 2 to EscapeLen - 1 do
  begin
    DebugString := DebugString
                   + ' '
                   + IntToHex( ( pData + EscCodeDataIndex )^, 2 );
  end;

  case EscapeCode of

    ecSetLeftMargin:
    begin
      Margin := integer( ( pData + 2 )^ ) - 1;
      if Margin < 0 then
        Margin := 0;
      GetMarginTag( Margin, State.FontState, OutputString );
    end;

    ecSetLeftMarginNewLine:
    begin
      Margin := integer( ( pData + 2 )^ ) - 1;
      if Margin < 0 then
        Margin := 0;
      GetMarginTag( Margin, State.FontState, OutputString );
      OutputString := OutputString
                      + #10;
    end;

    ecSetLeftMarginFit:
    begin
      Margin := integer( ( pData + 2 )^ ) - 1;
      if Margin < 0 then
        Margin := 0;
      GetMarginTag( Margin, State.FontState, OutputString );
      // note that this will cause following tex to be "tabbed" across to the
      // new margin position, if not yet there.
    end;

    ecHighlight1:
    begin
      StyleCode := ( pData + 2 ) ^;
      if StyleCode <= High( IPFHighlight1Tags ) then
        OutputString := IPFHighlight1Tags[ StyleCode ];
      if StyleCode = 0 then
        State.ForegroundColorTag := '</color>';
    end;

    ecHighlight2:
    begin
      StyleCode := ( pData + 2 ) ^;
      if StyleCode <= High( IPFHighlight2Tags ) then
        OutputString := IPFHighlight2Tags[ StyleCode ];
      if StyleCode = 0 then
        State.ForegroundColorTag := '</color>'
      else
        State.ForegroundColorTag := OutputString; // only colours
    end;

    ecLinkStart:
    begin
      if CreateLink( State.LinkIndex, Link, TInternalHelpLink ) then
      begin
        Link.TopicIndex := pUInt16( pData + 2 )^;

        if EscapeLen >= 6 then
        begin
          GetExtraLinkData( Link, pData + 4 );
        end;
      end;

      // If it's not an automatic link
      // then put code in to show it.
      if not Link.Automatic then
      begin
        OutputString := '<blue>'
                        + GetBeginLink( State.LinkIndex );
      end;

      inc( State.LinkIndex );
    end;

    ecFootnoteLinkStart:
    begin
      if CreateLink( State.LinkIndex, FootnoteLink, TFootnoteHelpLink ) then
      begin
        FootnoteLink.TopicIndex:= pUInt16( pData + 2 )^;
      end;

      OutputString := '<blue>'
                      + GetBeginLink( State.LinkIndex );

      inc( State.LinkIndex );
    end;

    ecStartLinkByResourceID:
    begin
      if CreateLink( State.LinkIndex, LinkByResourceID, THelpLinkByResourceID ) then
      begin
        LinkByResourceID.ResourceID := pUInt16( pData + 2 )^;

        if EscapeLen >= 6 then
        begin
          GetExtraLinkData( LinkByResourceID, pData + 4 );
        end;
      end;

      OutputString := '<blue>'
                      + GetBeginLink( State.LinkIndex );

      inc( State.LinkIndex );
    end;

    ecExternalLink:
    begin
      ExternalLinkFileIndex := ( pData + 2 )^;
      ExternalLinkTopicID := StrNPas( pchar( pData + 4 ), ( pData + 3 )^ );
      OutputString := '<blue><link external '
                      + IntToStr( ExternalLinkFileIndex )
                      + ' '
                      + ExternalLinkTopicID
                      + '>'

    end;

    ecLinkEnd:
    begin
      OutputString := GetEndLinkStyle( State );
    end;

    ecStartCharGraphics:
    begin
      State.FontState := fsFixed;
      OutputString := #10 + '<tt><wrap no>';
      State.Spacing := false;
      WordsOnLine := 0;
    end;

    ecEndCharGraphics:
    begin
      State.FontState := fsNormal;
      OutputString := '</tt><wrap yes>';
      State.Spacing := true;
    end;

    ecImage:
    begin
      BitmapFlags := ( pData + 2 )^;
      BitmapOffset := pUInt32( pData + 3 )^;

      OutputString := GetImageText( State.Alignment,
                                    BitmapOffset,
                                    BitmapFlags,
                                    ImageOffsets );
      if State.Spacing then
        OutputString := OutputString + ' ';
    end;

    ecLinkedImage:
    begin
      ProcessLinkedImage( State,
                          pData,
                          OutputString,
                          DebugString,
                          ImageOffsets );
      if State.Spacing then
        OutputString := OutputString + ' ';

      // Note! Early exit, since the procedure
      // will update pData.
      exit;
    end;

    ecStartLines:
    begin
      // aligned text
      case ( pData + 2 )^ of
        1:
        begin
          OutputString := #10 + '<align left>';
          State.Alignment := itaLeft;
        end;

        2:
        begin
          OutputString := #10 + '<align right>';
          State.Alignment := itaRight;
        end;

        4:
        begin
          OutputString := #10 + '<align center>';
          State.Alignment := itaCenter;
        end;
      end;
      OutputString := OutputString + '<wrap no>';
      State.Spacing := false;
      WordsOnLine := 0;
    end;

    ecEndLines:
    begin
      // supposed to turn word wrap on, default font
      OutputString := '<align left><wrap yes>'; // I guess...
      State.Alignment := itaLeft;
      State.Spacing := true;
    end;

    ecForegroundColor:
    begin
      ColorCode := ( pData + 2 )^;
      if ColorCode = 0 then
        State.ForegroundColorTag := '</color>'
      else if ColorCode <= High( IPFColors ) then
        State.ForegroundColorTag := '<color ' + IPFColors[ ColorCode ] + '>';
      OutputString := State.ForegroundColorTag;
    end;

    ecBackgroundColor:
    begin
      ColorCode := ( pData + 2 )^;
      if ColorCode = 0 then
        State.BackgroundColorTag := '</backcolor>'
      else if ColorCode <= High( IPFColors ) then
        State.BackgroundColorTag := '<backcolor ' + IPFColors[ ColorCode ] + '>';
      OutputString := State.BackgroundColorTag;
    end;

    ecFontChange:
    begin
      FontIndex := ( pData + 2 )^;
      if FontIndex = 0 then
      begin
        // back to default font
        OutputString := '</font>';
        State.FontState := fsNormal;
      end
      else if FontIndex < _FontTable.Count then
      begin
        // valid font index
        pFontSpec := _FontTable[ FontIndex ];
        FaceName := StrNPas( pFontSpec ^. FaceName,
                             sizeof( pFontSpec ^. FaceName ) );

        // arbitrarily and capriciously use specified height * 2/3
        // as the point size - seems to correspond to what original
        // view wanted...  note this doesn't necessarily scale
        // correctly, since default font could be different. whatever.
        PointSize := pFontSpec ^. Height * 2 div 3;

        if PointSize < 8 then
          PointSize := 8;
        // quote font name, escape double quotes with duplicates
        // e.g. Bob's "Big" Font would become
        //      "Bob's ""Big"" Font"
        QuotedFaceName := DoubleQuote
                          + InsertDuplicateChars( FaceName,
                                                  DoubleQuote )
                          + DoubleQuote;
        OutputString := '<font '
                        + QuotedFaceName
                        + ' '
                        + IntToStr( PointSize )
                        + '>';
                        {
                         // for when (if ever) RTV allows setting font
                         // by precise dimensions
                        + '['
                        + IntToStr( pFontSpec ^. Width )
                        + 'x'
                        + IntToStr( pFontSpec ^. Height )
                        + ']';
                        }
        State.FontState := fsCustom;
      end;
    end

    else
    begin
      // Unknown/unhandled code
      DebugString := DebugString + ' ?';
    end;

  end; // case escape code of...
  inc( pData, EscapeLen );
end;

procedure TTopic.EnsureSlotsLoaded;
var
  i: longint;
  pSlotNumber: puint16;
  SlotNumber: uint16;
  SlotHeader: TSlotHeader;
  Slot: THelpTopicSlot;
begin
  if _Slots = nil then
  begin
    _Slots := TList.Create;

    // Read slot data
    pSlotNumber := _pSlotNumbers;

    for i := 0 to _NumSlots - 1 do
    begin
      SlotNumber := pSlotNumber^;

      // Seek to start of slot
      MySeek( _FileHandle,
              _pSlotOffsets^[ SlotNumber ] );

      // Read header
      MyRead( _FileHandle,
              Addr( SlotHeader ),
              sizeof( SlotHeader ) );

      // Create slot object
      Slot := THelpTopicSlot.Create;

      Slot.LocalDictSize := SlotHeader.nLocalDict;
      Slot.Size := SlotHeader.ntext;

      // Allocate and read slot dictionary
      ReadFileBlock( _FileHandle,
                     Slot.pLocalDictionary,
                     SlotHeader.localDictPos,
                     uint32( Slot.LocalDictSize ) * sizeof( uint16 ) );

      // Allocate and read slot data (text)
      ReadFileBlock( _FileHandle,
                     Slot.pData,
                     _pSlotOffsets^[ SlotNumber ] + sizeof( TSlotHeader ),
                     Slot.Size );

      _Slots.Add( Slot );

      inc( pSlotNumber, sizeof( UInt16 ) );
    end;
  end;
end;

const
  BlankString: string = '';

// Main translation function. Turns the IPF data into
// a text string. Translates formatting codes into tags
// as for Rich Text Viewer.
// Uses TAString for speed without length limits
// - string is too short
// - PChar is slow to concatenate (unless you keep track of the insert point)
// - AnsiString is slow
procedure TTopic.GetText( HighLightWords: UInt32ArrayPointer;
                          ShowCodes: boolean;
                          ShowWordSeparators: boolean;
                          Text: TAString;
                          ImageOffsets: TList );
var
  SlotIndex: integer;
  Slot: THelpTopicSlot;
  pData: pUInt8;
  pSlotEnd: pUInt8;

  GlobalDictIndex: uint32;

  WordsOnLine: longint;

  StringToAdd: string;
  LocalDictIndex: uint8;
  DebugString: string;
  EscapeDebugString: string;

  State: TParseState;
begin
  if Links = nil then
    Links := TList.Create;

  EnsureSlotsLoaded;

  Text.Clear;

  ImageOffsets.Clear;

  WordsOnLine := 0;

  State.LinkIndex := 0;
  State.FontState := fsNormal; // ? Not sure... this could be reset at start of slot
  State.Spacing := true;
  State.ForegroundColorTag := '</color>';
  State.BackgroundColorTag := '</backcolor>';

  for SlotIndex := 0 to _NumSlots - 1 do
  begin
    if State.FontState <> fsFixed then
      State.Spacing := true; // this is just a guess as to the exact view behaviour.
                             // inf.txt indicates that spacing is reset to true at
                             // slot (cell) start, but that doesn't seem to be the
                             // case when in fixed font... hey ho.

    Slot := _Slots[ SlotIndex ];

    pData := Slot.pData;

    pSlotEnd := pData + Slot.Size;

    State.Alignment := itaLeft;

    while pData < pSlotEnd do
    begin
      LocalDictIndex := pData^;

      if LocalDictIndex < Slot.LocalDictSize then
      begin
        // Normal word lookup
        GlobalDictIndex := Slot.pLocalDictionary^[ LocalDictIndex ];

        // normal lookup
        if GlobalDictIndex < _GlobalDictionary.Count then
          StringToAdd := pstring( _GlobalDictionary[ GlobalDictIndex ] )^
        else
          StringToAdd := '';

        SubstituteAngleBrackets( StringToAdd );

        if HighlightWords <> nil then
          if HighlightWords[ GlobalDictIndex ] > 0 then
            StringToAdd := '<backcolor #'
                           + IntToHex( Settings.Colors[ SearchHighlightTextColorIndex ], 6 )
                           + '>'
                           + StringToAdd
                           + State.BackgroundColorTag;

        Text.AddString( StringToAdd );

        if State.Spacing then
          Text.AddString( ' ' );
        if ShowWordSeparators then
          Text.AddString( '[' + IntToStr( GlobalDictIndex )+ ']' );

        inc( WordsOnLine );
        inc( pData );
      end
      else
      begin
        // special code
        StringToAdd := '';
        if ShowCodes then
          DebugString := '[' + IntToHex( LocalDictIndex, 2 );
        case LocalDictIndex of
          IPF_END_PARA:
          begin
            StringToAdd := '';
            if State.Alignment = itaCenterOnePara then
            begin
              State.Alignment := itaLeft;
              StringToAdd := '<align left>';
            end;
            StringToAdd := StringToAdd + #10;

            if WordsOnLine > 0 then
              StringToAdd := StringToAdd + #10;

            if SlotIndex = 0 then
              if pData = Slot.pData then
                // ignore first FA, not needed with RichTextView
                StringToAdd := '';

            if State.FontState <> fsFixed then
              State.Spacing := true;
            WordsOnLine := 0;
            inc( pData );
          end;

          IPF_CENTER:
          begin
            StringToAdd := #10 + '<align center>';
            State.Alignment := itaCenterOnePara;
            inc( pData );
          end;

          IPF_INVERT_SPACING:
          begin
            State.Spacing := not State.Spacing;
            inc( pData );
          end;

          IPF_LINEBREAK:
          begin
            StringToAdd := '';
            if State.Alignment = itaCenterOnePara then
            begin
              State.Alignment := itaLeft;
              StringToAdd := '<align left>';
            end;
            StringToAdd := StringToAdd + #10;
            if State.FontState <> fsFixed then
              State.Spacing := true;
            WordsOnLine := 0;
            inc( pData );
          end;

          IPF_SPACE:
          begin
            if State.Spacing then
              StringToAdd := '  '
            else
              StringToAdd := ' ';
            inc( pData );
          end;

          IPF_ESC:
          begin
            // escape sequence
            inc( pData );
            TranslateIPFEscapeCode( State,
                                    pData,
                                    StringToAdd,
                                    EscapeDebugString,
                                    WordsOnLine,
                                    ImageOffsets );

            if ShowCodes then
              DebugString := DebugString + ' ' + EscapeDebugString;
          end // case code of ff:

          else
          begin
            // Unrecongised code
            if ShowCodes then
              DebugString := DebugString + '?';
            inc( pData );
          end;

        end; // case code of...
        if ShowCodes then
          Text.AddString( DebugString + ']' );

        Text.AddString( StringToAdd );
      end;
    end; // for slotindex = ...
  end;

end;

function TTopic.SearchForWord( DictIndex: integer;
                               StopAtFirstOccurrence: boolean )
  : longint;
var
  SlotIndex: integer;
  Slot: THelpTopicSlot;
  pData: pUInt8;
  pSlotEnd: pUInt8;

  EscapeLen: longint;

  GlobalDictIndex: uint32;

  LocalDictIndex: uint8;
begin
  EnsureSlotsLoaded;

  Result := 0;
  for SlotIndex := 0 to _NumSlots - 1 do
  begin
    Slot := _Slots[ SlotIndex ];

    pData := Slot.pData;

    pSlotEnd := pData + Slot.Size;

    while pData < pSlotEnd do
    begin
      LocalDictIndex := pData^;

      if LocalDictIndex < Slot.LocalDictSize then
      begin
        // Normal word lookup
        GlobalDictIndex:= Slot.pLocalDictionary^[ LocalDictIndex ];

        if GlobalDictIndex = DictIndex then
        begin
          inc( result );
          if StopAtFirstOccurrence then
            exit;
        end;
      end
      else
      begin
        // special code
        if GlobalDictIndex = $ff then
        begin
          // escape string, skip it
          EscapeLen := ( pData + 1 ) ^;
          inc( pData, EscapeLen );
        end;
      end;

      inc( pData );
    end; // for slotindex = ...
  end;
end;

function TTopic.SearchForWordSequences( WordSequence: TList;
                                        StopAtFirstOccurrence: boolean )
  : longint;
var
  SlotIndex: integer;
  Slot: THelpTopicSlot;
  pData: pUInt8;
  pSlotEnd: pUInt8;

  EscapeLen: longint;

  GlobalDictIndex: uint32;
  IsWord: boolean;
  WordRelevance: uint32;

  CurrentMatchRelevance: uint32; // total relevances for words matched so far
                                 // in the current sequence

//  CurrentMatch: string;  // useful for debugging only
  LocalDictIndex: uint8;

  SequenceIndex: longint;
  SequenceStartSlotIndex: longint;
  pSequenceStartData: pUInt8;

  pStepWordRelevances: UInt32ArrayPointer; // word relevances for the current step in the sequence

  // get the current slot start and end pointers
  procedure GetSlot;
  begin
    Slot := self._Slots[ SlotIndex ];
    pData := Slot.pData;
    pSlotEnd := pData + Slot.Size;
  end;

  // get pointer to the current set of word relevances
  procedure GetStepFlags;
  begin
    pStepWordRelevances := WordSequence[ SequenceIndex ];
  end;

  // store the current point as start of a sequence
  procedure StoreStartOfSequence;
  begin
    SequenceIndex := 0;
    SequenceStartSlotIndex := SlotIndex;
    pSequenceStartData := pData;
    CurrentMatchRelevance := 0;
//    CurrentMatch := '';
    GetStepFlags;
  end;

begin
  Result := 0;

  EnsureSlotsLoaded;

  if _NumSlots = 0 then
    // thar's nowt in yon topic, cannae be a match laid
    exit;

  SlotIndex := 0;

  GetSlot;

  StoreStartOfSequence;

  while true do
  begin
    LocalDictIndex := pData^;
    IsWord := false;
    if LocalDictIndex < Slot.LocalDictSize then
    begin
      IsWord := true;
      // Normal word lookup, so get the global dict idnex before we
      // (potentially) move to next slot
      GlobalDictIndex := Slot.pLocalDictionary^[ LocalDictIndex ];
    end;

    inc( pData );
    if pData >= pSlotEnd then
    begin
      // reached end of slot, next please
      inc( SlotIndex );
      if SlotIndex < _NumSlots then
        GetSlot;
      // else - there is nothing more to search
      // but we need to check this last item
    end;

    if IsWord then
    begin
      // Normal word lookup
      WordRelevance := 0;

      if GlobalDictIndex < _GlobalDictionary.Count then
        WordRelevance := pStepWordRelevances[ GlobalDictIndex ];

      if WordRelevance > 0 then
      begin
        // Found a matching word
        inc( CurrentMatchRelevance, WordRelevance );
//        CurrentMatch := CurrentMatch +
//          pstring( _GlobalDictionary[ GlobalDictIndex ] )^;

        if SequenceIndex = 0 then
        begin
          // remember next start point
          SequenceStartSlotIndex := SlotIndex;
          pSequenceStartData := pData;
        end;

        inc( SequenceIndex );

        if SequenceIndex < WordSequence.Count then
        begin
          // get next set of flags.
          GetStepFlags;
        end
        else
        begin
          // found a complete sequence. Cool!

          inc( result, CurrentMatchRelevance );

          if StopAtFirstOccurrence then
            exit;

          // start looking from the beginning of the sequence again.
          StoreStartOfSequence;
        end;
      end
      else
      begin
        // not a match at this point, restart search
        if SequenceIndex > 0 then
        begin
          // we had matched one or more steps already,
          // back to start of sequence AND back to
          // point we started matching from (+1)
          SequenceIndex := 0;
          CurrentMatchRelevance := 0;
//          CurrentMatch := '';
          SlotIndex := SequenceStartSlotIndex;
          GetSlot;
          pData := pSequenceStartData;
          GetStepFlags;
        end
        else
        begin
          // haven't matched anything yet.
          // update start of sequence
          SequenceStartSlotIndex := SlotIndex;
          pSequenceStartData := pData;
        end;
      end;
    end
    else
    begin
      // special code
      if LocalDictIndex = $ff then
      begin
        // escape string, skip it
        EscapeLen := pData ^;
        inc( pData, EscapeLen );
      end;
    end;

    if SlotIndex >= _NumSlots then
    begin
      // finished searching topic
      break;
    end;

    // next item
  end;
end;


function TTopic.CountWord( DictIndex: integer ): longint;
begin
  Result := SearchForWord( DictIndex, false );
end;

function TTopic.ContainsWord( DictIndex: integer ): boolean;
begin
  Result := SearchForWord( DictIndex, true ) > 0;
end;

// Gets the window dimensions specified by this topic's
// contents header
procedure TTopic.GetContentsWindowRect( ContentsRect: THelpWindowRect );
var
  extendedinfo: TExtendedTOCEntry;
  XY: THelpXYPair;
  p: pbyte;

  Flags: byte;
begin
  Flags := _pTOCEntry ^.flags;
  p := pByte( _pTOCEntry + sizeof( TTOCEntryStart ) );

  ContentsRect.Left := 0;
  ContentsRect.Bottom := 0;
  ContentsRect.Width := 100;
  ContentsRect.Height := 100;

  if ( Flags and TOCEntryExtended ) > 0 then
  begin
    // have more details available...
    ExtendedInfo.w1 := p^;
    ExtendedInfo.w2 := ( p+1) ^;
    inc( p, sizeof( ExtendedInfo ) );

    if (  ExtendedInfo.w1 and 1 ) > 0 then
    begin
      // read origin
      XY := pHelpXYPair( p )^;
      inc( p, sizeof( XY ) );
      ReadHelpPosition( XY, ContentsRect );
    end;
    if ( ExtendedInfo.w1 and 2 ) > 0 then
    begin
      // read size
      XY := pHelpXYPair( p )^;
      inc( p, sizeof( XY ) );
      ReadHelpSize( XY, ContentsRect );
    end;
  end;
end;

// Compares two topics for purposes of sorting by
// search match relevance
function TopicRelevanceCompare( Item1, Item2: pointer ): longint;
var
  Topic1, Topic2: TTopic;
begin
  Topic1 := Item1;
  Topic2 := Item2;

  if Topic1.SearchRelevance > Topic2.SearchRelevance then
    Result := -1
  else if Topic1.SearchRelevance < Topic2.SearchRelevance then
    Result := 1
  else
    Result := 0;
end;

// Compares two topics for purposes of sorting by
// title
function TopicTitleCompare( Item1, Item2: pointer ): longint;
begin
  Result := CompareText( TTopic( Item1 )._Title^,
                         TTopic( Item2 )._Title^ );
end;

Initialization
   RegisterProcForLanguages( OnLanguageEvent );
End.
