Unit MainFormUnit;

Interface

Uses
  Classes, Forms, Graphics, ExtCtrls, Buttons, StdCtrls, TabCtrls, ComCtrls;

Type
  TMainForm = Class (TForm)
    Image1: TImage;
    CancelButton: TButton;
    BackButton: TButton;
    NextButton: TButton;
    Notebook: TNoteBook;
    Label2: TLabel;
    Label8: TLabel;
    Label9: TLabel;
    Label10: TLabel;
    Label11: TLabel;
    InstallProgressBar: TProgressBar;
    Label12: TLabel;
    RunAppCheckBox: TCheckBox;
    Label13: TLabel;
    Bevel1: TBevel;
    InstallToSourceCheckbox: TCheckBox;
    InstallTypeRadioGroup: TRadioGroup;
    InstallFolderLabel: TLabel;
    InstallFolderEdit: TEdit;
    ChooseInstallFolderButton: TButton;
    CreateIconCheckBox: TCheckBox;
    InstallTypeHelpLabel: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    RestartRequiredLabel: TLabel;
    RestartCheckBox: TCheckBox;
    Label1: TLabel;
    Label7: TLabel;
    Label15: TLabel;
    Label16: TLabel;
    Procedure InstallToSourceCheckboxOnClick (Sender: TObject);
    Procedure InstallTypeRadioGroupOnClick (Sender: TObject);
    Procedure MainFormOnCreate (Sender: TObject);
    Procedure MainFormOnCloseQuery (Sender: TObject; Var CanClose: Boolean);
    Procedure RestartCheckBoxOnClick (Sender: TObject);
    Procedure ChooseInstallFolderButtonOnClick (Sender: TObject);
    Procedure Memo2OnChange (Sender: TObject);
    Procedure BackButtonOnClick (Sender: TObject);
    Procedure NextButtonOnClick (Sender: TObject);
    Procedure MainFormOnShow (Sender: TObject);
    Procedure Label5OnClick (Sender: TObject);
    Procedure CancelButtonOnClick (Sender: TObject);
  Protected
    Procedure EnableButtons;
    Function SkipPage( Page: longint ): boolean;
    Function GetInstallType: longint;

    FCancelled: boolean;
    FAllowClose: boolean;
    FAppInUse: boolean;
    FDLLInUse: boolean;
    FApplicationInstallPath: string;

    Function Install: boolean;
    Function InstallFile( const Source: string;
                          const Dest: string;
                          const Backup: string;
                          const IsModule: boolean; // true if an executeable module
                          var DestInUse: boolean ): boolean;
    Procedure RunNewView;
    Procedure RefreshInstallTypeHelp;
  End;

Var
  MainForm: TMainForm;

Implementation

uses
  BseDos, OS2Def, BseErr, PmWp, PmWin, PmShl, PmErr,
  SysUtils, Dos, Dialogs,
  ACLUtility, ACLFileUtility, ACLStringUtility, ACLDialogs,
  ControlsUtility,
  ChooseFolderFormUnit;

{$R NewViewInstall}

const
  pgWelcome = 0;
  pgInstallType = 1;
  pgInstallFolder = 2;
  pgReady = 3;
  pgInstalling = 4;
  pgDone = 5;

const // install types
  itComplete = 0;
  itViewOnly = 1;
  itStandAlone = 2;

Procedure TMainForm.InstallToSourceCheckboxOnClick (Sender: TObject);
Begin
  if InstallToSourceCheckbox.Checked then
    InstallFolderLabel.PenColor := clBtnShadow
  else
    InstallFolderLabel.ParentPenColor := true;

  InstallFolderEdit.Enabled := not InstallToSourceCheckbox.Checked;
  ChooseInstallFolderButton.Enabled := not InstallToSourceCheckbox.Checked;
End;

Function TMainForm.GetInstallType: longint;
begin
  Result := InstallTypeRadioGroup.ItemIndex;
end;

Procedure TMainForm.InstallTypeRadioGroupOnClick (Sender: TObject);
Begin
  RefreshInstallTypeHelp;
End;

Procedure TMainForm.RefreshInstallTypeHelp;
var
  Help: string;
begin
  case InstallTypeRadioGroup.ItemIndex of
    itComplete:
      Help := 'This option will replace both online help and help icons. '
              + 'It will backup and replace View.exe and HelpMgr.dll.';
    itViewOnly:
      Help := 'This option will replace help icons only, by '
              + 'backing up and replacing View.exe.';
    itStandAlone:
      Help := 'This option will not change the existing help system. '
              + 'NewView will be installed as a normal application.';
  end;

  InstallTypeHelpLabel.Caption := Help;
end;

Procedure TMainForm.MainFormOnCreate (Sender: TObject);
Begin
  // set up form icons
  Forms.FormIconResourceID := 1;
  Font := GetNiceDefaultFont;
End;

Procedure TMainForm.MainFormOnCloseQuery (Sender: TObject;
  Var CanClose: Boolean);
Begin
  if FAllowClose then
  begin
    Canclose := true;
    exit;
  end;

  FCancelled := true;
  CanClose := false;
End;

Procedure TMainForm.RestartCheckBoxOnClick (Sender: TObject);
Begin
  RunAppCheckBox.Enabled := not RestartCheckBox.Checked;
  if RestartCheckbox.Checked then
    RunAppCheckBox.Checked := false;
End;

Procedure TMainForm.ChooseInstallFolderButtonOnClick (Sender: TObject);
Begin
  ChooseFolderForm.Folder := InstallFolderEdit.Text;
  if ChooseFolderForm.ShowModal <> mrOK then
    exit;
  InstallFolderEdit.Text := ChooseFolderForm.Folder;
End;

Procedure TMainForm.Memo2OnChange (Sender: TObject);
Begin

End;

Procedure TMainForm.BackButtonOnClick (Sender: TObject);
var
  PreviousPage: longint;
Begin
  PreviousPage := Notebook.PageIndex - 1;
  while SkipPage( PreviousPage ) do
    dec( PreviousPage );

  Notebook.PageIndex := PreviousPage;

  EnableButtons;
End;

Function TMainForm.SkipPage( Page: longint ): boolean;
begin
  Result := false;
  if Page = pgInstallFolder then
    if GetInstallType <> itStandAlone then
      result := true;
end;

Procedure TMainForm.NextButtonOnClick (Sender: TObject);
var
  i : integer;
  NextPage: longint;
Begin
  FCancelled := false;

  NextPage := Notebook.PageIndex + 1;
  while SkipPage( NextPage ) do
    inc( NextPage );

  Notebook.PageIndex := NextPage;

  EnableButtons;

  case Notebook.PageIndex of
    pgInstalling:
    begin
      FAllowClose := false;

      if not Install then
      begin
        FAllowClose := true;
        Close;
        exit;
      end;

      FAllowClose := true;

      RestartRequiredLabel.Visible := FDLLInUse or FAppInUse;

      if FDLLInUse then
      begin
        RestartRequiredLabel.Caption :=
          'NOTE: You will need to restart your computer for '
          + 'the installation to take effect.';

        RestartCheckBox.Visible := true;
      end
      else
      begin
        RestartCheckBox.Visible := false;

        if FAppInUse then
        begin
          RestartRequiredLabel.Caption :=
            'NewView is currently running. Restart it to activate the new version.';
          RunAppCheckBox.Checked := false;
          RunAppCheckBox.Enabled := false;

        end;
      end;

      Notebook.PageIndex := Notebook.PageIndex + 1;

      EnableButtons;

    end;

    pgDone:
    begin
      Close;
      if RestartCheckBox.Checked then
      begin
        // prevent ourselves from hanging the shutdown
        WinShutdownSystem( AppHandle, HMQ_CURRENT )
      end
      else if RunAppCheckBox.Checked then
      begin
        RunNewView;
      end;
    end;
  end;
End;

Procedure TMainForm.MainFormOnShow (Sender: TObject);
Begin
  Notebook.PageIndex := 0;
  EnableButtons;
  InstallFolderEdit.Text := GetBootDrive + ':\NewView';
  RefreshInstallTypeHelp;

  FAllowClose := true;
End;

Procedure TMainForm.EnableButtons;
Begin
  BackButton.Visible :=     ( Notebook.PageIndex > 0 )
                        and ( Notebook.PageIndex < pgInstalling );
  NextButton.Enabled :=    ( Notebook.PageIndex < pgInstalling )
                        or ( Notebook.PageIndex = pgDone );

  if Notebook.PageIndex < pgReady then
    NextButton.Caption := '~Next >'
  else if Notebook.PageIndex = pgReady then
    NextButton.Caption := '~Install >'
  else
    NextButton.Caption := '~Close';

  CancelButton.Enabled := Notebook.PageIndex < pgDone;

End;

Procedure TMainForm.Label5OnClick (Sender: TObject);
Begin

End;

Procedure TMainForm.CancelButtonOnClick (Sender: TObject);
Begin
  Close;
End;

Imports
  Function DosReplaceModule( pszOldModule: pchar;
                             pszNewModule: pchar;
                             pszBackupModule: pchar )
    : APIRET;
    apientry;
    'DosCalls' index 417;
  end;

// Install specified module from source, to dest.
// If backup is not '' then the original file will
// be copied to Backup.
// If the file is in use then:
// If IsModule is false then the install will fail.
// If IsModule is true then DosReplaceModule will
// be used to unlock the module, and DestInUse will be set true.
Function TMainForm.InstallFile( const Source: string;
                                const Dest: string;
                                const Backup: string;
                                const IsModule: boolean;
                                var DestInUse: boolean ): boolean;
var
  rc: APIRET;
  szDest: cstring;
  szSource: cstring;
  szBackup: cstring;
  FileHandle: HFILE;
  ActionTaken: ULONG;
begin
  Result := false;
  DestInUse := false;

  // Check the source file exists.
  if not FileExists( Source ) then
  begin
    DoErrorDlg( 'Internal Error',
                'The file '
                + Source
                + ' was not found for installation' );
    exit;
  end;

  // Convert to null-terminated strings
  szDest := Dest;
  szSource := Source;
  szBackup := Backup;

  // If the destination exists, unlock and back it up
  if FileExists( Dest ) then
  begin
    if FileIsReadOnly( Dest ) then
    begin
      DoErrorDlg( 'Installation Error',
                  'The file ' + EndLine
                  + ' ' + Dest + EndLine
                  + 'is read-only and cannot be replaced.' );
      exit;
    end;
    // see if it's in use.
    rc := DosOpen( szDest,
                   FileHandle,
                   ActionTaken,
                   0, // new size: not used
                   0, // attributes: not used
                   OPEN_ACTION_FAIL_IF_NEW
                   + OPEN_ACTION_OPEN_IF_EXISTS,
                   OPEN_FLAGS_FAIL_ON_ERROR
                   + OPEN_SHARE_DENYREADWRITE
                   + OPEN_ACCESS_READWRITE,
                   nil ); // e.a.s: not used
    DosClose( FileHandle );

    if rc = ERROR_SHARING_VIOLATION then
    begin
      // file in use
      DestInUse := true;

      if not IsModule then
      begin
        // Show error. It would be nicer to
        // fall back on alternative update method e.g.
        // locked file device driver IBMLANLK.SYS
        // But that's overkill for NewView
        DoErrorDlg( 'Installation Error',
                    'This file is in use: ' + EndLine
                    + ' ' + Dest + EndLine
                    + 'and cannot be replaced.' );
        exit;
      end;

      // unlock the module
      rc := DosReplaceModule( Addr( szDest ),
                              nil,
                              nil );

      if rc <> 0 then
      begin
        // error
        DoErrorDlg( 'Install Error',
                    'Could not unlock ' + EndLine
                    + ' ' + Dest + EndLine
                    + SysErrorMessage( rc ) );

        exit;
      end;
    end
    else if rc <> 0 then
    begin
      DoErrorDlg( 'Install Error',
                  'Unable to acces ' + Endline
                  + ' ' + Dest + EndLine
                  + SysErrorMessage( rc ) );
      exit;
    end;

    // OK, done...

    if Backup <> '' then
    begin
      // make backup if it doesn't already exist.
      if not FileExists( Backup ) then
      begin
        rc := DosCopy( szDest,
                       szBackup,
                       0 ); // no special options (don't overwrite).
        if rc <> 0 then
        begin
          // error
          DoErrorDlg( 'Install Error',
                      'Could not backup ' + EndLine
                      + ' ' + Dest + EndLine
                      + ' to' + EndLine
                      + ' ' + Backup + EndLine
                      + EndLine
                      + SysErrorMessage( rc ) );
          exit;
        end;
      end;
    end;
  end;

  // OK, now copy the new file on
  rc := DosCopy( szSource,
                 szDest,
                 DCPY_EXISTING ); // overwrite
  if rc <> 0 then
  begin
    // error
    DoErrorDlg( 'Install Error',
                'Could not copy new file ' + EndLine
                + ' ' + Source + EndLine
                + ' to' + EndLine
                + ' ' + Dest + EndLine
                + EndLine
                + SysErrorMessage( rc ) );
    exit;
  end;

  // done
  result := true;
end;

Function TMainForm.Install: boolean;
var
  SourceDir: string;
  InstallDir: string;
  SystemDir: string;
  SystemDLLDir: string;
  HelpFileInUse: boolean;
  AppBackupPath: string;
  ProgramObjectHandle: HOBJECT;
  szSetupString: cstring;
  PMError: ERRORID;
  LanguageFiles: TStringList;
  i: longint;
  LanguageDir: string;
  HelpFileDir: string;
  CopyFilesRequired: boolean;
begin
  Result := false;

  FAppInUse := false;
  FDLLInUse := false;

  InstallProgressBar.Position := 0;
  Application.ProcessMessages;

  SourceDir := GetApplicationDir;

  SystemDir := GetBootDrive
               + ':\os2\';
  SystemDLLDir := SystemDir
                  + 'dll\';

  // get/validate/create folders

  CopyFilesRequired := true;

  case GetInstallType of
    itStandAlone:
    begin
      // validate/create install dir
      InstallDir := AddSlash( InstallFolderEdit.Text );
      if InstallToSourceCheckbox.Checked then
      begin
        InstallDir := SourceDir;
        CopyFilesRequired := false;
      end
      else if not DirectoryExists( InstallDir ) then
      begin
        try
          MakeDirs( InstallDir );
        except
          on E: EInOutError do
          begin
            DoErrorDlg( 'Folder Error',
                        'Could not create the installation folder '
                        + InstallDir + EndLine
                        + SysErrorMessage( E.ErrorCode ) );
            exit;
          end;
        end;
      end
      else if UpperCase( SourceDir ) = UpperCase( InstallDir ) then
      begin
        CopyFilesRequired := false;
      end;

      FApplicationInstallPath := InstallDir + 'NewView.exe';
      // nothing to back up if not replacing original view
      AppBackupPath := '';

      HelpFileDir := InstallDir;
      LanguageDir := InstallDir;
    end;

    itViewOnly,
    itComplete:
    begin
      InstallDir := SystemDir;
      FApplicationInstallPath := InstallDir + 'View.exe';
      AppBackupPath := InstallDir + 'view.bak';

      // validate system path
      if not DirectoryExists( SystemDir ) then
      begin
        DoErrorDlg( 'System Folder Error',
                    'The system folder '
                    + SystemDir
                    + ' does not exist!' );
        exit;
      end;

      if GetInstallType = itComplete then
      begin
        // validate system DLL path
        if not DirectoryExists( SystemDLLDir ) then
        begin
          DoErrorDlg( 'System Folder Error',
                      'The system DLL folder '
                      + SystemDLLDir
                      + ' does not exist!' );
          exit;
        end;
      end;

      HelpFileDir := SystemDir + 'book\';
      if not DirectoryExists( HelpFileDir ) then
      begin
        DoErrorDlg( 'System Folder Error',
                    'The system help folder '
                    + HelpFileDir
                    + ' does not exist!' );
        exit;
      end;

      // delete old newview.inf help file
      // (shouldn't have gone in the \os2\ dir)
      if FileExists( SystemDir + 'newview.inf' ) then
        DeleteFile( SystemDir + 'newview.inf' );

      LanguageDir := SystemDir; // for now.
    end;
  end;

  // ShowMessage( 'Language: ' + LanguageDir );
  // ShowMessage( 'Help: ' + HelpFileDir );

  InstallProgressBar.Position := 15;

  if CopyFilesRequired then
  begin
    // install exe
    if not InstallFile( SourceDir + 'NewView.exe',
                        FApplicationInstallPath,
                        AppBackupPath,
                        true,
                        FAppInUse ) then
      exit;

    InstallProgressBar.Position := 30;

    // install DLL
    if GetInstallType = itComplete then
    begin
      if not InstallFile( SourceDir + 'HelpMgr.dll',
                          SystemDLLDir + 'HelpMgr.dll',
                          SystemDLLDir + 'HelpMgr.bak',
                          true,
                          FDLLInUse ) then
        exit;
    end;

    InstallProgressBar.Position := 50;

    // Install Help File
    if not InstallFile( SourceDir + 'NewView.inf',
                        HelpFileDir + 'NewView.inf',
                        '', // no backup
                        false, // not in use
                        HelpFileInUse ) then
        exit;

    InstallProgressBar.Position := 70;

    LanguageFiles := TStringList.Create;
    ListDirectory( SourceDir,
                   '*.lng',
                   LanguageFiles,
                   nil ); // don't need subdirs

    for i := 0 to LanguageFiles.Count - 1 do
    begin
      // ShowMessage( 'Lang file: ' + LanguageFiles[ i ] );
      // Install language File
      if not InstallFile( SourceDir + LanguageFiles[ i ],
                          LanguageDir + LanguageFiles[ i ],
                          '', // no backup
                          false, // not in use
                          HelpFileInUse ) then
        exit;
    end;
    LanguageFiles.Destroy;
  end;

  InstallProgressBar.Position := 85;

  if CreateIconCheckBox.Checked then
  begin
    szSetupString := 'PROGTYPE=PM;EXENAME='
                     + FApplicationInstallPath
                     + ';OBJECTID=<NewView>';
    ProgramObjectHandle :=
      WinCreateObject( 'WPProgram', // class
                       'NewView Help Viewer',
                       szSetupString, // setup string
                       '<WP_DESKTOP>',
                       CO_UPDATEIFEXISTS );

    if ProgramObjectHandle = NULLHANDLE then
    begin
      PMError := WinGetLastError( AppHandle );

      // Handle a few specific errors

      case ( PMError and $ffff ) of
        WPERR_INVALID_FOLDER:
          DoErrorDlg( 'Warning',
                      'Unable to create desktop icon:' + EndLine
                      + IntToHex( PMError, 8 )
                      + ': The desktop is not correctly installed '
                      + '(<WP_DESKTOP> missing). ' );

        WPERR_NOT_WORKPLACE_CLASS:
          DoErrorDlg( 'Warning',
                      'Unable to create desktop icon:' + EndLine
                      + IntToHex( PMError, 8 )
                      + ': WPProgram class is missing.' );

        else
          DoErrorDlg( 'Installation Error',
                      'Unable to create desktop icon' + EndLine
                      + IntToHex( PMError, 8 )
                      + ': There may be some problem with the desktop.' );
      end;
    end;
  end;

  InstallProgressBar.Position := 100;

  Result := true;

end;

Procedure TMainForm.RunNewView;
begin
  Exec( FApplicationInstallPath, '' );
end;

Initialization
  RegisterClasses ([TMainForm, TImage, TLabel, TButton, TNoteBook,
    TCheckBox, TProgressBar, TBevel, TRadioGroup, TEdit]);
End.
