Unit MainForm;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

Uses
// system is a good unit to be able to open ;)
// the above line is here so you can use right-mouse open file on "system"
  OS2Def,
  SysUtils, Classes, Forms, Graphics, Messages,
  Buttons, ComCtrls, StdCtrls, ExtCtrls, TabCtrls,
  Dialogs,
// library
  ACLString, SharedMemoryUnit, ACLLanguageUnit,
// custom components
  SplitBar, Outline2, RichTextView, Coolbar2,
  CustomListBox,
// local
  HelpFile, HelpTopic, HelpWindowUnit, HelpWindowDimensions,
  NavigatePointUnit, BookmarkUnit, HelpManagerUnit,
  DataTypes;

const
  // Custom window messages for this form
  // NOTE! Sibyl uses WM_USER+1 and +2!
  WM_OPENED     = WM_USER + 10;
  WM_FOLLOWLINK = WM_USER + 11;

Type
  TWindowPosition = record
    Left: longint;
    Bottom: longint;
    Width: longint;
    Height: longint;
  end;

  TMainForm = Class (TForm)
    VSplitBar: TSplitBar;
    Notebook: TNoteBook;
    IndexSearchEdit: TEdit;
    SearchTextEdit: TEdit;
    SearchResultsListBox: TListBox;
    NotesListBox: TListBox;
    TabSet: TTabSet;
    CoolBar: TCoolBar2;
    AddNoteButton: TButton;
    EditNoteButton: TButton;
    DeleteNoteButton: TButton;
    GotoNoteButton: TButton;
    DebugCrashTestMI: TMenuItem;
    DebugLoadLanguageMI: TMenuItem;
    MenuItem23: TMenuItem;
    MenuItem24: TMenuItem;
    MenuItem25: TMenuItem;
    MenuItem26: TMenuItem;
    DebugSaveLanguageFileMI: TMenuItem;
    MenuItem28: TMenuItem;
    MenuItem22: TMenuItem;
    DebugMemoryFormMI: TMenuItem;
    DebugHelpManagerVersionMI: TMenuItem;
    DebugTopicByResourceIDMI: TMenuItem;
    MenuItem19: TMenuItem;
    ViewHighlightSearchWordsMI: TMenuItem;
    DebugHelpMgrMI: TMenuItem;
    MenuItem17: TMenuItem;
    MenuItem16: TMenuItem;
    MenuItem15: TMenuItem;
    FileNewWindowMI: TMenuItem;
    OpenSpecialMI: TMenuItem;
    MenuItem14: TMenuItem;
    SearchPMI: TMenuItem;
    ViewSourceMI: TMenuItem;
    FileCloseMI: TMenuItem;
    MenuItem10: TMenuItem;
    MenuItem12: TMenuItem;
    MenuItem13: TMenuItem;
    StatusPanel: TPanel;
    ProgressPanel: TPanel;
    ProgressBar: TProgressBar;
    DebugShowWordSeparatorsMI: TMenuItem;
    DebugStressTestMI: TMenuItem;
    TopicPropertiesPMI: TMenuItem;
    MenuItem3: TMenuItem;
    MenuItem2: TMenuItem;
    MenuItem8: TMenuItem;
    ToolsOptionsMI: TMenuItem;
    MenuItem9: TMenuItem;
    ToolsDebugMenu: TMenuItem;
    DebugShowParamsMI: TMenuItem;
    DebugShowCodesMI: TMenuItem;
    EditGlobalSearchMI: TMenuItem;
    ViewExpandAllMI: TMenuItem;
    EditBookmarksMI: TMenuItem;
    AddBookmarkMI: TMenuItem;
    ViewPopupMenu: TPopupMenu;
    SelectAllPMI: TMenuItem;
    CopyPMI: TMenuItem;
    ContentsOutline: TOutline2;
    IndexListBox: TCustomListBox;
    ViewCollapseAllMI: TMenuItem;
    MenuItem7: TMenuItem;
    SystemOpenDialog: TSystemOpenDialog;
    SearchButton: TButton;
    MenuItem1: TMenuItem;
    ViewIndexMI: TMenuItem;
    ViewContentsMI: TMenuItem;
    ViewSearchMI: TMenuItem;
    ViewNotesMI: TMenuItem;
    ViewRefreshMI: TMenuItem;
    MenuItem11: TMenuItem;
    DisplayPanel: TPanel;
    ButtonImages: TImageList;
    BookmarksMenu: TMenuItem;
    MainMenu: TMainMenu;
    FileMenu: TMenuItem;
    OpenMI: TMenuItem;
    FileSaveAsMI: TMenuItem;
    PrintMI: TMenuItem;
    FileInformationMI: TMenuItem;
    MenuItem4: TMenuItem;
    ExitMI: TMenuItem;
    EditMenu: TMenuItem;
    SelectAllMI: TMenuItem;
    CopyMI: TMenuItem;
    MenuItem5: TMenuItem;
    FindMI: TMenuItem;
    FindNextMI: TMenuItem;
    NavigateMenu: TMenuItem;
    NavigateBackMI: TMenuItem;
    NavigateForwardMI: TMenuItem;
    MenuItem6: TMenuItem;
    NavigateNextMI: TMenuItem;
    NavigatePreviousMI: TMenuItem;
    ToolsMenu: TMenuItem;
    GlobalSearchMI: TMenuItem;
    HelpMenu: TMenuItem;
    HelpMI: TMenuItem;
    HelpProductInformationMI: TMenuItem;
    AddNoteMI: TMenuItem;

    Procedure ContentsOutlineOnItemClick (Node: TNode);
    Procedure DebugCrashTestMIOnClick (Sender: TObject);
    Procedure DebugLoadLanguageMIOnClick (Sender: TObject);
    Procedure DebugSaveLanguageFileMIOnClick (Sender: TObject);
    Procedure SearchResultsListBoxOnClick (Sender: TObject);
    Procedure DebugMemoryFormOnClick (Sender: TObject);
    Procedure VSplitBarOnDblClick (Sender: TObject);
    Procedure DebugHelpManagerVersionMIOnClick (Sender: TObject);
    Procedure DebugTopicByResourceIDMIOnClick (Sender: TObject);
    Procedure ViewHighlightSearchWordsMIOnClick (Sender: TObject);
    Procedure DebugHelpMgrMIOnClick (Sender: TObject);
    Procedure FileNewWindowMIOnClick (Sender: TObject);
    Procedure OpenSpecialMIOnClick (Sender: TObject);
    Procedure NotesListBoxOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure ViewPopupMenuOnPopup (Sender: TObject);
    Procedure SearchPMIOnClick (Sender: TObject);
    Procedure ViewSourceMIOnClick (Sender: TObject);
    Procedure PrintMIOnClick (Sender: TObject);
    Procedure DebugShowWordSeparatorsMIOnClick (Sender: TObject);
    Procedure DebugStressTestMIOnClick (Sender: TObject);
    Procedure TopicPropertiesPMIOnClick (Sender: TObject);
    Procedure NavigateForwardMIOnClick (Sender: TObject);
    Procedure IndexListBoxOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure SearchResultsListBoxOnScan (Sender: TObject;
      Var KeyCode: TKeyCode);
    Procedure ContentsOutlineOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure ToolsOptionsMIOnClick (Sender: TObject);
    Procedure EditGlobalSearchMIOnClick (Sender: TObject);
    Procedure DebugShowParamsMIOnClick (Sender: TObject);
    Procedure ViewExpandAllMIOnClick (Sender: TObject);
    Procedure EditBookmarksMIOnClick (Sender: TObject);
    Procedure CopyPMIOnClick (Sender: TObject);
    Procedure SelectAllPMIOnClick (Sender: TObject);
    Procedure AddNoteButtonOnClick (Sender: TObject);
    Procedure SearchTextEditOnChange (Sender: TObject);
    Procedure IndexListBoxOnClick (Sender: TObject);
    Procedure ViewCollapseAllMIOnClick (Sender: TObject);
    Procedure NotesListBoxOnDblClick (Sender: TObject);
    Procedure HelpMIOnClick (Sender: TObject);
    Procedure NotebookOnPageChanged (Sender: TObject);
    Procedure ViewRefreshMIOnClick (Sender: TObject);
    Procedure ViewNotesMIOnClick (Sender: TObject);
    Procedure ViewSearchMIOnClick (Sender: TObject);
    Procedure ViewIndexMIOnClick (Sender: TObject);
    Procedure ViewContentsMIOnClick (Sender: TObject);
    Procedure MainFormOnCloseQuery (Sender: TObject; Var CanClose: Boolean);
    Procedure GlobalSearchMIOnClick (Sender: TObject);
    Procedure GotoNoteButtonOnClick (Sender: TObject);
    Procedure EditNoteButtonOnClick (Sender: TObject);
    Procedure NotesListBoxOnItemFocus (Sender: TObject; Index: LongInt);
    Procedure DeleteNoteButtonOnClick (Sender: TObject);
    Procedure AddBookmarkMIOnClick (Sender: TObject);
    Procedure AddNoteMIOnClick (Sender: TObject);
    Procedure FileCloseMIOnClick (Sender: TObject);
    Procedure CoolBarOnSectionResize (HeaderControl: THeaderControl;
      section: THeaderSection);
    Procedure CoolBarOnSectionClick (HeaderControl: THeaderControl;
      section: THeaderSection);
    Procedure MainFormOnDestroy (Sender: TObject);
    Procedure MainFormOnSetupShow (Sender: TObject);
    Procedure MainFormOnCreate (Sender: TObject);
    Procedure MainFormOnShow (Sender: TObject);
    Procedure FindNextMIOnClick (Sender: TObject);
    Procedure FindMIOnClick (Sender: TObject);
    Procedure IndexSearchEditOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure IndexSearchEditOnChange (Sender: TObject);
    Procedure FileInformationMIOnClick (Sender: TObject);
    Procedure SearchTextEditOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure SearchButtonOnClick (Sender: TObject);
    Procedure FileSaveAsMIOnClick (Sender: TObject);
    Procedure OptionsMIOnClick (Sender: TObject);
    Procedure TabSetOnChange (Sender: TObject; NewTab: LongInt;
      Var AllowChange: Boolean);
    Procedure NotebookOnSetupShow (Sender: TObject);
    Procedure NavigateBackMIOnClick (Sender: TObject);
    Procedure NavigatePreviousMIOnClick (Sender: TObject);
    Procedure NavigateNextMIOnClick (Sender: TObject);
    Procedure CopyMIOnClick (Sender: TObject);
    Procedure SelectAllMIOnClick (Sender: TObject);
    Procedure DebugShowCodesMIOnClick (Sender: TObject);
    Procedure HelpProductInformationMIOnClick (Sender: TObject);
    Procedure OnOverLink ( Sender: TRichTextView; LinkString: String);
    Procedure OnNotOverLink ( Sender: TRichTextView; LinkString: String);
    Procedure OnClickLink ( Sender: TRichTextView; LinkString: String);
    Procedure BackButtonOnClick (Sender: TObject);
    Procedure RTViewOnSetupShow (Sender: TObject);
    Procedure OpenMIOnClick (Sender: TObject);
    Procedure ExitMIOnClick (Sender: TObject);

    Procedure MainFormOnResize (Sender: TObject);
    Procedure VSplitBarOnChange (NewSplit: LongInt);
  Protected
    // Custom window messages ----------------------------------

    // Handle our own WM_OPENED message
    Procedure WMOpened( Var Msg: TMessage ); Message WM_OPENED;
    Procedure WMFollowLink( Var Msg: TMessage ); Message WM_FOLLOWLINK;

    // Messages from new help manager
    Procedure NHMDisplayIndex( Var Msg: TMessage ); Message NHM_HELP_INDEX;
    Procedure NHMDisplayContents( Var Msg: TMessage ); Message NHM_HELP_CONTENTS;
    Procedure NHMTopicByResourceID( Var Msg: TMessage ); Message NHM_TOPIC_BY_RESOURCE_ID;
    Procedure NHMTopicByPanelName( Var Msg: TMessage ); Message NHM_TOPIC_BY_PANEL_NAME;

    Procedure NHMTest( Var Msg: TMessage ); Message NHM_TEST;

  Protected
    // GUI events set by code ----------------------------------

    Procedure OnHint( Sender: TObject );
    Procedure OnWindowClose( Window: THelpWindow );
    Procedure OnWindowAboutToClose( Window: THelpWindow;
                                    var CanClose: boolean );
    Procedure OnNavigateToMenuItemClick( Sender: TObject );

    Procedure OnException( Sender: TObject;
                           E: Exception );

  Public

    // Open the file or list of files in FileNames
    // Set the window title if given, otherwise get it from first file
    Function OpenFiles( const FileNames: TStrings;
                        const WindowTitle: string;
                        const DisplayFirstTopic: boolean ): boolean;

    // Open a single file
    Function OpenFile( const FileName: string;
                       const WindowTitle: string;
                       const DisplayFirstTopic: boolean ): boolean;

    Procedure CloseFile;

    Procedure TranslateIPFEnvironmentVars( Items: TStrings;
                                           ExpandedItems: TStrings );

    function DisplayTopicByResourceID( ID: uint16 ): boolean;

    Procedure DisplayIndex;
    Procedure DisplayContents;

  Protected
    // Startup functions ----------------------------------

    Procedure CheckEnvironmentVars;
    Procedure ParseCommandLineParameters;
    Procedure ShowUsage;

    // Loading functions ----------------------------------

    // Most recently used files list
    Procedure CreateMRUMenuItems;
    Procedure OnMRUMenuItemClick( Sender: TObject );

    // Given a "filename" which may include a path find
    // it it in various paths and extensions
    Function FindHelpFile( FileName: string ): string;

    Procedure OnHelpFileLoadProgress( n, outof: integer;
                                      message: string );

    // Returns nil if file is not open
    Function FindOpenHelpFile( FileName: string ): THelpFile;

    // Navigation -------------------------------------------------

    Procedure SaveNavigatePoint;
    Procedure SaveWindows( SourceList: TList;
                           DestList: TList;
                           Parent: TSavedHelpWindow );

    Procedure UpdateCurrentNavigatePoint;
    Procedure ClearPageHistory;

    Procedure NavigateToPoint( NavPoint: TNavigatePoint );
    Procedure NavigateToHistoryIndex( Index: longint );

    Procedure DisplayWindows( WindowList: TList;
                              Parent: THelpWindow );
    Procedure ShowWindows;
    Procedure ShowWindowList( WindowList: TList );
    Procedure CloseWindows;

    Procedure FocusFirstHelpWindow;

    Procedure NavigateBack;
    Procedure NavigateForward;
    Procedure NavigatePreviousInContents;
    Procedure NavigateNextInContents;

    Procedure CreateNavigateToMenuItems;

    // GUI status updates ---------------------------------

    Procedure EnableControls;
    Procedure EnableSearchButton;
    Procedure SetStatus( Text: String );
    Procedure ResetProgress;
    Procedure RefreshWindows( WindowList: TList );

    // language stuff
    Procedure LoadLanguage( const FilePath: string );
    Procedure LoadDefaultLanguage;
    function LoadAutoLanguage( const Filename: string ): boolean;

    // called by callback
    Procedure OnLanguageEvent( Language: TLanguageFile;
                               const Apply: boolean );

    // Loading views --------------------------------------

    // Used in loading contents
    Procedure AddChildNodes( HelpFile: THelpFile;
                             ParentNode: TNode;
                             Level: longint;
                             Var TopicIndex: longint );
    Procedure LoadContents;
    Procedure LoadIndex;

    // Note manipulations --------------------------------

    procedure AddNote;
    Procedure EditNote( NoteIndex: longint );
    procedure DeleteNote( NoteIndex: longint );
    Procedure SaveNotes;
    Procedure SaveNotesForFile( HelpFile: THelpFile );
    Procedure LoadNotes( HelpFile: THelpFile );
    Procedure GotoCurrentNote;

    // make sure that note insert positions are not in
    // the middle of tags due to help file or newview updates.
    Procedure CorrectNotesPositions( Topic: TTopic;
                                     Text: pchar );

    Procedure InsertNotesIntoTopicText( Topic: TTopic;
                                        Text: TAString );
    function FindOriginalNoteCharIndex( NoteCharIndex: longword;
                                        Topic: TTopic ): longword;
    function FindActualNoteCharIndex( NoteCharIndex: longword;
                                      MaxNoteIndex: longword;
                                      Topic: TTopic ): longword;
    procedure RefreshNoteInsertInfo( NoteIndex: longword );
    procedure ClearNotes;

    Procedure EnableNotesControls;
    Procedure UpdateNotesDisplay;

    // GUI actions ------------------------------------------

    procedure FileOpen;
    Procedure PrintTopics;
    Procedure DoFind( FindOrigin: TFindOrigin );

    // Bookmarks ------------------------------------------

    Procedure NavigateToBookmark( Bookmark: TBookmark );
    Procedure BuildBookmarksMenu;
    Procedure UpdateBookmarksForm;
    Procedure BookmarksMenuItemClick( Sender: TObject );
    procedure AddBookmark;
    procedure ClearBookmarks;
    procedure SaveBookmarks;
    procedure SaveBookmarksForFile( HelpFile: THelpFile );
    procedure LoadBookmarks( HelpFile: THelpFile );
    procedure OnBookmarksChanged( Sender: TObject );

    // Global search -------------------------------------

    procedure DoGlobalSearch( const SearchText: string );
    // Called when viewing topics from global search
    Procedure OnViewGlobalSearchTopic( FileName: string;
                                       TopicIndex: longint );

    // Options and appearance -----------------------------

    procedure DoOptions;

    procedure ApplySettings;

    // Retrieve control colours (in case drag'n'drop used to change)
    Procedure GetColors;
    // Set the layout of the main form
    Procedure SetLayout;
    // Lay out the specified list of help windows
    Procedure LayoutWindowList( WindowList: TList );
    // Setup the rich text views in the specified windows (e.g for changing global settings)
    Procedure SetupViews( WindowList: TList );

    // Topic display -------------------------------------

    // Major display topic function.
    procedure DisplayTopic( Topic: TTopic );

    Procedure DisplaySelectedIndexTopic;
    Procedure DisplaySelectedSearchResultTopic;
    Procedure DisplaySelectedContentsTopic;

    Procedure DisplayTopicInWindow( Window: THelpWindow;
                                    FollowAutoLinks: boolean;
                                    KeepPosition: boolean );

    function OpenWindow( Topic: TTopic;
                         Group: longint;
                         Parent: THelpWindow;
                         Rect: THelpWindowRect;
                         FollowAutoLinks: boolean ): THelpWindow;

    Procedure RemoveHelpWindowFromParent( Window: THelpWindow );

    Procedure FollowLink( Link: THelpLink;
                          SourceWindow: THelpWindow );

    Function FindTopicByResourceID( ID: uint16 ): TTopic;

    Function FindTopicForLink( Link: THelpLink ): TTopic;

    Function FindWindowFromView( View: TRichTextView; WindowList: TList ): THelpWindow;
    Function FindWindowFromGroup( Group: longint; WindowList: TList ): THelpWindow;
    Function FindWindowFromTopic( Topic: TTopic; WindowList: TList ): THelpWindow;
    Function GetActiveWindow: THelpWindow;

    function GetOwnHelpFileName: string;

    Procedure DoSearch( const SearchText: string );

    Procedure SetMainCaption;

    // cancel help manager mode
    Procedure ClearHelpManager;

    Files: TList; // current open help files.
    MRUMenuItems: TList; // most recently used file list
    NavigateToMenuItems: TList;
    MainTitle: string;

    // Current topic has the vague meaning that it was the last
    // topic selected by the user... (?)
    CurrentTopic: TTopic;

    // use during decode...
    TopicText: TAString;

    Windows: TList; // top level help windows

    PageHistory: TStringList; // history
    CurrentHistoryIndex: longint; // where we are in history

    Navigating: boolean; // true while going to a particular history point

    DisplayedIndex: TStringList; // duplicate of index listbox,
                                 // for fast case insensitive searching
    InIndexSearch: boolean; // true while searching index

    FindText: string; // last text found (Ctrl-F)

    Notes: TList; // Notes in current files.

    Bookmarks: TList;
    BookmarksMenuItems: TList;

    // while loading... so owe can display progress
    LoadingFilenameList: TStringList;
    LoadingFileIndex: integer;
    LoadingTotalSize: longint;
    LoadingSizeSoFar: longint;

    TopicDecodeRecusionLevel: longint;

    // Help manager mode
    MessageMemory: TSuballocatedSharedMemory;

    // Startup parameters

    Startup_OwnHelpMode: boolean; // indicates displaying NewView's own help file
    Startup_ShowUsageFlag: boolean;
    Startup_TopicParam: string;
    Startup_FilenamesParam: TAString;
    Startup_GlobalSearchText: string;
    Startup_GlobalSearchFlag: boolean;
    Startup_OwnerWindow: HWND;
    Startup_HelpManagerWindow: HWND;
    Startup_IsHelpManager: boolean;
    Startup_WindowTitle: string;
    Startup_Position: TWindowPosition;
    Startup_SetPosition: boolean;

  protected
    // language stuff
    FileOpenTitle: string;
    LoadingFileMsg: string;
    HelpFileError: string;
    LoadingStatusDisplaying: string;
    LoadingStatusNotesAndBookmarks: string;
    LoadingStatusContents: string;
    LoadingStatusIndex: string;
    LoadingStatusDone: string;

    AllFilesDesc: string;
    HelpFilesDesc: string;
    LanguageFilesDesc: string;

    SaveLanguageTitle: string;
    OpenLanguageTitle: string;
    SaveLanguageError: string;

    HelpManagerVersionTitle: string;

    FindResourceIDTitle: string;
    FindResourceIDPrompt: string;
    InvalidResourceIDError: string;
    ResourceIDNotFoundError: string;

    OpenSpecialTitle: string;
    OpenSpecialPrompt: string;

    PrintTopicTitle: string;
    NoPrinterError: string;
    SelectWindowToPrintError: string;
    PrintingError: string;

    TopicInfoTitle: string;
    TopicInfoTopicTitle: string;
    TopicInfoIndex: string;
    TopicInfoFile: string;
    TopicInfoResourceIDs: string;
    TopicInfoNoResourceIDs: string;

    ParameterCountLabel: string;

    NewViewHelpTitle: string;
    AlreadyNewviewHelp: string;
    NewViewHelpNotFound: string;

    InvalidLinkErrorTitle: string;
    InvalidLinkError: string;
    InvalidResourceIDLinkErrorA: string;
    InvalidResourceIDLinkErrorB: string;

    OpenedTopicMsg: string;

    AddNoteTitle: string;
    AddNoteCursorError: string;
    NoteWithinNoteError: string;
    LoadNotesTitle: string;
    LoadNotesError: string;
    SaveNotesTitle: string;
    SaveNotesError: string;

    UntitledBookmarkName: string;
    LoadBookmarksTitle: string;
    LoadBookmarksError: string;
    SaveBookmarksTitle: string;
    SaveBookmarksError: string;

    ApplicationErrorTitle: string;
    ApplicationErrorA: string;
    ApplicationErrorB: string;
    ApplicationErrorC: string;

    EnvironmentVarErrorTitle: string;
    EnvironmentVarError: string;
    EnvironmentVarUndefined: string;

    FindTitle: string;
    FindSelectWindowError: string;
    FindPrompt: string;
    TextNotFoundMsg: string;

    FilesInfoTitle: string;
    FilesInfoOverallTitle: string;
    FilesInfoFilename: string;
    FilesInfoFileTitle: string;
    FilesInfoTopicCount: string;
    FilesInfoIndexCount: string;
    FilesInfoDictionaryCount: string;
    FilesInfoFileSize: string;
    FilesInfoTotalTopicCount: string;
    FilesInfoTotalIndexCount: string;
    FilesInfoTotalFileSize: string;

    SearchTitle: string;
    SearchSyntaxError: string;
    NoSearchMatchesMsg: string;
    SearchFoundMsgA: string;
    SearchFoundMsgB: string;

    FileSaveTitle: string;
    FileSaveSelectWindowError: string;
    DefaultSaveTopicFilename: string;
    ReplaceFilePromptA: string;
    ReplaceFilePromptB: string;
    UnableToSaveError: string;

    UsageTitle: string;
    UsageText1: string;
    UsageText2: string;
    UsageText3: string;
    UsageText4: string;
    UsageText5: string;

    GoBackHint: string;

    SelectAllTitle: string;
    SelectAllWindowError: string;

    EditNoteMsg: string;
    ExternalLinkMsg: string;
    LinkMsg: string;
    UnknownLinkMsg: string;

    ExternalLinkTitle: string;
    ExternalLinkError: string;

    MRUMultipleFilesHint: string;

    HelpProgramTitle: string;

  End;

Var
  MainForm: TMainForm;

Implementation

uses
  BseDos, BseErr, PMWin,
  Dos,
  Printers,
  // Library
  ACLStringUtility, AStringUtilityUnit,
  ACLFileUtility, ACLFileIOUtility, ACLUtility,
  ACLProfile, ACLDialogs, ACLString, ACLLibraryVersionUnit,
  // Components
  RichTextPrintUnit, RichTextStyleUnit, RichTextDocumentUnit, ComponentsVersionUnit,
  ControlsUtility,
  // local: forms
  InformationFormUnit, OptionsForm, ProductInformationFormUnit, NoteForm,
  GlobalSearchForm, FileDialogForm, BookmarksFormUnit,
  PrintDialogUnit, MemoryFormUnit,
  // local: others
  SettingsUnit, VersionUnit,
  TextSearchQuery, SearchUnit, HelpNoteUnit;

{$R Images}

const
  // Coolbar button indexes
  ciOpen = 0;
  ciBack = 1;
  ciForward = 2;
  ciPrint = 3;
  ciAddNote = 4;
  ciAddBookmark = 5;
  ciPrevious = 6;
  ciNext = 7;

  // Page indexes.
  piContents = 0;
  piIndex = 1;
  piSearch = 2;
  piNotes = 3;

  HelpPathEnvironmentVar = 'HELP';
  BookshelfEnvironmentVar = 'BOOKSHELF';

  CrashLogFileName = 'NewView.log';

  _MAX_PATH = 260;

var
  StartMem: longword;
  LastMem: longword;

// --- TMainForm implementation ---

Procedure TMainForm.ContentsOutlineOnItemClick (Node: TNode);
Begin
  DisplaySelectedContentsTopic;
End;

Procedure TMainForm.DebugCrashTestMIOnClick (Sender: TObject);
var
  p: pbyte;
Begin
  p := nil;
  p ^ := 0;
End;

Procedure TMainForm.SetMainCaption;
begin
  if MainTitle = '' then
    Caption := HelpProgramTitle
  else
    Caption := HelpProgramTitle + ' - ' + MainTitle
end;

Procedure TMainForm.LoadDefaultLanguage;
var
  LanguageVar: string;
  MajorLanguage: string;
  MinorLanguage: string;
begin
  LanguageVar := GetEnv( 'LANG' );
  if LanguageVar = '' then
    LanguageVar := 'EN_US';

  MajorLanguage := ExtractNextValue( LanguageVar, '_' );
  MinorLanguage := LanguageVar;
  if MinorLanguage <> '' then
    if LoadAutoLanguage( MajorLanguage
                         + '_'
                         + MinorLanguage
                         + '.lng' ) then
      // found a specifc language
      exit;

  // try generic language?
  if not LoadAutoLanguage( MajorLanguage
                           + '.lng' ) then
    // load defaults
    LoadLanguage( '' );

end;

// load a language from the app's dir/path?
function TMainForm.LoadAutoLanguage( const Filename: string ): boolean;
var
  FilePath: string;
begin
  FilePath := GetApplicationDir + Filename;
  if not FileExists( FilePath ) then
  begin
    Result := false;
    exit;
  end;

  LoadLanguage( FilePath );
end;

procedure TMainForm.LoadLanguage( const FilePath: string );
var
  NewLanguage: TLanguageFile;
begin
  try
    NewLanguage := TLanguageFile.Create( FilePath );
  except
    exit;
  end;

  // get rid of mru menu items
  DestroyListObjects( MRUMenuItems );
  MRUMenuItems.Clear;

  ApplyLanguage( NewLanguage );

  SetMainCaption;
  CreateMRUMenuItems;
end;

Procedure TMainForm.OnLanguageEvent( Language: TLanguageFile;
                                     const Apply: boolean );
Begin
  Language.LoadComponentLanguage( self, Apply );

  // Load strings referred to by code...
  // ----------------------------------------------------------

  Language.LL( Apply, FileOpenTitle, 'FileOpenTitle', 'Open Help Files' );
  Language.LL( Apply, LoadingFileMsg, 'LoadingFileMsg', 'Loading file ' );
  Language.LL( Apply, HelpFileError, 'HelpFileError', 'Could not open ' );
  Language.LL( Apply, LoadingStatusDisplaying, 'LoadingStatusDisplaying', 'Displaying...' );
  Language.LL( Apply, LoadingStatusNotesAndBookmarks, 'LoadingStatusNotesAndBookmarks', 'Loading notes/bookmarks...' );
  Language.LL( Apply, LoadingStatusContents, 'LoadingStatusContents', 'Display contents... ' );
  Language.LL( Apply, LoadingStatusIndex, 'LoadingStatusIndex', 'Display index... ' );
  Language.LL( Apply, LoadingStatusDone, 'LoadingStatusDone', 'Done' );

  Language.LL( Apply, HelpFilesDesc, 'HelpFilesDesc', 'Help Files (*.inf,*.hlp)' );
  Language.LL( Apply, AllFilesDesc, 'AllFilesDesc', 'All Files (*)' );
  Language.LL( Apply, LanguageFilesDesc, 'LanguageFilesDesc', 'NewView Language Files (*.lng)' );

  Language.LL( Apply, SaveLanguageTitle, 'SaveLanguageTitle', 'Save/Update Language File' );
  Language.LL( Apply, OpenLanguageTitle, 'OpenLanguageTitle', 'Open Language File' );
  Language.LL( Apply, SaveLanguageError, 'SaveLanguageError', 'Error saving language file: ' );

  Language.LL( Apply, HelpManagerVersionTitle, 'HelpManagerVersionTitle', 'Help Manager Version' );

  Language.LL( Apply, FindResourceIDTitle, 'FindResourceIDTitle', 'Find Resource ID' );
  Language.LL( Apply, FindResourceIDPrompt, 'FindResourceIDPrompt', 'Enter the resource ID to find' );
  Language.LL( Apply, InvalidResourceIDError, 'InvalidResourceIDError', 'Invalid resource ID entered' );
  Language.LL( Apply, ResourceIDNotFoundError, 'ResourceIDNotFoundError', 'Resource ID not found' );

  Language.LL( Apply, OpenSpecialTitle, 'OpenSpecialTitle', 'Open Special' );
  Language.LL( Apply, OpenSpecialPrompt, 'OpenSpecialPrompt', 'Enter help file name/environment variable name' );

  Language.LL( Apply, PrintTopicTitle, 'PrintTopicTitle', 'Print Topic' );
  Language.LL( Apply, NoPrinterError, 'NoPrinterError', 'You don''t have a printer configured.' );
  Language.LL( Apply, SelectWindowToPrintError, 'SelectWindowToPrintError', 'You must select the window you want to print.' );
  Language.LL( Apply, PrintingError, 'PrintingError', 'Error while printing: ' );

  Language.LL( Apply, TopicInfoTitle, 'TopicInfo.Title', 'Topic Information' );
  Language.LL( Apply, TopicInfoTopicTitle, 'TopicInfo.TopicTitle',   'Title: ' );
  Language.LL( Apply, TopicInfoIndex, 'TopicInfo.Index',             'Index: ' );
  Language.LL( Apply, TopicInfoFile, 'TopicInfo.File',               'File:  ' );
  Language.LL( Apply, TopicInfoResourceIDs, 'TopicInfo.ResourceIDs', 'Resource IDs:' );
  Language.LL( Apply, TopicInfoNoResourceIDs, 'TopicInfo.NoResourceIDs', '  (None)' );

  Language.LL( Apply, ParameterCountLabel, 'ParameterCountLabel', 'Parameter Count: ' );

  Language.LL( Apply, NewViewHelpTitle, 'NewViewHelpTitle', 'NewView Help' );
  Language.LL( Apply, AlreadyNewviewHelp, 'AlreadyNewviewHelp', 'You are already viewing the NewView help file' );
  Language.LL( Apply, NewViewHelpNotFound, 'NewViewHelpNotFound', 'Couldn''t find the NewView helpfile: ' );

  Language.LL( Apply, InvalidLinkErrorTitle, 'InvalidLinkErrorTitle', 'Invalid Link' );
  Language.LL( Apply, InvalidLinkError, 'InvalidLinkError', 'Cannot follow link to nonexistent topic' );
  Language.LL( Apply, InvalidResourceIDLinkErrorA, 'InvalidResourceIDLinkErrorA', 'Could not find linked topic (Resource #' );
  Language.LL( Apply, InvalidResourceIDLinkErrorB, 'InvalidResourceIDLinkErrorB', '). This may be from another file.' );

  Language.LL( Apply, OpenedTopicMsg, 'OpenedTopicMsg', 'Opened topic #' );

  Language.LL( Apply, AddNoteTitle, 'AddNoteTitle', 'Add Note' );
  Language.LL( Apply, AddNoteCursorError, 'AddNoteCursorError', 'Before adding a note, position the cursor where you want the note to be placed.' );
  Language.LL( Apply, NoteWithinNoteError, 'NoteWithinNoteError', 'You can''t add a note within a link or another note' );
  Language.LL( Apply, LoadNotesTitle, 'LoadNotesTitle', 'Load Notes' );
  Language.LL( Apply, LoadNotesError, 'LoadNotesError', 'Error loading notes from ' );
  Language.LL( Apply, SaveNotesTitle, 'SaveNotesTitle', 'Save Notes' );
  Language.LL( Apply, SaveNotesError, 'SaveNotesError', 'Error saving notes to ' );

  Language.LL( Apply, UntitledBookmarkName, 'UntitledBookmarkName', '(Untitled)' );
  Language.LL( Apply, LoadBookmarksTitle, 'LoadBookmarksTitle', 'Load Bookmarks' );
  Language.LL( Apply, LoadBookmarksError, 'LoadBookmarksError', 'Could not load bookmarks: ' );
  Language.LL( Apply, SaveBookmarksTitle, 'SaveBookmarksTitle', 'Save Bookmarks' );
  Language.LL( Apply, SaveBookmarksError, 'SaveBookmarksError', 'Could not save bookmarks: ' );

  Language.LL( Apply, ApplicationErrorTitle, 'ApplicationErrorTitle', 'Application Error - Close?' );
  Language.LL( Apply, ApplicationErrorA, 'ApplicationErrorA', 'This application has crashed. ' );
  Language.LL( Apply, ApplicationErrorB, 'ApplicationErrorB', '(Details logged to ' );
  Language.LL( Apply, ApplicationErrorC, 'ApplicationErrorC', 'Close application? ' );

  Language.LL( Apply, EnvironmentVarErrorTitle, 'EnvironmentVarErrorTitle', 'Environment Variable Warning' );
  Language.LL( Apply, EnvironmentVarError,
      'EnvironmentVarError',
      'NewView found a problem with environment variables. '
      + 'These are used when finding help files. '
      + 'You may have problems launching help.' );
  Language.LL( Apply, EnvironmentVarUndefined, 'EnvironmentVarUndefined', 'Undefined: ' );

  Language.LL( Apply, FindTitle, 'FindTitle', 'Find' );
  Language.LL( Apply, FindSelectWindowError, 'FindSelectWindowError', 'Click in a window first' );
  Language.LL( Apply, FindPrompt, 'FindPrompt', 'Enter the text to find' );
  Language.LL( Apply, TextNotFoundMsg, 'TextNotFoundMsg', 'Text not found' );

  Language.LL( Apply, FilesInfoTitle, 'FilesInfoTitle', 'Open Files Information' );
  Language.LL( Apply, FilesInfoOverallTitle, 'FilesInfoOverallTitle', 'Title: ' );
  Language.LL( Apply, FilesInfoFilename, 'FilesInfoFilename', 'Filename: ' );
  Language.LL( Apply, FilesInfoFileTitle, 'FilesInfoFileTitle', '  Title: ' );
  Language.LL( Apply, FilesInfoTopicCount, 'FilesInfoTopicCount', '  Topic Count: ' );
  Language.LL( Apply, FilesInfoIndexCount, 'FilesInfoIndexCount', '  Index Count: ' );
  Language.LL( Apply, FilesInfoDictionaryCount, 'FilesInfoDictionaryCount', '  Dictionary Count: ' );
  Language.LL( Apply, FilesInfoFileSize, 'FilesInfoFileSize', '  Size: ' );
  Language.LL( Apply, FilesInfoTotalTopicCount, 'FilesInfoTotalTopicCount', 'Total Topic Count: ' );
  Language.LL( Apply, FilesInfoTotalIndexCount, 'FilesInfoTotalIndexCount', 'Total Index Count: ' );
  Language.LL( Apply, FilesInfoTotalFileSize, 'FilesInfoTotalFileSize', 'Total File Size: ' );

  Language.LL( Apply, SearchTitle, 'SearchTitle', 'Search' );
  Language.LL( Apply, SearchSyntaxError, 'SearchSyntaxError', 'Error in search syntax: ' );
  Language.LL( Apply, NoSearchMatchesMsg, 'NoSearchMatchesMsg', 'No matches found for ' );
  Language.LL( Apply, SearchFoundMsgA, 'SearchFoundMsgA', 'Found ' );
  Language.LL( Apply, SearchFoundMsgB, 'SearchFoundMsgB', ' matches for ' );

  Language.LL( Apply, FileSaveTitle, 'FileSaveTitle', 'Save Topic' );
  Language.LL( Apply, FileSaveSelectWindowError, 'FileSaveSelectWindowError', 'Before saving, click in the window you want to save.' );
  Language.LL( Apply, DefaultSaveTopicFilename, 'DefaultSaveTopicFilename', 'topic.txt' );

  Language.LL( Apply, ReplaceFilePromptA, 'ReplaceFilePromptA', 'Replace existing file ' );
  Language.LL( Apply, ReplaceFilePromptB, 'ReplaceFilePromptB', '?' );
  Language.LL( Apply, UnableToSaveError, 'UnableToSaveError', 'Unable to save file: ' );

  Language.LL( Apply, UsageTitle, 'UsageTitle', 'NewView Command Line' );
  Language.LL( Apply, UsageText1, 'UsageText1', 'Usage: ' );
  Language.LL( Apply, UsageText2, 'UsageText2', 'NewView <filename> [<search text>]' );
  Language.LL( Apply, UsageText3, 'UsageText3', ' /g:<text>  Do global search for <text>' );
  Language.LL( Apply, UsageText4, 'UsageText4', ' /pos:l,r,w,h  Set window position' );
  Language.LL( Apply, UsageText5, 'UsageText5', 'See help for details' );

  Language.LL( Apply, GoBackHint, 'GoBackHint', 'Go back to ' );

  Language.LL( Apply, SelectAllTitle, 'SelectAllTitle', 'Select All' );
  Language.LL( Apply, SelectAllWindowError, 'SelectAllWindowError', 'Click in a text window first' );

  Language.LL( Apply, EditNoteMsg, 'EditNoteMsg', 'Click to edit note' );
  Language.LL( Apply, ExternalLinkMsg, 'ExternalLinkMsg', 'Link to another file (not implemented)' );
  Language.LL( Apply, LinkMsg, 'LinkMsg', 'Link to ' );
  Language.LL( Apply, UnknownLinkMsg, 'UnknownLinkMsg', 'Unknown link' );

  Language.LL( Apply, ExternalLinkTitle, 'ExternalLinkTitle', 'File Link' );
  Language.LL( Apply, ExternalLinkError, 'ExternalLinkError', 'Sorry, this is a link to another file, which is not currently implemented in NewView' );

  Language.LL( Apply, MRUMultipleFilesHint, 'MRUMultipleFilesHint', 'files' );
  Language.LL( Apply, HelpProgramTitle, 'HelpProgramTitle', 'Help' );

  // ----------------------------------------------------------
end;

Procedure TMainForm.DebugLoadLanguageMIOnClick (Sender: TObject);
Var
  Dir: string;
  Filename: string;
Begin
  Dir := GetApplicationDir;
  if not DoOpenFileDialog( OpenLanguageTitle,
                           LanguageFilesDesc
                           + '|*.lng|'
                           + AllFilesDesc
                           + '|*',
                           '*.lng',
                           Dir,
                           Filename ) then
    exit;

  LoadLanguage( Filename );
End;

Procedure TMainForm.DebugSaveLanguageFileMIOnClick (Sender: TObject);
Var
  LanguageFile: TLanguageFile;
  Dir: string;
  Filename: string;
Begin
  Dir := GetApplicationDir;
  if not DoSaveFileDialog( SaveLanguageTitle,
                           LanguageFilesDesc
                           + '|*.lng|'
                           + AllFilesDesc
                           + '|*',
                           '*.lng',
                           Dir,
                           Filename ) then
    exit;

  // get rid of mru menu items so they don't clutter the language file
  DestroyListObjects( MRUMenuItems );
  MRUMenuItems.Clear;

  try
    LanguageFile := TLanguageFile.Create( Filename );

    UpdateLanguage( LanguageFile );
  except
    on E: Exception do
    begin
      DoErrorDlg( SaveLanguageTitle,
                  SaveLanguageError + E.Message );
      exit;
    end;
  end;

  LanguageFile.Destroy;

  CreateMRUMenuItems;
End;

Procedure TMainForm.SearchResultsListBoxOnClick (Sender: TObject);
Begin
  DisplaySelectedSearchResultTopic;
End;

Procedure TMainForm.DebugMemoryFormOnClick (Sender: TObject);
Begin
  MemoryForm.Show;
End;

Procedure TMainForm.VSplitBarOnDblClick (Sender: TObject);
Begin
  SetLayout;
End;

Procedure TMainForm.DebugHelpManagerVersionMIOnClick (Sender: TObject);
Begin
  DoMessageDlg( HelpManagerVersionTitle,
                HelpManagerVersion );
End;

Procedure TMainForm.DebugTopicByResourceIDMIOnClick (Sender: TObject);
var
  ResourceIDString: string;
  ResourceID: USHORT;
Begin
  if not DoInputQuery( FindResourceIDTitle,
                       FindResourceIDPrompt,
                       ResourceIDString ) then
    exit;
  try
    ResourceID := StrToInt( ResourceIDString );
  except
    DoErrorDlg( FindResourceIDTitle,
                InvalidResourceIDError );
    exit;
  end;

  if not DisplayTopicByResourceID( ResourceID ) then
    DoErrorDlg( FindResourceIDTitle,
                ResourceIDNotFoundError );
End;

Procedure TMainForm.ViewHighlightSearchWordsMIOnClick (Sender: TObject);
Begin
  ViewHighlightSearchWordsMI.Checked := not ViewHighlightSearchWordsMI.Checked;
  RefreshWindows( Windows );
End;

Procedure TMainForm.DebugHelpMgrMIOnClick (Sender: TObject);
Begin
  Application.HelpContents;
End;

Procedure TMainForm.FileNewWindowMIOnClick (Sender: TObject);
Begin
  Exec( GetApplicationFilename, '' );
End;

Procedure TMainForm.OpenSpecialMIOnClick (Sender: TObject);
var
  Parameter: string;
  Filenames: TStringList;
Begin
  if DoInputQuery( OpenSpecialTitle,
                   OpenSpecialPrompt,
                   Parameter ) then
  begin
    Filenames := TStringList.Create;
    StringToList( Parameter, Filenames, '+' );
    if Filenames.Count > 0 then
    begin
      if OpenFiles( Filenames, '', true ) then
      begin
        Startup_OwnHelpMode := false;
        ClearHelpManager;
      end;
    end;
    Filenames.Destroy;
  end;
End;

Procedure TMainForm.NotesListBoxOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  if KeyCode in [ kbDel, kbBkSp ] then
    if NotesListBox.ItemIndex <> -1 then
      DeleteNote( NotesListBox.ItemIndex );
End;

Procedure TMainForm.ViewPopupMenuOnPopup (Sender: TObject);
var
  Window: THelpWindow;
Begin
  Window := GetActiveWindow;
  if Window = nil then
  begin
    SearchPMI.Enabled := false;
    exit;
  end;
  SearchPMI.Enabled := Window.View.SelectionLength > 0;
End;

Procedure TMainForm.SearchPMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
Begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;

  DoSearch( Window.View.GetSelectionAsString );
end;

Procedure TMainForm.DoSearch( const SearchText: string );
begin
  SearchTextEdit.Text := SearchText;

  TabSet.TabIndex := piSearch;
  SearchTextEdit.Focus;
  Application.ProcessMessages;

  SearchButtonOnClick( self );
End;

function TMainForm.DisplayTopicByResourceID( ID: uint16 ): boolean;
var
  Topic: TTopic;
begin
  Topic := FindTopicByResourceID( ID );
  if Topic = nil then
  begin
    Result := false;
    exit;
  end;

  result := true;

  DisplayTopic( Topic );
end;

Procedure TMainForm.ViewSourceMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
Begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;
  InformationForm.FText := Window.View.Text;
  InformationForm.ShowModal;
End;

Procedure TMainForm.PrintMIOnClick (Sender: TObject);
Begin
  PrintTopics;
End;

Procedure TMainForm.PrintTopics;
var
  Window: THelpWindow;
  PageY: longint;
  RichTextSettings: TRichTextSettings;
  PrinterResolution: longint;
  MarginSize: longint;
  PrintText: TAString;
Begin
  if Printer.Printers.Count = 0 then
  begin
    DoErrorDlg( PrintTopicTitle,
                NoPrinterError );
    exit;
  end;

  Window:= GetActiveWindow;
  if Window = nil then
  begin
    DoErrorDlg( PrintTopicTitle,
                SelectWindowToPrintError );
    exit;
  end;

  if NewViewPrintDialog.ShowModal <> mrOK then
    exit;

  Printer.Title := Window.Topic.Title;

  PrintText := TAString.Create;
  PrintText.AddString( '<b><center>'
                       + Window.Topic.Title
                       + '</b><left>'
                       + #10
                       + #10 );
  PrintText.AddPChar( Window.View.Text );

  Printer.BeginDoc;

  PageY := Printer.PageHeight - 1;

  RichTextSettings := TRichTextSettings.Create( nil );
  RichTextSettings.NormalFont := Settings.NormalFont;
  RichTextSettings.FixedFont := Settings.FixedFont;

  // set half inch margins
  PrinterResolution := Printer.Canvas.HorizontalResolution; // pixels per meter!
  MarginSize := Round( PrinterResolution * 0.0125 ); // 12.5 mm = 0.5 inch
  RichTextSettings.Margins := Rect( MarginSize,
                                    MarginSize,
                                    MarginSize,
                                    MarginSize );

  try
    PrintRichText( PrintText.AsPChar,
                   Window.View.Images,
                   RichTextSettings,
                   PageY );
  except
    on E: EPrinter do
    begin
      DoErrorDlg( PrintTopicTitle,
                  PrintingError + E.Message );
    end;
  end;

  PrintText.Destroy;

  Printer.EndDoc;

End;

// --------------------------------------------------

Procedure TMainForm.DebugShowWordSeparatorsMIOnClick (Sender: TObject);
Begin
  DebugShowWordSeparatorsMI.Checked := not DebugShowWordSeparatorsMI.Checked;
  RefreshWindows( Windows );
End;

Procedure TMainForm.DebugStressTestMIOnClick (Sender: TObject);
Begin
  while ContentsOutline.GotoNextNodeDown do
  begin
    DisplaySelectedContentsTopic;
    Application.ProcessMessages;
  end;
End;

Function TMainForm.FindTopicByResourceID( ID: uint16 ): TTopic;
var
  FileIndex: longint;
  HelpFile: THelpFile;
begin
  for FileIndex:= 0 to Files.Count - 1 do
  begin
    HelpFile := Files[ FileIndex ];

    Result := HelpFile.FindTopicByResourceID( ID );
    if Result <> nil then
      // found
      exit;
  end;

  // not found.
  Result:= nil;
end;

// Find the target topic for the given link
Function TMainForm.FindTopicForLink( Link: THelpLink ): TTopic;
var
  HelpFile: THelpFile;
begin
  HelpFile := Link.HelpFile as THelpFile;
  if Link is TFootnoteHelpLink then
  begin
    Result := HelpFile.Topics[ TFootnoteHelpLink( Link ).TopicIndex ];
  end
  else if Link is TInternalHelpLink then
  begin
    Result := HelpFile.Topics[ TInternalHelpLink( Link ).TopicIndex ];
  end
  else if Link is THelpLinkByResourceID then
  begin
    Result := FindTopicByResourceID( THelpLinkByResourceID( Link ).ResourceID );
  end
end;

Procedure TMainForm.TopicPropertiesPMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
  Topic: TTopic;
  HelpFile: THelpFile;
  ResourceIDs: TList;
  i: longint;
Begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;
  Topic := Window.Topic;
  HelpFile := Topic.HelpFile as THelpFile;

  ResourceIDs := TList.Create;
  HelpFile.FindResourceIDsForTopic( Topic,
                                    ResourceIDs );

  with InformationForm.InformationMemo do
  begin
    Lines.Clear;
    Lines.Add( TopicInfoTitle );
    Lines.Add( TopicInfoTopicTitle );
    Lines.Add( TopicInfoIndex );
    Lines.Add( TopicInfoFile );
    Lines.Add( TopicInfoResourceIDs );
    for i := 0 to ResourceIDs.Count - 1 do
      Lines.Add( '  ' + IntToStr( longint( ResourceIDs[ i ] ) ) );
    if ResourceIDs.Count = 0 then
      Lines.Add( TopicInfoNoResourceIDs );
  end;
  ResourceIDs.Destroy;

  InformationForm.ShowModal;
End;
Procedure TMainForm.NavigateForwardMIOnClick (Sender: TObject);
Begin
  NavigateForward;
End;

Procedure TMainForm.FocusFirstHelpWindow;
begin
  if Windows.Count > 0 then
    THelpWindow( Windows[ 0 ] ).View.Focus;
end;

Procedure TMainForm.IndexListBoxOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  case KeyCode of
    kbTab:
      FocusFirstHelpWindow;
    kb_VK + VK_NEWLINE:
      DisplaySelectedIndexTopic;
  end;
End;

Procedure TMainForm.SearchResultsListBoxOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  case KeyCode of
    kbTab:
      FocusFirstHelpWindow;
    kb_VK + VK_NEWLINE:
      DisplaySelectedSearchResultTopic;
  end;
End;

Procedure TMainForm.ContentsOutlineOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  case KeyCode of
    kbTab:
      FocusFirstHelpWindow;
    kb_VK + VK_NEWLINE:
      DisplaySelectedContentsTopic;
  end;
End;

Procedure TMainForm.ToolsOptionsMIOnClick (Sender: TObject);
Begin
  DoOptions;
End;

Procedure TMainForm.EditGlobalSearchMIOnClick (Sender: TObject);
Begin
  DoGlobalSearch( '' );
End;

Procedure TMainForm.ViewExpandAllMIOnClick (Sender: TObject);
Begin
  ContentsOutline.ExpandAll;
End;

Procedure TMainForm.DebugShowParamsMIOnClick (Sender: TObject);
var
  i: integer;
Begin
  with InformationForm.InformationMemo do
  begin
    Lines.Clear;
    Lines.Add( ParameterCountLabel
               + IntToStr( ParamCount ) );
    for i := 1 to ParamCount do
      Lines.Add( ' '
                 + IntToStr( i )
                 + ' ['
                 + ParamStr( i )
                 + ']' );
  end;

  InformationForm.ShowModal;
End;

Procedure TMainForm.EditBookmarksMIOnClick (Sender: TObject);
Begin
  BookmarksForm.BookmarkList := Bookmarks;
  BookmarksForm.OpenBookmarkCallback := NavigateToBookmark;
  BookmarksForm.BookmarksChangedCallback := OnBookmarksChanged;
  BookmarksForm.Show;

  // Since we are showing a nonmodal dialog, set the PM owner window
  // so that the bookmarks form remains on top.
  WinSetOwner( BookmarksForm.Frame.Handle,
               Frame.Handle );
End;

Procedure TMainForm.CopyPMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
Begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;
  Window.View.CopySelectionToClipboard;
End;

Procedure TMainForm.SelectAllPMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
Begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;
  Window.View.SelectAll;
End;

Procedure TMainForm.AddNoteButtonOnClick (Sender: TObject);
Begin
  AddNote;
End;

Procedure TMainForm.EnableSearchButton;
var
  CanSearch: boolean;
begin
  CanSearch := false;
  if Files.Count > 0 then
    if trim( SearchTextEdit.Text ) > '' then
      CanSearch := true;
  SearchButton.Enabled := CanSearch;
end;

Procedure TMainForm.SearchTextEditOnChange (Sender: TObject);
Begin
  EnableSearchButton;
End;

Procedure TMainForm.OnHint( Sender: TObject );
begin
  SetStatus( Application.Hint );
end;

Procedure TMainForm.DisplaySelectedIndexTopic;
var
  Topic: TTopic;
Begin
  if IndexListBox.ItemIndex = -1 then
    exit;
  Topic := DisplayedIndex.Objects[ IndexListBox.ItemIndex ] as TTopic;
  DisplayTopic( Topic );
End;

Procedure TMainForm.IndexListBoxOnClick (Sender: TObject);
Begin
  DisplaySelectedIndexTopic;
End;

Procedure TMainForm.ViewCollapseAllMIOnClick (Sender: TObject);
Begin
  ContentsOutline.CollapseAll;
  DisplaySelectedContentsTopic;
End;

Procedure TMainForm.NotesListBoxOnDblClick (Sender: TObject);
Begin
  GotoCurrentNote;
End;

function TMainForm.GetOwnHelpFileName: string;
begin
  Result := FindHelpFile( 'NewView' );
  if Result = '' then
    Result := AddSlash( GetApplicationDir )
              + 'NewView.inf';
end;

Procedure TMainForm.HelpMIOnClick (Sender: TObject);
Begin
  if Startup_OwnHelpMode then
  begin
    DoMessageDlg( NewViewHelpTitle,
                  AlreadyNewviewHelp );
    exit;
  end;

  if not FileExists( GetOwnHelpFileName ) then
  begin
    DoErrorDlg( NewViewHelpTitle,
                NewViewHelpNotFound
                + EndLine
                + EndLine
                + GetOwnHelpFileName );
    exit;
  end;

  Exec( GetApplicationFilename,
        '/nvhelp' ); // tell the new instance it's doing help for us.
End;

Procedure TMainForm.NotebookOnPageChanged (Sender: TObject);
Begin
  if Notebook.PageIndex = piSearch then
  begin
    SearchButton.Focus;
    SearchTextEdit.Focus;
  end;
End;

Procedure TMainForm.ViewRefreshMIOnClick (Sender: TObject);
Begin
  RefreshWindows( Windows );
End;

Procedure TMainForm.ViewNotesMIOnClick (Sender: TObject);
Begin
  TabSet.TabIndex:= piNotes;
  NotesListBox.Focus;
End;

Procedure TMainForm.ViewSearchMIOnClick (Sender: TObject);
Begin
  TabSet.TabIndex:= piSearch;
  SearchTextEdit.Focus;
End;

Procedure TMainForm.ViewIndexMIOnClick (Sender: TObject);
Begin
  DisplayIndex;
End;

Procedure TMainForm.DisplayIndex;
Begin
  TabSet.TabIndex:= piIndex;
  IndexSearchEdit.Focus;
End;

Procedure TMainForm.ViewContentsMIOnClick (Sender: TObject);
Begin
  DisplayContents;
End;

Procedure TMainForm.DisplayContents;
Begin
  Tabset.TabIndex:= piContents;
  ContentsOutline.Focus;
End;

function TMainForm.OpenWindow( Topic: TTopic;
                               Group: longint;
                               Parent: THelpWindow;
                               Rect: THelpWindowRect;
                               FollowAutoLinks: boolean ): THelpWindow;
var
  Window: THelpWindow;
  DisplayTopicRequired: boolean;
begin
  Window := nil;

  if ( Group <> DefaultGroupIndex ) and ( Parent = nil ) then
  begin
    // Normal window (not a split window) and a specific group is desired.
    // So see if we can find one with that group number
    Window := FindWindowFromGroup( Group, Windows );

  end
  else
  begin
    // only reuse window if it has the same topic.
    Window := FindWindowFromTopic( Topic, Windows );
  end;

  if Window = nil then
  begin
    // not found, or want a new one
    Window := THelpWindow.Create( Parent = nil );

    // add to parent
    if Parent = nil then
    begin
      Window.Parent := DisplayPanel;
      Windows.Add( Window );
    end
    else
    begin
      Window.Parent := Parent.View;
      Parent.ChildWindows.Add( Window );
    end;

    Window.ParentHelpWindow := Parent;
  end
  else
  begin
    // reusing an existing window. Don't change parent
  end;

  Window.Group := Group;

  Window.View.PopupMenu := ViewPopupMenu;

  Window.View.Images := Window.Images;

  Window.View.OnClickLink := OnClickLink;
  Window.View.OnOverLink := OnOverLink;
  Window.View.OnNotOverLink := OnNotOverLink;

  Window.OnClose := OnWindowClose;
  Window.OnCloseQuery := OnWindowAboutToClose;

  // Use the contents rect by default...
  Topic.GetContentsWindowRect( Window.Rect );
  if Rect <> nil then
  begin
    // the rect is being overridden, so use it instead
    if Rect.Left <> -1 then
      Window.Rect.Left := Rect.Left;
    if Rect.Bottom <> -1 then
      Window.Rect.Bottom := Rect.Bottom;
    if Rect.Width <> -1 then
      Window.Rect.Width := Rect.Width;
    if Rect.Height <> -1 then
      Window.Rect.Height := Rect.Height;
  end;

  if Window.ChildWindows.Count > 0 then
  begin
    // close existing child windows
    DestroyListObjects( Window.ChildWindows );
    Window.ChildWindows.Clear;
    Window.View.Show; // show the view again
  end;

  DisplayTopicRequired := Window.Topic <> Topic;
  Window.Topic := Topic; // set this now so that SetLayout can log meaninful stuff

  // Adjust the window size to it's specified Rect
  // Must do this before displaying child windows (DisplayTopicInWindow,
  // split window autolinks),
  // otherwise they will not size themselves correctly
  Window.SetLayout;

  if DisplayTopicRequired then
  begin
    DisplayTopicInWindow( Window, FollowAutoLinks, false );
  end;

  // Bring this window to the front
  Window.BringToFront;

  Result := Window;
end;

// Find an existing help window containing the given richtext view control
Function TMainForm.FindWindowFromView( View: TRichTextView;
                                       WindowList: TList ): THelpWindow;
var
  WindowIndex: longint;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Result:= WindowList[ WindowIndex ];
    if Result.View = View then
      exit;
    Result:= FindWindowFromView( View, Result.ChildWindows );
    if Result <> nil then
      exit;
  end;
  Result:= nil;
end;

// Find an existing help window with the given group number (if any)
Function TMainForm.FindWindowFromGroup( Group: longint; WindowList: TList ): THelpWindow;
var
  WindowIndex: longint;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Result:= WindowList[ WindowIndex ];
    if Result.Group = Group then
      exit;
    Result:= FindWindowFromGroup( Group, Result.ChildWindows );
    if Result <> nil then
      exit;
  end;
  Result:= nil;
end;

// Find an existing help window alreadying displaying the given topic
Function TMainForm.FindWindowFromTopic( Topic: TTopic; WindowList: TList ): THelpWindow;
var
  WindowIndex: longint;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Result:= WindowList[ WindowIndex ];
    if Result.Topic = Topic then
      exit;
    Result:= FindWindowFromTopic( Topic, Result.ChildWindows );
    if Result <> nil then
      exit;
  end;
  Result:= nil;
end;

// Redisplay topics in all windows
Procedure TMainForm.RefreshWindows( WindowList: TList );
var
  WindowIndex: longint;
  Window: THelpWindow;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Window:= WindowList[ WindowIndex ];
    DisplayTopicInWindow( Window,
                          false, // don't follow links!
                          true ); // keep position
    RefreshWindows( Window.ChildWindows );
  end;
end;

// We are following a link, specified as param1 in msg
Procedure TMainForm.WMFollowLink( Var Msg: TMessage );
var
  NewTopic: TTopic;
  Link: THelpLink;
  SourceWindow: THelpWindow;
begin
  Link:= THelpLink( Msg.Param1 );
  SourceWindow:= THelpWindow( Msg.Param2 );
  NewTopic:= FindTopicForLink( Link );

  // remove the link target info from status
  SetStatus( '' );

  if NewTopic = nil then
  begin
    // Linked topic not found, this is an error... which kind?

    if Link is THelpLinkByResourceID then
    begin
      // may happen if e.g. PM4.INF is loaded by itself,
      // and a link references a resource ID from e.g PM2.INF
      DoErrorDlg( InvalidLinkErrorTitle,
                  InvalidResourceIDLinkErrorA
                  + IntToStr( THelpLinkByResourceID( Link ).ResourceID )
                  + InvalidResourceIDLinkErrorB );
    end
    else
    begin
      // should never happen, given valid help files
      DoErrorDlg( InvalidLinkErrorTitle,
                  InvalidLinkError );
    end;

    exit;
  end;

  UpdateCurrentNavigatePoint;

  FollowLink( Link, SourceWindow );

  if NewTopic.ShowInContents then
  begin
    Navigating:= true;
    if ContentsOutline.SelectedNode = nil then
      ContentsOutline.SetSelectedObject( NewTopic )
    else if ContentsOutline.SelectedNode.Data <> NewTopic then
      ContentsOutline.SetSelectedObject( NewTopic );
    Navigating:= false;
  end;
  SaveNavigatePoint;
  EnableControls;
  ShowWindows;

end;

// Follow the given link from the given window.
// ie. open the topic or footnote it points to
Procedure TMainForm.FollowLink( Link: THelpLink;
                                SourceWindow: THelpWindow );
var
  LinkedTopic: TTopic;
  ParentWindow: THelpWindow;
  WindowedLink: TWindowedHelpLink;
begin
  LinkedTopic := FindTopicForLink( Link );

  if LinkedTopic = nil then
  begin
    exit;
  end;

  if Link is TFootnoteHelpLink then
  begin
    OpenWindow( LinkedTopic,
                DefaultGroupIndex,
                nil, // no parent
                FootnoteRect,
                true );
    exit;
  end;

  WindowedLink := Link as TWindowedHelpLink;

  ParentWindow:= nil;
  if WindowedLink.Split then
    ParentWindow:= SourceWindow;

  if WindowedLink.ViewPort then
    // link always wants a new window
    OpenWindow( LinkedTopic,
                DefaultGroupIndex,
                ParentWindow,
                WindowedLink.Rect,
                true )

  else if WindowedLink.GroupIndex <> DefaultGroupIndex then
    // link overrides group index
    OpenWindow( LinkedTopic,
                WindowedLink.GroupIndex,
                ParentWindow,
                WindowedLink.Rect,
                true )
  else
    // no special case
    OpenWindow( LinkedTopic,
                LinkedTopic.ContentsGroupIndex,
                ParentWindow,
                WindowedLink.Rect,
                true );
end;

// Decode and display the topic for the given window.
Procedure TMainForm.DisplayTopicInWindow( Window: THelpWindow;
                                          FollowAutoLinks: boolean;
                                          KeepPosition: boolean );
var
  ImageIndices: TList;
  LinkIndex: longint;
  Link: THelpLink;
  WindowedHelpLink: TWindowedHelpLink;
  InternalHelpLink: TInternalHelpLink;
  HelpFile: THelpFile;
  LinkedTopic: TTopic;
  SourceTopic: TTopic;
  HighlightWords: UInt32ArrayPointer;
  TopCharIndex: longint;
  i: longint;
Begin
  ProfileEvent( 'DisplayTopicInWindow' );

  if TopicDecodeRecusionLevel = 0 then
    Screen.Cursor := crHourglass;
  inc( TopicDecodeRecusionLevel );

  TopCharIndex := Window.View.TopCharIndex;

  Window.View.Hide;
  Window.View.Clear;
  ImageIndices:= TList.Create;

  HelpFile := TopicFile( Window.Topic );

  if ViewHighlightSearchWordsMI.Checked then
    HighlightWords := HelpFile.HighlightWords
  else
    HighlightWords := nil;
  Window.Topic.GetText( HighlightWords,
                        DebugShowCodesMI.Checked,
                        DebugShowWordSeparatorsMI.Checked,
                        TopicText,
                        ImageIndices );

  HelpFile.GetImages( ImageIndices, Window.Images );

  InsertNotesIntoTopicText( Window.Topic, TopicText );

  Window.View.AddText( TopicText.AsPChar );

  if KeepPosition then
    Window.View.TopCharIndex := TopCharIndex;
  Window.View.Show;

  Window.Caption := Window.Topic.Title;
  Window.BringToFront;
  Window.View.Focus;

  // Take a copy of the topic, because the window in question could be changing
  // due to recursion!
  SourceTopic := Window.Topic;

  if FollowAutoLinks then
  begin
    i := 0;
    for LinkIndex:= 0 to SourceTopic.Links.Count - 1 do
    begin
      Link:= SourceTopic.Links[ LinkIndex ];
      if Link is TWindowedHelpLink then
      begin
        WindowedHelpLink := Link as TWindowedHelpLink;
        if WindowedHelpLink.Automatic then
        begin
          if Link is TInternalHelpLink then
          begin
            InternalHelpLink := TInternalHelpLink( Link );
            LinkedTopic :=
              THelpFile( InternalHelpLink.HelpFile ).
                Topics[ InternalHelpLink.TopicIndex ];
            if LinkedTopic.Index = SourceTopic.Index then
              // what the - ? The link wants to open the same topic again
              continue;
          end;

          FollowLink( Link, Window );
          Window := Window;

          inc( i );
          // Note 1: it is possible to crash here if
          // e.g. window1 auto-opens window2 which auto-opens window1 with a different topic..
          // I can't think of a nice, easy way to detect this
          //
          // Note 2: If there is no group number specified
          // (ie. group = default = 0 ) then behaves as if viewport is set:
          // always opens another window.
        end;
      end;
    end;
  end;

  dec( TopicDecodeRecusionLevel );

  if TopicDecodeRecusionLevel = 0 then
    Screen.Cursor := crDefault;

  ImageIndices.Destroy;
End;

Procedure TMainForm.MainFormOnCloseQuery (Sender: TObject;
  Var CanClose: Boolean);
Begin
  ProfileEvent( '-------- Shutdown ----------' );

  CloseFile;
End;

procedure TMainForm.DisplayTopic( Topic: TTopic );
begin
  if Navigating then
    exit;

  UpdateCurrentNavigatePoint;

  CloseWindows;

  CurrentTopic:= Topic;

  OpenWindow( CurrentTopic,
              CurrentTopic.ContentsGroupIndex,
              nil,
              nil,
              true );
  SetStatus( OpenedTopicMsg
             + IntToStr( Topic.Index ) );

  Navigating:= true;

  if ContentsOutline.SelectedNode = nil then
    ContentsOutline.SetSelectedObject( Topic )
  else if ContentsOutline.SelectedNode.Data <> Topic then
    ContentsOutline.SetSelectedObject( Topic );

  SaveNavigatePoint;

  Navigating:= false;
  // find in index...
  // find in search results...

  EnableControls;

  ShowWindows;

//  if Windows.Count > 0 then
//    THelpWindow( Windows[ 0 ] ).View.Focus;

end;

// Make the all current help windows visible
Procedure TMainForm.ShowWindows;
begin
  ShowWindowList( Windows );
end;

// Make the specified windows visible
Procedure TMainForm.ShowWindowList( WindowList: TList );
var
  i: integer;
  Window: THelpWindow;
begin
  for i := 0 to WindowList.Count - 1 do
  begin
    Window:= WindowList[ i ];
    Window.Show;
    ShowWindowList( Window.ChildWindows );
  end;
end;

// Global search -----------------------------------------------------------

Procedure TMainForm.GlobalSearchMIOnClick (Sender: TObject);
begin
  DoGlobalSearch( '' );
end;

Procedure TMainForm.DoGlobalSearch( const SearchText: string );
Begin
  EnsureGlobalSearchFormLoaded;

  GlobalSearchForm.ViewTopicCallback:= OnViewGlobalSearchTopic;
  GlobalSearchForm.Show;
  WinSetOwner( GlobalSearchForm.Frame.Handle, Frame.Handle );

  if SearchText <> '' then
  begin
    GlobalSearchForm.SearchTextEdit.Text := SearchText;
    GlobalSearchForm.DoSearch;
  end;
End;

Procedure TMainForm.OnViewGlobalSearchTopic( FileName: string;
                                             TopicIndex: longint );
var
  HelpFile: THelpFile;
begin
  HelpFile:= FindOpenHelpFile( FileName );

  if HelpFile = nil then
  begin
    if OpenFile( Filename, '', false ) then
    begin
      HelpFile := Files[ 0 ];
      Startup_OwnHelpMode := false;
      ClearHelpManager;
    end;
  end;

  if HelpFile <> nil then
    if TopicIndex <> -1 then
      DisplayTopic( HelpFile.Topics[ TopicIndex ] );

end;

// Notes -----------------------------------------------------------

Procedure TMainForm.GotoNoteButtonOnClick (Sender: TObject);
begin
  GotoCurrentNote;
end;

Procedure TMainForm.EditNoteButtonOnClick (Sender: TObject);
Begin
  if NotesListBox.ItemIndex = -1 then
    exit;
  EditNote( NotesListBox.ItemIndex );
End;

Procedure TMainForm.NotesListBoxOnItemFocus (Sender: TObject; Index: LongInt);
var
  Note: THelpNote;
Begin
  Note:= NotesListBox.Items.Objects[ NotesListBox.ItemIndex ] as THelpNote;
  EnableNotesControls;
End;

Procedure TMainForm.DeleteNoteButtonOnClick (Sender: TObject);
Begin
  if NotesListBox.ItemIndex = -1 then
    exit;
  DeleteNote( NotesListBox.ItemIndex );
End;

Procedure TMainForm.AddBookmarkMIOnClick (Sender: TObject);
Begin
  AddBookmark;
End;

Procedure TMainForm.AddNoteMIOnClick (Sender: TObject);
Begin
  AddNote;
End;

// -----------------------------------------------------------

Procedure TMainForm.FileCloseMIOnClick (Sender: TObject);
Begin
  CloseFile;
End;

Procedure TMainForm.SetStatus( Text: String );
begin
  StatusPanel.Caption:= Text;
  StatusPanel.Refresh;
end;

Procedure TMainForm.ResetProgress;
begin
  ProgressBar.Position:= 0;
  ProgressBar.Hide;
end;

Procedure TMainForm.CoolBarOnSectionResize (HeaderControl: THeaderControl;
  section: THeaderSection);
Begin

End;

Procedure TMainForm.CoolBarOnSectionClick (HeaderControl: THeaderControl;
  section: THeaderSection);
Begin
  case Section.Index of
    ciOpen:
      FileOpen;
    ciBack:
      NavigateBack;
    ciForward:
      NavigateForward;
    ciPrint:
      PrintTopics;
    ciAddNote:
      AddNote;
    ciAddBookmark:
      AddBookmark;
    ciPrevious:
      NavigatePreviousInContents;
    ciNext:
      NavigateNextInContents;
  end;

End;

// ---------------- Notes ----------------------

function TMainForm.FindOriginalNoteCharIndex( NoteCharIndex: longword;
                                              Topic: TTopic ): longword;
var
  NoteIndex: longint;
  Note: THelpNote;
begin
  Result := NoteCharIndex;
  for NoteIndex := 0 to Notes.Count - 1 do
  begin
    Note := Notes[ NoteIndex ];
    if Note.Topic = Topic then
      if Note.InsertPoint < NoteCharIndex then
        dec( Result, Note.InsertText.Length );
  end;
end;

function TMainForm.FindActualNoteCharIndex( NoteCharIndex: longword;
                                            MaxNoteIndex: longword;
                                            Topic: TTopic ): longword;
var
  NoteIndex: longint;
  Note: THelpNote;
begin
  NoteIndex:= 0;
  Result:= NoteCharIndex;
  for NoteIndex:= 0 to MaxNoteIndex - 1 do
  begin
    Note:= Notes[ NoteIndex ];
    if Note.Topic = Topic then
      if Note.InsertPoint < NoteCharIndex then
        inc( Result, Note.InsertText.Length );
  end;
end;

procedure TMainForm.RefreshNoteInsertInfo( NoteIndex: longword );
var
  Note: THelpNote;
begin
  Note:= Notes[ NoteIndex ];

  if Note.Topic = nil then
    exit;
  with Note do
  begin
    InsertText.AssignString( '<color #'
                             + IntToHex( Settings.Colors[ NotesTextColorIndex ], 6 )
                             + '><link note'
                             + IntToStr( NoteIndex )
                             + '>' );
    InsertText.Add( Text );
    InsertText.AddString( '</color></link>' );
  end;
end;

procedure TMainForm.ClearNotes;
begin
  DestroyListObjects( Notes );
  Notes.Clear;
end;

procedure TMainForm.AddNote;
var
  Note: THelpNote;
  Window: THelpWindow;
begin
  Window := GetActiveWindow;
  if Window = nil then
  begin
    DoErrorDlg( AddNoteTitle,
                AddNoteCursorError );
    exit;
  end;

  if Window.View.CursorIndex = -1 then
  begin
    DoErrorDlg( AddNoteTitle,
                AddNoteCursorError );
    exit;
  end;

  // check that the note position isn't
  // within a note already
  if Window.View.LinkFromIndex( Window.View.CursorIndex ) <> '' then
  begin
    DoErrorDlg( AddNoteTitle,
                NoteWithinNoteError );
    exit;
  end;

  // ask for note text
  NoteForm.DeleteNoteButton.Enabled := false; // can't delete it while creating!
  NoteForm.Text.Clear;
  if NoteForm.ShowModal <> mrOK then
    exit;

  // store note data
  Note := THelpNote.Create;
  Note.Text.Assign( NoteForm.Text );

  // compensate for existing notes
  if Window.View.CursorIndex <> -1 then
    Note.InsertPoint := FindOriginalNoteCharIndex( Window.View.CursorIndex, Window.Topic )
  else
    Note.InsertPoint := 0;

  Note.Topic := Window.Topic;

  Notes.Add( Note );

  // redisplay topic
  DisplayTopicInWindow( Window,
                        false, // don't follow links!
                        true ); // keep position
  Window.View.SelectionStart := FindActualNoteCharIndex( Note.InsertPoint,
                                                         Notes.Count - 1,
                                                         Window.Topic );
  UpdateNotesDisplay;

  SaveNotes;

end;

procedure TMainForm.DeleteNote( NoteIndex: longint );
var
  Note: THelpNote;
begin
  Note := Notes[ NoteIndex ];
  Notes.Delete( NoteIndex );

  RefreshWindows( Windows );

  Note.Destroy;

  UpdateNotesDisplay;

  SaveNotes;
end;

Procedure TMainForm.EditNote( NoteIndex: longint );
var
  Note: THelpNote;
begin
  Note:= Notes[ NoteIndex ];

  NoteForm.Text.Assign( Note.Text );

  NoteForm.DeleteNoteButton.Enabled:= true;

  if NoteForm.ShowModal = mrCancel then
    exit;

  if NoteForm.ModalResult = cmDiscard then
  begin
    DeleteNote( NoteIndex );
    exit;
  end;

  Note.Text.Assign( NoteForm.Text );

  RefreshWindows( Windows );

  UpdateNotesDisplay;

  SaveNotes;
end;

Procedure TMainForm.GotoCurrentNote;
var
  Note: THelpNote;
Begin
  if NotesListBox.ItemIndex = -1 then
    exit;
  Note:= NotesListBox.Items.Objects[ NotesListBox.ItemIndex ] as THelpNote;
  DisplayTopic( Note.Topic );
End;

// ---------------- Bookmarks ----------------------

procedure TMainForm.OnBookmarksChanged( Sender: TObject );
begin
  BuildBookmarksMenu;
  SaveBookmarks;
end;

procedure TMainForm.AddBookmark;
var
  Bookmark: TBookmark;
begin
  Bookmark:= TBookmark.Create;
  SaveWindows( Windows, Bookmark.Windows, nil );

  if ContentsOutline.SelectedNode <> nil then
  begin
    Bookmark.ContentsTopic:=
      ContentsOutline.SelectedNode.Data as TTopic;
    Bookmark.Name := Bookmark.ContentsTopic.Title;
  end
  else
  begin
    Bookmark.ContentsTopic:= nil;
    Bookmark.Name := UntitledBookmarkName;
  end;

  Bookmarks.Add( Bookmark );
  OnBookmarksChanged( self );
end;

Procedure TMainForm.BookmarksMenuItemClick( Sender: TObject );
var
  Tag: longint;
  MenuItem: TMenuItem;
  Bookmark: TBookmark;
begin
  MenuItem:= Sender as TMenuItem;
  Tag:= MenuItem.Tag;
  Bookmark := Bookmarks[ Tag ];

  NavigateToBookmark( Bookmark );
end;

Procedure TMainForm.NavigateToBookmark( Bookmark: TBookmark );
Begin
  UpdateCurrentNavigatePoint;
  NavigateToPoint( Bookmark );
  SaveNavigatePoint;
End;

Procedure TMainForm.BuildBookmarksMenu;
var
  i: integer;
  Bookmark: TBookmark;
  MenuItem: TMenuItem;
begin
  DestroyListObjects( BookmarksMenuItems );
  BookmarksMenuItems.Clear;

  if Bookmarks.Count > 0 then
  begin
    MenuItem:= TMenuItem.Create( self );
    MenuItem.Caption:= '-';
    BookmarksMenu.Add( MenuItem );
    BookmarksMenuItems.Add( MenuItem );
  end;

  for i:= 0 to Bookmarks.Count -1 do
  begin
    Bookmark := Bookmarks[ i ];
    MenuItem:= TMenuItem.Create( self );

    MenuItem.Caption:= Bookmark.Name;
    MenuItem.OnClick:= BookmarksMenuItemClick;
    MenuItem.Tag:= i;
    BookmarksMenu.Add( MenuItem );
    BookmarksMenuItems.Add( MenuItem );
  end;
end;

Procedure TMainForm.UpdateBookmarksForm;
begin
  if Assigned( BookmarksForm ) then
    if BookmarksForm.Visible then
      BookmarksForm.RefreshList;
end;

Procedure TMainForm.ClearBookmarks;
begin
  ClearListAndObjects( Bookmarks );
  BuildBookmarksMenu;
  if Assigned( BookmarksForm ) then
    BookmarksForm.Hide;

end;

procedure TMainForm.LoadBookmarks( HelpFile: THelpFile );
var
  Bookmark: TBookmark;
  BookmarksFile: TextFile;
  BookmarksFileName: string;
  S: string;
begin
  BookmarksFileName:= ChangeFileExt( HelpFile.FileName, '.bmk' );

  if not FileExists( BookmarksFileName ) then
    exit;

  AssignFile( BookmarksFile, BookmarksFileName );
  try
    Reset( BookmarksFile );
    try
      while not Eof( BookmarksFile ) do
      begin
        ReadLn( BookmarksFile, s );
        if trim( Uppercase( s ) ) = '[BOOKMARK]' then
        begin
          Bookmark:= TBookmark.Load( BookmarksFile, HelpFile );
          Bookmarks.Add( Bookmark );
        end;
      end;
    finally
      System.Close( BookmarksFile );
    end;
  except
    on e: exception do
      DoErrorDlg( LoadBookmarksTitle,
                  LoadBookmarksError
                  + e.message );
  end;
end;

procedure TMainForm.SaveBookmarks;
var
  FileIndex: integer;
  HelpFile: THelpFile;
begin
  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    SaveBookmarksForFile( HelpFile );
  end;
end;

procedure TMainForm.SaveNotes;
var
  FileIndex: integer;
  HelpFile: THelpFile;
begin
  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    SaveNotesForFile( HelpFile );
  end;
end;

procedure TMainForm.SaveBookmarksForFile( HelpFile: THelpFile );
var
  i: integer;
  Bookmark: TBookmark;
  BookmarksFile: TextFile;
  BookmarksFileName: string;
  BookmarkCount: integer;
begin
  BookmarksFileName:= ChangeFileExt( HelpFile.FileName, '.bmk' );

  BookmarkCount:= 0;
  for i:= 0 to Bookmarks.Count - 1 do
  begin
    Bookmark := Bookmarks[ i ];

    if Bookmark.ContentsTopic.HelpFile = HelpFile then
      inc( BookmarkCount );
  end;

  if BookmarkCount = 0 then
  begin
    if FileExists( BookmarksFileName ) then
      DeleteFile( BookmarksFileName );
    exit;
  end;

  AssignFile( BookmarksFile, BookmarksFileName );
  try
    Rewrite( BookmarksFile );
    try
      for i := 0 to Bookmarks.Count - 1 do
      begin
        Bookmark:= Bookmarks[ i ];
        if Bookmark.ContentsTopic.HelpFile = HelpFile then
        begin
          WriteLn( BookmarksFile, '[Bookmark]' );
          Bookmark.Save( BookmarksFile );
        end;
      end;
    finally
      System.Close( BookmarksFile );
    end;
  except
    on e: exception do
      DoErrorDlg( SaveBookmarksTitle,
                  SaveBookmarksError
                  + e.message );
  end;
end;

// Reads colors back from controls, in case they have changed by
// drag and drop
Procedure TMainForm.GetColors;
begin
  with Settings do
  begin
    Colors[ ContentsBackgroundColorIndex ] := ContentsOutline.Color;
    Colors[ ContentsTextColorIndex ] := ContentsOutline.PenColor;
    Colors[ IndexBackgroundColorIndex ] := IndexListBox.Color;
    Colors[ IndexTextColorIndex ] := IndexListBox.PenColor;
    Colors[ SearchBackgroundColorIndex ] := SearchResultsListBox.Color;
    Colors[ SearchTextColorIndex ] := SearchResultsListBox.PenColor;
    Colors[ NotesListBackgroundColorIndex ] := NotesListBox.Color;
    Colors[ NotesListTextColorIndex ] := NotesListBox.PenColor;
  end;
end;

Procedure TMainForm.MainFormOnDestroy (Sender: TObject);
Begin
  ProfileEvent( 'MainFormOnDestroy' );

  ProfileEvent( 'Write window position' );

  if not Startup_OwnHelpMode then
  begin
    WriteWindowPos( self );

    ProfileEvent( 'Save settings' );

    GetColors;
    SaveSettings;
  end;
  // else- don't save position/size if doing own help

  TopicText.Destroy;

  if Assigned( MessageMemory ) then
  begin
    ProfileEvent( 'Close shared memory' );
    MessageMemory.Destroy;
  end;

  ProfileEvent( 'Destroy MRU menu items' );
  MRUMenuItems.Destroy;

  ProfileEvent( 'Destroy navigate to menu items' );
  NavigateToMenuItems.Destroy;

  ProfileEvent( 'Destroy pagehistory' );
  PageHistory.Destroy;

  ProfileEvent( 'Clear/destroy notes' );
  ClearNotes;
  Notes.Destroy;

  ProfileEvent( 'Clear/destroy bookmarks' );
  ClearBookmarks;
  Bookmarks.Destroy;

  ProfileEvent( 'Destroy bookmark menu items' );
  BookmarksMenuItems.Destroy;

  ProfileEvent( 'Destroy files/index/windows' );
  Files.Destroy;
  DisplayedIndex.Destroy;
  Windows.Destroy;

  DestroyListObjects( Settings.MRUList );
  Settings.MRUList.Destroy;

  Startup_FilenamesParam.Destroy;

  ProfileEvent( 'MainFormOnDestroy done' );

  if g_CurrentLanguageFile <> nil then
    g_CurrentLanguageFile.Destroy;

End;

Procedure TMainForm.MainFormOnSetupShow (Sender: TObject);
Begin
  ProfileEvent( 'OnSetupShow' );
  TabSet.TabIndex := 0;
  Notebook.PageIndex := 0;
End;

Function GetSystemInfoItem( Item: ULONG ): ULONG;
begin
  DosQuerySysInfo( Item,
                   Item,
                   Result,
                   sizeof( Result ) );
end;

Procedure TMainForm.OnException( Sender: TObject;
                                 E: Exception );
var
  TheText : string;
  F: TextFile;
  i: integer;
  HelpFile: THelpFile;
  OSMajorVersion: ULONG;
  OSMinorVersion: ULONG;
  OSRevision: ULONG;
  RamMB: ULONG;
  BootDrive: ULONG;
  LogFilename: string;
begin
  LogFilename := GetLogFilesDir + CrashLogFileName;
  AssignFile( F, LogFilename );
  FileMode := fmInOut;

  try
    if FileExists( LogFilename ) then
      Append( F )
    else
      Rewrite( F );

    WriteLn( F, 'NewView crash log' );
    WriteLn( F, 'App Version: ' +  GetAppVersion );
    WriteLn( F, 'Components Version: ' +  GetComponentsVersion );
    WriteLn( F, 'Library Version: ' +  GetACLLibraryVersion );
    WriteLn( F, '' );
    WriteLn( F, 'Running as process: ' + GetApplicationFilename );
    WriteLn( F, FormatDateTime( 'd mmmm yyyy, hh:mm:ss', now ) );
    WriteLn( F, 'Exception type: ' + E.ClassName );
    WriteLn( F, 'Description: ' + E.Message );
    WriteLn( F, 'Location: $'
                 + IntToHex( longword( E.ExcptAddr ), 8 ) );

    WriteLn( F, 'Callstack:' );

    for i := 0 to GetExceptionCallCount - 1 do
    begin
      WriteLn( F, ' $' + IntToHex( GetExceptionCallstackEntry( i ), 8 ) );
    end;

    if Files <> nil then
    begin
      WriteLn( F, 'Loaded files ('
                  + IntToStr( Files.Count )
                  + '):' );
      try
        for i := 0 to Files.Count - 1 do
        begin
          HelpFile := Files[ i ];
          WriteLn( F, HelpFile.Filename );
        end;
      except
      end;
    end;
    try
      if CurrentTopic <> nil then
        WriteLn( F, 'Last major topic index: '
                    + IntToStr( CurrentTopic.Index ) );
    except
      // ignore exceptions if there isn't a valid current topic
    end;
    if Windows <> nil then
    begin
      WriteLn( F, 'Top-level open windows: '
                  + IntToStr( Windows.Count ) );
    end;

    OSMajorVersion := GetSystemInfoItem( QSV_VERSION_MAJOR );
    OSMinorVersion := GetSystemInfoItem( QSV_VERSION_MINOR );
    OSRevision :=     GetSystemInfoItem( QSV_VERSION_REVISION );

    WriteLn( F, 'System version:'
                 + IntToStr( OSMajorVersion )
                 + '.'
                 + IntToStr( OSMinorVersion )
                 + '.'
                 + IntToStr( OSRevision ) );
    if OSMajorVersion = 20 then
    begin
      case OSMinorVersion of
        00: WriteLn( F, '  OS/2 2.0' );
        10: WriteLn( F, '  OS/2 2.1' );
        11: WriteLn( F, '  OS/2 2.11' );
        30: WriteLn( F, '  OS/2 3.0' );
        40: WriteLn( F, '  OS/2 4.0' );
        45: WriteLn( F, '  OS/2 4.5' ); // I guess
      end;
    end;

    RamMB := GetSystemInfoItem( QSV_TOTPHYSMEM )
             div ( 1024 * 1024 );
    WriteLn( F, 'RAM: '
                + IntToStr( RamMB )
                + ' MB' );

    BootDrive := GetSystemInfoItem( QSV_BOOT_DRIVE ); // nA = 1
    WriteLn( F, 'Boot drive: '
                + Chr( Ord( 'A' )
                + BootDrive - 1 ) );

    // Video information
    WriteLn( F, 'Video resolution: '
                + IntToStr( Screen.Width )
                + 'x'
                + IntToStr( Screen.Height ) );
    WriteLn( F, 'Color depth: '
                + IntToStr( GetScreenColorDepth )
                + ' bits' );
    WriteLn( F, 'Video driver: '
                + GetVideoDriverName );
    WriteLn( F, '---- End of report ----' );
    WriteLn( F, '' );

    System.Close( F );

    TheText := ApplicationErrorA + EndLine
             + EndLine
             + E.Message + EndLine
             + ApplicationErrorB
             + LogFilename
             + ')' + EndLine
             + EndLine
             + ApplicationErrorC
             + EndLine;

  except
    // Crap! We had a problem writing to the log file...
    // So the message doesn't say there is one.
    TheText := ApplicationErrorA + EndLine
             + EndLine
             + E.Message + EndLine
             + EndLine
             + ApplicationErrorC
             + EndLine;

  end;

  if DoYesNoDlg( ApplicationErrorTitle,
                 TheText ) then
  begin
    Application.Terminate;
  end;
end;

Procedure TMainForm.MainFormOnCreate (Sender: TObject);
Begin
  ProfileEvent( 'MainFormOnCreate' );

  RegisterForLanguages( OnLanguageEvent );

  // set up globals for Exec
  ExecViaSession := true;
  AsynchExec := true;

  // set up form icons
  Forms.FormIconResourceID := 1;

  Application.OnException := OnException;

//  StartProfile( GetLogFilesDir + 'newview.prf' );
  ContentsOutline.SmoothScroll := false;

  ProfileEvent( 'Choosing default font: Trying WarpSans' );

  Font := GetNiceDefaultFont;

  // Set the menu fonts, because they remember their own specific one
  MainMenu.Font := Screen.MenuFont;
  ViewPopupMenu.Font := Screen.MenuFont;

  // NOTE: SPCC will copy this font to TApplication
  // in TApplication.Run

  ProfileEvent( 'Starting NewView: MainFormOnCreate' );

  Application.OnHint := OnHint;

  StartMem := MemAvailBytes;

  DisplayedIndex := TStringList.Create;

  Startup_FilenamesParam := TAString.Create;

  Files := TList.Create;
  Notes := TList.Create;
  Bookmarks := TList.Create;
  BookmarksMenuItems := TList.Create;
  Windows := TList.Create;

  TopicText:= TAString.Create;

  Navigating := false;
  InIndexSearch := false;

  PageHistory := TStringList.Create;
  CurrentHistoryIndex := -1;

  Settings.MRUList := TList.Create;

  MRUMenuItems := TList.Create;
  NavigateToMenuItems := TList.Create;

  // load default strings
  ProfileEvent( 'Loading language' );

  LoadDefaultLanguage;

  ProfileEvent( 'Loading settings' );

  LoadSettings;

  ProfileEvent( 'Applying settings' );

  ApplySettings;

  // default position is centered..
  ProfileEvent( 'Set default position' );
  if Width > Screen.Width then
    Width := Screen.Width;
  if Height > Screen.Height then
    Height := Screen.Height;
  Left := ( Screen.Width - Width ) div 2;
  Bottom := ( Screen.Height - Height ) div 2;

  ProfileEvent( 'ReadWindowPos' );
  ReadWindowPos( Self );

  ProfileEvent( 'Creating MRU list' );

  CreateMRUMenuItems;

  CloseFile;

  ProfileEvent( 'OnCreate done' );

  ParseCommandLineParameters;

  if Startup_OwnHelpMode then
  begin
    ProfileEvent( 'OwnHelpMode, repositioning' );
    // position in top right quarter of screen
    SmartSetWindowPos( self,
                       Screen.Width div 2,
                       Screen.Height div 2,
                       Screen.Width div 2,
                       Screen.Height div 2,
                       false );
  end
  else if Startup_SetPosition then
  begin
    SmartSetWindowPos( self,
                       Startup_Position.Left,
                       Startup_Position.Bottom,
                       Startup_Position.Width,
                       Startup_Position.Height,
                       false );
  end;

  ProfileEvent( 'MainFormOnCreate Done' );
End;

Procedure TMainForm.ApplySettings;
var
  ToolbarBitmap: TBitmap;
begin
  ContentsOutline.Color:= Settings.Colors[ ContentsBackgroundColorIndex ];
  ContentsOutline.PenColor:= Settings.Colors[ ContentsTextColorIndex ];
  ContentsOutline.TreeLineColor:= Settings.Colors[ ContentsLinesColorIndex ];
  IndexListBox.Color:= Settings.Colors[ IndexBackgroundColorIndex ];
  IndexListBox.PenColor:= Settings.Colors[ IndexTextColorIndex ];
  SearchResultsListBox.Color:= Settings.Colors[ SearchBackgroundColorIndex ];
  SearchResultsListBox.PenColor:= Settings.Colors[ SearchTextColorIndex ];
  NotesListBox.Color:= Settings.Colors[ NotesListBackgroundColorIndex ];
  NotesListBox.PenColor:= Settings.Colors[ NotesListTextColorIndex ];

  Coolbar.BackgroundBitmap := nil;
  if FileExists( Settings.ToolbarBackgroundImageFilename ) then
  begin
    ToolbarBitmap:= TBitmap.Create;
    try
      ToolbarBitmap.LoadFromFIle( Settings.ToolbarBackgroundImageFilename );
      Coolbar.BackgroundBitmap := ToolbarBitmap;
    except
    end;
    ToolbarBitmap.Destroy;
  end;

  CoolBar.ShowImages := Settings.ToolbarStyle in [ tsImages, tsImagesAndText ];
  CoolBar.ShowText := Settings.ToolbarStyle in [ tsText, tsImagesAndText ];

  CoolBar.SetMinConstButtonWidth;

  DisplayPanel.Color := Settings.Colors[ TopicBackgroundColorIndex ];

  SetupViews( Windows );
end;

// Setup the rich text views in the specified windows (e.g for changing global settings)
Procedure TMainForm.SetupViews( WindowList: TList );
var
  WindowIndex: longint;
  Window: THelpWindow;
begin
  for WindowIndex := 0 to WindowList.Count - 1 do
  begin
    Window := WindowList[ WindowIndex ];
    Window.SetupRTView;
    SetupViews( Window.ChildWindows );
  end;
end;

Procedure TMainForm.ClearHelpManager;
Begin
  if not Startup_IsHelpManager then
    exit;
  PostMsg( Startup_HelpManagerWindow,
           NHM_FORGET_VIEWER,
           0,
           0 );

  Startup_IsHelpManager := false;
  Startup_HelpManagerWindow := 0;
End;

Procedure TMainForm.MainFormOnShow (Sender: TObject);
var
  SharedMemName: string;
  pSharedStruct: TPNewHelpMgrSharedStruct;
Begin
  ProfileEvent( 'MainFormOnShow' );

  if Startup_OwnerWindow <> NULLHANDLE then
  begin
    ProfileEvent( 'Setting owner: '
                  + IntToStr( Startup_OwnerWindow ) );
    WinSetOwner( Frame.Handle,
                 Startup_OwnerWindow );

  end;

  if Startup_IsHelpManager then
  begin
    ProfileEvent( 'HelpMgr mode: allocate shared memory' );
    SharedMemName := SharedMemBaseName
                     + IntToHex( Startup_HelpManagerWindow, 8 );
    ProfileEvent( '  Shared mem name: '
                  + SharedMemName );
    MessageMemory := TSuballocatedSharedMemory.Create( SharedMemName,
                                                       SharedMemSize,
                                                       SharedMemReserveSize );

    pSharedStruct := TPNewHelpMgrSharedStruct( MessageMemory.Data );
    ProfileEvent( '  Help Manager Title: '
                  + StrNPas( pSharedStruct ^. Title,
                             sizeof( pSharedStruct ^. Title ) ) );
    HelpManagerVersion := StrNPas( pSharedStruct ^. Version,
                                   sizeof( pSharedStruct ^. Version ) );
    ProfileEvent( '  Help Manager Version: '
                  + HelpManagerVersion );
  end;

  CoolBar.SetMinConstButtonWidth;

  ProfileEvent( 'Post WM_OPENED' );

  ResetProgress;

  PostMsg( Handle, WM_OPENED, 0, 0 );
End;

Procedure TMainForm.DisplaySelectedContentsTopic;
var
  Topic: TTopic;
Begin
  if ContentsOutline.SelectedNode = nil then
    exit;
  Topic := ContentsOutline.SelectedNode.Data as TTopic;
  DisplayTopic( Topic );
End;

// Check that the HELP and BOOKSHELF environment variables
// are defined (as they should be on any working OS/2 system).
// Show a warning message if not.
Procedure TMainForm.CheckEnvironmentVars;
var
  HelpOK: boolean;
  BookshelfOK: boolean;
  ErrorText: string;
begin
  HelpOK := GetEnv( HelpPathEnvironmentVar ) <> '';
  BookshelfOK := GetEnv( BookshelfEnvironmentVar ) <> '';
  if HelpOK and BookshelfOK then
    // all ok.
    exit;

  // One or both missing

  ErrorText := '';
  if not BookshelfOK then
    ErrorText := ErrorText
                 + EnvironmentVarUndefined
                 + BookshelfEnvironmentVar
                 + EndLine;

  if not HelpOK then
    ErrorText := ErrorText
                 + EnvironmentVarUndefined
                 + HelpPathEnvironmentVar
                 + EndLine;

  DoWarningDlg( EnvironmentVarErrorTitle,
                EnvironmentVarError
                + EndLine
                + EndLine
                + ErrorText );

end;

Procedure TMainForm.WMOpened( Var Msg: TMessage );
var
  Filenames: TStringList;
  M1: longword;
  OpenFirstTopic: boolean;
begin
  ProfileEvent( 'WMOpened: SetLayout' );
  SetLayout;

//  ProfileEvent( 'Apply settings' );
//  ApplySettings;

  ProfileEvent( 'Enable controls' );
  EnableControls;

  TopicDecodeRecusionLevel := 0;

//  ProfileEvent( 'ReadWindowPos' );
//  ReadWindowPos( Self );

  ProfileEvent( 'Finish paint' );
  Update;

  if not Startup_IsHelpManager then
  begin
    ProfileEvent( 'Check environment vars' );
    CheckEnvironmentVars;

    if Startup_ShowUsageFlag then
    begin
      ProfileEvent( 'Showing usage' );
      ShowUsage;
      exit;
    end;
  end;

  if Startup_OwnHelpMode then
  begin
    // Open our own help file
    OpenFile( GetOwnHelpFilename, '', true );
  end
  else if Startup_FilenamesParam.Length > 0 then
  begin
    // open specified files
    Filenames := TStringList.Create;

    AStringToList( Startup_FilenamesParam, Filenames, '+' );

    ProfileEvent( 'Call OpenFiles' );

    OpenFirstTopic := true;
    if Startup_TopicParam <> '' then
      // if we're going to search, don't open first topic
      OpenFirstTopic := false;
    if Startup_IsHelpManager then
      // don't open first topic if we're online help
      OpenFirstTopic := false;

    OpenFiles( Filenames,
               Startup_WindowTitle,
               OpenFirstTopic );

    Filenames.Destroy;

    if Startup_TopicParam <> '' then
    begin
      // search in specified files
      ProfileEvent( 'Do search for topic' );
      Tabset.TabIndex := piSearch;
      SearchTextEdit.Text := Startup_TopicParam;
      SearchButton.Click;
    end;
  end;

  if Startup_GlobalSearchFlag then
  begin
    // Global search
    ProfileEvent( 'Do global search: ' + Startup_GlobalSearchText );
    DoGlobalSearch( Startup_GlobalSearchText );
  end;

  if     ( Startup_FilenamesParam.Length = 0 )
     and ( not Startup_GlobalSearchFlag ) then
  begin
    // user hasn't requested any particular file
    // at startup, so if the option is still set,
    // load the NewView help file
    if Settings.StartupHelp then
    begin
      if FileExists( GetOwnHelpFileName ) then
      begin
        OpenFile( GetOwnHelpFileName, '', true );
      end;
    end;
  end;

  ProfileEvent( 'Open finished' );

  if Startup_IsHelpManager then
  begin
    // Tell helpmanager our window handle
    PostMsg( Startup_HelpManagerWindow,
             NHM_VIEWER_READY,
             Handle,
             0 );
  end;

  M1:= MemAvail;

  ProfileEvent( 'RUN PROGRAM' );
end;

// Return true if param matches the form
// /Flag:value
// dash (-) can be used instead of slash (/)
// colon can be omitted
function MatchValueParam( const Param: string;
                          const Flag: string;
                          var Value: string ): boolean;
begin
  Result := false;

  if Param = '' then
    exit;

  if     ( Param[ 1 ] <> '/' )
     and ( Param[ 1 ] <> '-' ) then
    exit;

  if not StringsSame( Copy( Param, 2, Length( Flag ) ),
                      Flag ) then
    exit;

  Result := true;

  Value := StrRightFrom( Param, 2 + Length( Flag ) );
  if Value <> '' then
    if Value[ 1 ] = ':' then
      Delete( Value, 1, 1 );
end;

// Return true if param matches the form
// /Flag
// dash (-) can be used instead of slash (/)
function MatchFlagParam( const Param: string;
                         const Flag: string ): boolean;
begin
  Result := false;

  if Param = '' then
    exit;

  if     ( Param[ 1 ] <> '/' )
     and ( Param[ 1 ] <> '-' ) then
    exit;

  Result := StringsSame( StrRightFrom( Param, 2 ),
                         Flag );
end;

// Extract a single element of a window position spec
// - take a value from comma-separated list
// - convert to numeric
// - if the number ends with P then take as
//   a percentage of given dimension
Function ExtractPositionElement( Var ParamValue: string;
                                 ScreenDimension: longint ): longint;
var
  Element: string;
begin
  Element := ExtractNextValue( ParamValue, ',' );
  if Element = '' then
    raise Exception.Create( 'Missing position element' );
  if StrEnds( 'P', Element ) then
  begin
    Delete( Element, Length( Element ), 1 );
    if Element = '' then
      raise Exception.Create( 'Missing position element' );
    Result := StrToInt( Element );
    if Result < 0 then
      Result := 0;
    if Result > 100 then
      Result := 100;
    Result := Round( Result / 100 * ScreenDimension );
  end
  else
  begin
    Result := StrToInt( Element );
  end;
end;

// Extract a specified window position:
// X,Y,W,H
Function ExtractPositionSpec( ParamValue: string;
                              Var Position: TWindowPosition ): boolean;
begin
  try
    Position.Left := ExtractPositionElement( ParamValue, Screen.Width );
    Position.Bottom := ExtractPositionElement( ParamValue, Screen.Height );
    Position.Width := ExtractPositionElement( ParamValue, Screen.Width );
    if Position.Width < 50 then
      Position.Width := 50;
    Position.Height := ExtractPositionElement( ParamValue, Screen.Height );
    if Position.Height < 50 then
      Position.Height := 50;
    Result := true;
  except
    Result := false;
  end;
end;

// Parse command line parameters newview was launched with.
// Store them into the Startup_ variables for later processing.
Procedure TMainForm.ParseCommandLineParameters;
var
  ParamIndex: longint;
  Param: string;
  ParamValue: string;
begin
  ProfileEvent( 'ParseCommandLineParameters started' );
  Startup_FilenamesParam.AssignString( '' );
  Startup_TopicParam := '';
  Startup_ShowUsageFlag := false;
  Startup_GlobalSearchFlag := false;
  Startup_OwnHelpMode := false;
  Startup_OwnerWindow := 0;
  Startup_IsHelpManager := false;
  Startup_HelpManagerWindow := 0;
  Startup_WindowTitle := '';
  Startup_SetPosition := false;

  for ParamIndex := 1 to ParamCount do
  begin
    Param := ParamStr( ParamIndex );
    if    MatchFlagParam( Param, '?' )
       or MatchFlagParam( Param, 'H' )
       or MatchFlagParam( Param, 'HELP' ) then
    begin
      Startup_ShowUsageFlag := true
    end
    else if MatchValueParam( Param, 'G', Startup_GlobalSearchText ) then
    begin
      Startup_GlobalSearchFlag := true;
    end
    else if MatchValueParam( Param, 'HM', ParamValue ) then
    begin
      try
        Startup_HelpManagerWindow := StrToInt( ParamValue );
        Startup_IsHelpManager := true;
      except
        // ignore invalid window value
      end;
    end
    else if MatchValueParam( Param, 'OWNER', ParamValue ) then
    begin
      Startup_OwnerWindow := StrToInt( ParamValue );
    end
    else if MatchValueParam( Param, 'TITLE', ParamValue ) then
    begin
      Startup_WindowTitle := ParamValue;
    end
    else if MatchFlagParam( Param, 'NVHELP' ) then
    begin
      Startup_OwnHelpMode := true
    end
    else if MatchFlagParam( Param, 'PROFILE' ) then
    begin
      StartProfile( GetLogFilesDir + 'newview.prf' );
    end
    else if MatchValueParam( Param, 'POS', ParamValue ) then
    begin
      // set window position/size
      if ExtractPositionSpec( ParamValue,
                              Startup_Position ) then
      begin
        Startup_SetPosition := true;
      end
      else
      begin
        // invalid...
        Startup_ShowUsageFlag := true;
      end;
    end
    else
    begin
      if Startup_FilenamesParam.Length = 0 then
        // filename(s)
        AString_ParamStr( ParamIndex, Startup_FilenamesParam )
      else if Startup_TopicParam = '' then
        // search (topic) parameter
        Startup_TopicParam := Param
      else
        // too many parameters
        Startup_ShowUsageFlag := true;
    end;
  end;

  ProfileEvent( 'Parameters parsed' );
  ProfileEvent( '  Filenames: '
                + Startup_FilenamesParam.AsString );
  ProfileEvent( '  Topic: '
                + Startup_TopicParam );

  // params will be acted on later...
  ProfileEvent( '...done' );

end;

Procedure TMainForm.MainFormOnResize (Sender: TObject);
Begin
  if not Visible then
    exit;
  if Handle = 0 then
    exit;

  SetLayout;

End;

Procedure TMainForm.VSplitBarOnChange (NewSplit: LongInt);
Begin
  Settings.LeftPanelWidth := VSplitBar.Left;
  SetLayout;
End;

// Set the layout of the main form
Procedure TMainForm.SetLayout;
var
  RealClientHeight : longint;
  CoolbarSpace: longint;
Begin

  Coolbar.Visible := Settings.ToolbarStyle <> tsNone;

  case Settings.ToolbarStyle of
    tsImages:
      Coolbar.Height := Coolbar.Sections[ 0 ].Width;
    tsText:
      Coolbar.Height := 22;
    tsImagesAndText:
      CoolBar.Height := 45
  end;

  CoolbarSpace := Coolbar.Height;
  if not Coolbar.Visible then
    CoolbarSpace := 0;

  RealClientHeight := ClientHeight
                      - CoolbarSpace
                      - Notebook.Bottom;

  ProfileEvent( 'TMainForm.SetLayout' );
  ProfileEvent( '  RealClientHeight: '  + IntToStr( RealClientHeight ) );

  ProfileEvent( '  Form Width: '  + IntToStr( Width ) );
  ProfileEvent( '  Form Height: '  + IntToStr( Height ) );
  ProfileEvent( '  Form ClientWidth: '  + IntToStr( ClientWidth ) );
  ProfileEvent( '  Form ClientHeight: '  + IntToStr( ClientHeight ) );
  ProfileEvent( '  CoolBar.Height: '  + IntToStr( CoolBar.Height ) );
  ProfileEvent( '  CoolBar.Bottom: '  + IntToStr( CoolBar.Bottom ) );

  VSplitBar.Left := Settings.LeftPanelWidth;

  Tabset.Left := 0;
  Tabset.Width := VSplitBar.Left;
  Tabset.Bottom := ClientHeight
                   - CoolbarSpace
                   - Tabset.Height;

  Notebook.Left := 0;
  Notebook.Width := VSplitBar.Left;
  Notebook.Height := RealClientHeight
                     - Tabset.Height
                     - 3;

  VSplitBar.Height := RealClientHeight;

  DisplayPanel.Left:= VSplitBar.Left + VSplitBar.Width;
  DisplayPanel.Width:= ClientWidth - DisplayPanel.Left;
  DisplayPanel.Height := RealClientHeight;
  ProfileEvent( '  DisplayPanel.Width: '  + IntToStr( DisplayPanel.Width ) );
  ProfileEvent( '  DisplayPanel.Height: '  + IntToStr( DisplayPanel.Height ) );
  ProfileEvent( '  DisplayPanel.Bottom: '  + IntToStr( DisplayPanel.Bottom ) );

  StatusPanel.Left:= 0;
  StatusPanel.Width:= ClientWidth div 2 - 2;

  ProgressPanel.Left:= ClientWidth div 2 + 2;
  ProgressPanel.Width:= ClientWidth - ProgressPanel.Left;

  ProgressBar.Left:= 1;
  ProgressBar.Bottom:= 1;
  ProgressBar.Width:= ProgressPanel.Width - 2;
  ProgressBar.Height:= ProgressPanel.Height - 2;

  // Layout the visible help windows also
  LayoutWindowList( Windows );
End;

// Lay out the specified list of help windows
Procedure TMainForm.LayoutWindowList( WindowList: TList );
var
  Window: THelpWindow;
  WindowIndex: longint;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Window:= WindowList[ WindowIndex ];
    Window.SetLayout;
  end;
end;

Procedure TMainForm.FindNextMIOnClick (Sender: TObject);
begin
  if FindText = '' then
  begin
    FindMIOnClick( sender );
    exit;
  end;

  if GetActiveWindow = nil then
  begin
    DoErrorDlg( FindTitle,
                FindSelectWindowError );
    exit;
  end;

  DoFind( foFromCurrent );
end;

Procedure TMainForm.DoFind( FindOrigin: TFindOrigin );
var
  Window: THelpWindow;
begin
  Screen.Cursor := crHourglass;
  Window := GetActiveWindow;
  if not Window.View.Find( FindOrigin, FindText ) then
  begin
    SetStatus( TextNotFoundMsg );
    Beep( 1000, 100 );
  end;
  Screen.Cursor := crDefault;
End;

Procedure TMainForm.FindMIOnClick (Sender: TObject);
begin
  if GetActiveWindow = nil then
  begin
    DoErrorDlg( FindTitle,
                FindSelectWindowError );
    exit;
  end;
  if not DoInputQuery( FindTitle,
                       FindPrompt,
                       FindText ) then
    exit;

  DoFind( foFromStart );
End;

Procedure TMainForm.IndexSearchEditOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  case KeyCode of
    kbCUp:
    begin
      if IndexListBox.ItemIndex > 0 then
        IndexListBox.ItemIndex:= IndexListBox.ItemIndex - 1;
      KeyCode:= kbNull;
    end;

    kbCDown:
    begin
      if IndexListBox.ItemIndex < IndexListBox.Items.Count - 1 then
        IndexListBox.ItemIndex:= IndexListBox.ItemIndex + 1;
      KeyCode:= kbNull;
    end;

    kb_VK + VK_NEWLINE:
    begin
      DisplaySelectedIndexTopic;
    end;
  end;

End;

Procedure TMainForm.IndexSearchEditOnChange (Sender: TObject);
var
  MatchIndex: longint;
  IndexINdex: longint;
  SearchText: string;
Begin
  if InIndexSearch then
    exit;

  MatchIndex:= -1;
  SearchText:= trim( IndexSearchEdit.Text );
  for IndexIndex:= 0 to DisplayedIndex.Count - 1 do
  begin
    if StrStarts( SearchText, DisplayedIndex[ IndexIndex ] ) then //IndexEntry ) then
    begin
      MatchIndex:= IndexIndex;
      break;
    end;
  end;

  if MatchIndex = -1 then
    exit;

  InIndexSearch:= true;

  if IndexListBox.ItemIndex <> MatchIndex then
    IndexListBox.ItemIndex:= MatchIndex;

  InIndexSearch:= false;
End;

Procedure TMainForm.FileInformationMIOnClick (Sender: TObject);
var
  FileIndex: longint;
  HelpFile: THelpFile;

  TotalTopicCount: longint;
  TotalIndexCount: longint;
  TotalFileSize: longint;

Begin
  TotalTopicCount := 0;
  TotalIndexCount := 0;
  TotalFileSize := 0;

  with InformationForm.InformationMemo do
  begin
    BeginUpdate;
    Lines.Clear;
    Lines.Add( FilesInfoTitle );
    Lines.Add( FilesInfoOverallTitle + THelpFile( Files[ 0 ] ).Title );
    for FileIndex := 0 to Files.Count - 1 do
    begin
      HelpFile := Files[ FileIndex ];

      Lines.Add( FilesInfoFilename + HelpFile.FileName );
      Lines.Add( FilesInfoFileTitle
                 + HelpFile.Title );
      Lines.Add( FilesInfoTopicCount
                 + IntToStr( HelpFile.TopicCount ) );
      Lines.Add( FilesInfoIndexCount
                 + IntToStr( HelpFile.Index.Count ) );
      Lines.Add( FilesInfoDictionaryCount
                 + IntToStr( HelpFile.DictionaryCount ) );
      Lines.Add( FilesInfoFileSize
                 + IntToStr( HelpFile.FileSize ) );

      inc( TotalTopicCount, HelpFile.TopicCount );
      inc( TotalIndexCount, HelpFile.Index.Count );
      inc( TotalFileSize, HelpFile.FileSize );

    end;

    Lines.Add( '' );
    Lines.Add( FilesInfoTotalTopicCount
               + IntToStr( TotalTopicCount ) );
    Lines.Add( FilesInfoTotalIndexCount
               + IntToStr( TotalIndexCount ) );
    Lines.Add( FilesInfoTotalFileSize
               + IntToStr( TotalFileSize ) );
    Lines.Add( '' );

    EndUpdate;
  end;
  InformationForm.ShowModal;
End;

Procedure TMainForm.DisplaySelectedSearchResultTopic;
var
  Topic: TTopic;
Begin
  if SearchResultsListBox.ItemIndex = -1 then
    exit;
  if SelectedObject( SearchResultsListBox ) = nil then
    // the "no results" place holder
    exit;
  Topic := SelectedObject( SearchResultsListBox ) as TTopic;
  DisplayTopic( Topic );
End;

Procedure TMainForm.SearchTextEditOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  case KeyCode of
    kbCUp:
    begin
      if SearchResultsListBox.ItemIndex > 0 then
        SearchResultsListBox.ItemIndex := SearchResultsListBox.ItemIndex - 1;
      KeyCode := kbNull;
      SearchResultsListBox.Focus;
    end;

    kbCDown:
    begin
      if SearchResultsListBox.ItemIndex < SearchResultsListBox.Items.Count - 1 then
        SearchResultsListBox.ItemIndex := SearchResultsListBox.ItemIndex + 1;
      KeyCode := kbNull;
      SearchResultsListBox.Focus;
    end;
  end;
End;

Procedure TMainForm.SearchButtonOnClick (Sender: TObject);
var
  SearchText: string;
  SearchResults: TList;
  FileIndex: longint;
  HelpFile: THelpFile;
  TopicIndex: longint;
  Topic: TTopic;
  Query: TTextSearchQuery;
Begin
  SearchText := trim( SearchTextEdit.Text );
  if SearchText = '' then
    exit;

  try
    Query := TTextSearchQuery.Create( SearchText );
  except
    on e: ESearchSyntaxError do
    begin
      DoErrorDlg( SearchTitle,
                  SearchSyntaxError
                  + e.Message );
      exit;
    end;
  end;

  Cursor := crHourGlass;

  ViewHighlightSearchWordsMI.Checked := true;

  SearchResults := TList.Create;

  // Search open help file
  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile := Files[ FileIndex ];
    SearchHelpFile( HelpFile,
                    Query,
                    SearchResults,
                    HelpFile.HighlightWords );
  end;

  // Sort results across all files by relevance
  SearchResults.Sort( TopicRelevanceCompare );

  // Load topics into search results list.
  SearchResultsListBox.BeginUpdate;

  SearchResultsListBox.Clear;
  for TopicIndex := 0 to SearchResults.Count - 1 do
  begin
    Topic := SearchResults[ TopicIndex ];
    SearchResultsListBox.Items.AddObject( Topic.Title
                                          + ' ['
                                          + IntToStr( Topic.SearchRelevance )
                                          + ']',
                                          Topic );
  end;

  SearchResultsListBox.ItemIndex := -1;
  SearchResultsListBox.EndUpdate;
  Application.ProcessMessages; // make sure list gets displayed

  Query.Destroy;
  SearchResults.Destroy;

  if SearchResultsListBox.Items.Count > 0 then
  begin
    SearchResultsListBox.ItemIndex := 0;
  end
  else
  begin
    SearchResultsListBox.Items.Add(   NoSearchMatchesMsg
                                    + StrDoubleQuote( SearchText ) );
  end;
  SetStatus( SearchFoundMsgA
             + IntToStr( SearchResultsListBox.Items.Count )
             + SearchFoundMsgB
             + StrDoubleQuote( SearchText ) );

  Cursor := crDefault;

  DisplaySelectedSearchResultTopic;

End;

Procedure TMainForm.FileSaveAsMIOnClick (Sender: TObject);
var
  F: File;
  EntryText: PChar;
  TextLength: longint;
  Window: THelpWindow;
  Filename: string;
Begin
  Window := GetActiveWindow;
  if Window = nil then
  begin
    DoErrorDlg( FileSaveTitle,
                FileSaveSelectWindowError );
    exit;
  end;

  if DoSaveFileDialog( FileSaveTitle,
                       AllFilesDesc + '|*',
                       DefaultSaveTopicFilename,
                       Settings.LastSaveDirectory,
                       Filename ) then
  begin
    if FileExists( Filename ) then
      if not DoConfirmDlg( FileSaveTitle,
                           ReplaceFilePromptA
                           + Filename
                           + ReplaceFilePromptB ) then
        exit;
    System.Assign( F, Filename );

    try
      Rewrite( F );
    except
      on E: Exception do
      begin
        DoErrorDlg( FileSaveTitle,
                    UnableToSaveError
                    + Filename
                    + ': '
                    + E.Message );
        exit;
      end;
    end;

    // find out length of (plain) text
    TextLength := Window.View.CopyTextToBuffer( nil, -1 );

    // allocate space
    EntryText:= StrAlloc( TextLength );

    // get the plain text
    Window.View.CopyTextToBuffer( EntryText, TextLength );

    // save to file
    System.BlockWrite( F, EntryText^, TextLength );

    // free space
    StrDispose( EntryText );

    System.Close( F );
  end;
End;

Procedure TMainForm.OptionsMIOnClick (Sender: TObject);
begin
  DoOptions;
end;

Procedure TMainForm.DoOptions;
Begin
  EnsureOptionsFormLoaded;

  GetColors; // in case changed by drag drop

  if OptionsForm.ShowModal = mrOK then
  begin
    ApplySettings;
    SetLayout;
    RefreshWindows( Windows );
  end;
End;

Procedure TMainForm.ShowUsage;
begin
  DoMessageDlg( UsageTitle,
                  UsageText1 + EndLine
                + UsageText2 + EndLine
                + UsageText3 + EndLine
                + UsageText4 + EndLine
                + UsageText5 );
end;

Procedure TMainForm.TabSetOnChange (Sender: TObject; NewTab: LongInt;
  Var AllowChange: Boolean);
Begin
  NoteBook.PageIndex := NewTab;
End;

Procedure TMainForm.NotebookOnSetupShow (Sender: TObject);
Begin
  ContentsOutline.xStretch := xsFrame;
  ContentsOutline.yStretch := ysFrame;

  IndexSearchEdit.yAlign := yaTop;
  IndexSearchEdit.xStretch := xsFrame;
  IndexListBox.yStretch := ysFrame;
  IndexListBox.xStretch := xsFrame;

  SearchTextEdit.yAlign := yaTop;
  SearchTextEdit.xStretch := xsFrame;
  SearchButton.yAlign := yaTop;
  SearchButton.xAlign := xaRight;
  SearchResultsListBox.xStretch := xsFrame;
  SearchResultsListBox.yStretch := ysFrame;

  NotesListBox.xStretch := xsFrame;
  NotesListBox.yStretch := ysFrame;
End;

Procedure TMainForm.EnableControls;
var
  BackEnabled: boolean;
  ForwardEnabled: boolean;
  FileOpen: boolean;
  WindowOpen: boolean;
begin
  BackEnabled := CurrentHistoryIndex > 0;
  ForwardEnabled := CurrentHistoryIndex < PageHistory.Count - 1;


  FileOpen := Files.Count > 0;
  WindowOpen := Windows.Count > 0;

  Coolbar.Sections[ ciBack ].Disabled := not BackEnabled;
  NavigateBackMI.Enabled := BackEnabled;
  Coolbar.Sections[ ciForward ].Disabled := not ForwardEnabled;
  NavigateForwardMI.Enabled := ForwardEnabled;

  FileSaveAsMI.Enabled := FileOpen;

  Coolbar.Sections[ ciPrint ].Disabled := not FileOpen;
  PrintMI.Enabled := FileOpen;
  FileInformationMI.Enabled := FileOpen;

  Coolbar.Sections[ ciAddBookmark ].Disabled := not FileOpen;
  AddBookmarkMI.Enabled := FileOpen;
  EditBookmarksMI.Enabled := FileOpen;
  Coolbar.Sections[ ciAddNote ].Disabled := not FileOpen;
  AddNoteMI.Enabled := FileOpen;

  Coolbar.Sections[ ciNext ].Disabled := not FileOpen;
  NavigateNextMI.Enabled := FileOpen;
  Coolbar.Sections[ ciPrevious ].Disabled := not FileOpen;
  NavigatePreviousMI.Enabled := FileOpen;

  FileCloseMI.Enabled := FileOpen;

  FindMI.Enabled := WindowOpen;
  FindNextMI.Enabled := WindowOpen;
  CopyMI.Enabled := WindowOpen;
  SelectAllMI.Enabled := WindowOpen;

  ViewExpandAllMI.Enabled := ContentsOutline.ChildCount > 0;
  ViewCollapseAllMI.Enabled := ContentsOutline.ChildCount > 0;

  EnableSearchButton;
  EnableNotesControls;
end;

Procedure TMainForm.NavigateBackMIOnClick (Sender: TObject);
begin
  NavigateBack;
end;

Procedure TMainForm.SaveNavigatePoint;
var
  NavPoint: TNavigatePoint;
begin
  // delete rest of history.
  while CurrentHistoryIndex < PageHistory.Count - 1 do
  begin
    NavPoint:= PageHistory.Objects[ CurrentHistoryIndex + 1 ] as TNavigatePoint;
    NavPoint.Destroy;
    PageHistory.Delete( CurrentHistoryIndex + 1 );
  end;

  NavPoint:= TNavigatePoint.Create;
  SaveWindows( Windows, NavPoint.Windows, nil );

  if ContentsOutline.SelectedNode <> nil then
    NavPoint.ContentsTopic:= ContentsOutline.SelectedNode.Data as TTopic
  else
    NavPoint.ContentsTopic:= nil;

  if CurrentTopic <> nil then
    PageHistory.AddObject( CurrentTopic.Title, NavPoint )
  else
    PageHistory.AddObject( '', NavPoint );

  inc( CurrentHistoryIndex );

  CreateNavigateToMenuItems;
end;

Procedure TMainForm.UpdateCurrentNavigatePoint;
var
  NavPoint: TNavigatePoint;
begin
  if CurrentHistoryIndex = -1 then
    exit;

  NavPoint:= PageHistory.Objects[ CurrentHistoryIndex ] as TNavigatePoint;

  DestroyListObjects( NavPoint.Windows );
  NavPoint.Windows.Clear;

  SaveWindows( Windows, NavPoint.Windows, nil );
end;

Procedure TMainForm.ClearPageHistory;
var
  i: longint;
  NavPoint: TNavigatePoint;
begin
  for i := 0 to PageHistory.Count - 1 do
  begin
    NavPoint := PageHistory.Objects[ i ] as TNavigatePoint;
    NavPoint.Destroy;
  end;
  PageHistory.Clear;
  CurrentHistoryIndex := -1;
  CreateNavigateToMenuItems;
  EnableControls;
end;

Procedure TMainForm.SaveWindows( SourceList: TList;
                                 DestList: TList;
                                 Parent: TSavedHelpWindow );

var
  WindowIndex: longint;
  Window: THelpWindow;
  WindowCopy: TSavedHelpWindow;
begin
  // limit storage to only what's need since list will be static.
  DestList.Capacity := SourceList.Count;
  for WindowIndex := 0 to SourceList.Count - 1 do
  begin
    Window := SourceList[ WindowIndex ];
    WindowCopy := TSavedHelpWindow.Create;
    WindowCopy.Parent := Parent;
    WindowCopy.Rect.Assign( Window.Rect );
    WindowCopy.Topic := Window.Topic;
    WindowCopy.Group := Window.Group;
    WindowCopy.TopCharIndex := Window.View.TopCharIndex;
    SaveWindows( Window.ChildWindows, WindowCopy.ChildWindows, WindowCopy );
    DestList.Add( WindowCopy );
  end;
end;

Procedure TMainForm.DisplayWindows( WindowList: TList;
                                    Parent: THelpWindow );
var
  WindowIndex: longint;
  WindowCopy: TSavedHelpWindow;
  NewWindow: THelpWindow;
begin
  for WindowIndex := 0 to WindowList.Count - 1 do
  begin
    WindowCopy := WindowList[ WindowIndex ];
    NewWindow := OpenWindow( WindowCopy.Topic,
                             WindowCopy.Group,
                             Parent,
                             WindowCopy.Rect,
                             false ); // don't follow links
    NewWindow.View.TopCharIndex := WindowCopy.TopCharIndex;
    DisplayWindows( WindowCopy.ChildWindows, NewWindow );
  end;
end;

Procedure TMainForm.CreateNavigateToMenuItems;
var
  MenuItem: TMenuItem;
  i: integer;
begin
  // clear existing items
  DestroyListObjects( NavigateToMenuItems );
  NavigateToMenuItems.Clear;

  if CurrentHistoryIndex > 0 then
  begin
    // We are going to add some items, so
    // add a seperator from the rest of the menu first
    MenuItem := TMenuItem.Create( self );
    MenuItem.Caption:= '-';
    NavigateMenu.Add( MenuItem );
    NavigateToMenuItems.Add( MenuItem );
  end;

  i := CurrentHistoryIndex - 1; // don't include the current history item
  while (     ( i >= 0 )
          and ( i > CurrentHistoryIndex - 10 ) ) do
  begin
    MenuItem := TMenuItem.Create( self );
    MenuItem.Caption := PageHistory[ i ];
    MenuItem.Hint := GoBackHint
                     + StrDoubleQuote( PageHistory[ i ] );
    MenuItem.OnClick := OnNavigateToMenuItemClick;
    MenuItem.Tag := i;

    NavigateMenu.Add( MenuItem );
    NavigateToMenuItems.Add( MenuItem );
    dec( i );
  end;
end;

Procedure TMainForm.NavigateToPoint( NavPoint: TNavigatePoint );
begin
  Navigating := true;

  // close current windows
  CloseWindows;

  // Display windows for the navigate point
  DisplayWindows( NavPoint.Windows, nil );

  // Select the contents topic
  ContentsOutline.SetSelectedObject( NavPoint.ContentsTopic );

  // Update the navigate menu (since a different set of
  // back-points are now available)
  CreateNavigateToMenuItems;

  // Make the topic windows visible
  ShowWindows;

  // Update back buttons etc...
  EnableControls;

  Navigating := false;
end;

Procedure TMainForm.NavigateToHistoryIndex( Index: longint );
var
  NavPoint: TNavigatePoint;
begin
  UpdateCurrentNavigatePoint;
  CurrentHistoryIndex := Index;
  NavPoint := PageHistory.Objects[ CurrentHistoryIndex ] as TNavigatePoint;
  NavigateToPoint( NavPoint );
end;

Procedure TMainForm.NavigateForward;
Begin
  if CurrentHistoryIndex < PageHistory.Count - 1 then
  begin
    NavigateToHistoryIndex( CurrentHistoryIndex + 1 );
  end;
End;

Procedure TMainForm.NavigateBack;
Begin
  if CurrentHistoryIndex > 0 then
  begin
    NavigateToHistoryIndex( CurrentHistoryIndex - 1 );
  end;
End;

Procedure TMainForm.NavigatePreviousInContents;
begin
  ContentsOutline.GotoNextNodeUp;
  DisplaySelectedContentsTopic;
end;

Procedure TMainForm.NavigateNextInContents;
begin
  ContentsOutline.GotoNextNodeDown;
  DisplaySelectedContentsTopic;
end;

Procedure TMainForm.CorrectNotesPositions( Topic: TTopic;
                                           Text: pchar );
var
  NoteIndex: longint;
  Note: THelpNote;
  p: pchar;
  NextP: pchar;
  Element: TTextElement;
  TextIndex: longint;
begin
  NoteIndex := 0;
  for NoteIndex := 0 to Notes.Count - 1 do
  begin
    Note := Notes[ NoteIndex ];
    if Note.Topic = Topic then
    begin
      // this note belongs the the specified topic.
      p := Text;

      while true do
      begin
        Element := ExtractNextTextElement( p, NextP );
        if Element.ElementType = teTextEnd then
          break;
        TextIndex := PCharDiff( p, Text );
        if TextIndex >= Note.InsertPoint then
        begin
          // found a safe point to insert
          if TextIndex <> Note.InsertPoint then
          begin
            // correct it.
            Note.InsertPoint := TextIndex;
          end;
          break;
        end;

        p := NextP;
      end;
    end;
  end;
end;

Procedure TMainForm.InsertNotesIntoTopicText( Topic: TTopic;
                                              Text: TAString );
var
  NoteIndex: longint;
  Note: THelpNote;
  ActualInsertPoint: longword;
begin
  CorrectNotesPositions( Topic, Text.AsPChar );

  for NoteIndex := 0 to Notes.Count - 1 do
  begin
    Note := Notes[ NoteIndex ];
    if Note.Topic = Topic then
    begin
         // Adjust insert point for any notes we have already inserted.
      ActualInsertPoint := FindActualNoteCharIndex( Note.InsertPoint,
                                                    NoteIndex,
                                                    Topic );
      RefreshNoteInsertInfo( NoteIndex );
      Text.Insert( ActualInsertPoint, Note.InsertText );
    end;
  end;
end;

Procedure TMainForm.NavigatePreviousMIOnClick (Sender: TObject);
Begin
  NavigatePreviousInContents;
End;

Procedure TMainForm.NavigateNextMIOnClick (Sender: TObject);
Begin
  NavigateNextInContents;
End;

Function TMainForm.GetActiveWindow: THelpWindow;
var
  View: TRichTextView;
  FirstWindow: THelpWindow;
begin
  Result := nil;
  if Screen.ActiveControl is TRichTextView then
  begin
    View := Screen.ActiveControl as TRichTextView;
    Result := FindWindowFromView( View, Windows );
  end
  else if Windows.Count = 1 then
  begin
    FirstWindow := Windows[ 0 ];
    if FirstWindow.ChildWindows.Count = 0 then
      Result := FirstWindow;
  end;
end;

Procedure TMainForm.CopyMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;

  Window.View.CopySelectionToClipboard;
End;

Procedure TMainForm.SelectAllMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
begin
  Window:= GetActiveWindow;
  if Window = nil then
  begin
    DoErrorDlg( SelectAllTitle,
                SelectAllWindowError );
    exit;
  end;
  Window.View.SelectAll;
End;

Procedure TMainForm.DebugShowCodesMIOnClick (Sender: TObject);
Begin
  DebugShowCodesMI.Checked:= not DebugShowCodesMI.Checked;
  RefreshWindows( Windows );
End;

Procedure TMainForm.HelpProductInformationMIOnClick (Sender: TObject);
Begin
  EnsureProductInformationFormLoaded;
  ProductInformationForm.ShowModal;
End;

Procedure TMainForm.OnOverLink ( Sender: TRichTextView; LinkString: String);
var
  Link: THelpLink;
  LinkIndex: longint;
  Window: THelpWindow;
  LinkedTopic: TTopic;
Begin
  if StrLeft( LinkString, 4 ) = 'note' then
  begin
    SetStatus( EditNoteMsg )
  end
  else if StrLeft( LinkString, 8 ) = 'external' then
  begin
    SetStatus( ExternalLinkMsg );
  end
  else
  begin
    Window:= FindWindowFromView( Sender, Windows );
    LinkIndex:= StrToInt( LinkString );
    Link:= Window.Topic.Links[ LinkIndex ];

    LinkedTopic := FindTopicForLink( Link );

    if LinkedTopic <> nil then
    begin
      SetStatus( LinkMsg
                 + StrDoubleQuote( Trim( LinkedTopic.Title ) ) );
    end
    else
    begin
      SetStatus( UnknownLinkMsg );
    end;
  end;
End;

Procedure TMainForm.OnNotOverLink ( Sender: TRichTextView; LinkString: String);
Begin
  SetStatus( '' );
end;

Procedure TMainForm.OnClickLink ( Sender: TRichTextView; LinkString: String);
var
  Link: THelpLink;
  LinkIndex: longint;
  SourceWindow: THelpWindow;
  NoteIndex: longint;
Begin
  if StrLeft( LinkString, 4 ) = 'note' then
  begin
    NoteIndex := StrToInt( StrRightFrom( LinkString, 5 ) );
    NotesListBox.ItemIndex := NoteIndex;
    EditNote( NoteIndex );
  end
  else if StrLeft( LinkString, 8 ) = 'external' then
  begin
    DoMessageDlg( ExternalLinkTitle,
                  ExternalLinkError );
  end
  else
  begin
    SourceWindow:= FindWindowFromView( Sender, Windows );
    LinkIndex:= StrToInt( LinkString );
    Link:= SourceWindow.Topic.Links[ LinkIndex ];

    PostMsg( Self.Handle,
             WM_FOLLOWLINK,
             longint( Link ),
             longint( SourceWindow ) );

  end;
End;

Procedure TMainForm.OnWindowAboutToClose( Window: THelpWindow;
                                          var CanClose: boolean );
begin
  if Navigating then
    exit;

  UpdateCurrentNavigatePoint; // Save it before close...

  CanClose := true;
end;

Procedure TMainForm.RemoveHelpWindowFromParent( Window: THelpWindow );
var
  ParentWindow: THelpWindow;
  WindowIndex: longint;
Begin
  if Navigating then
    exit;

  if Window.ParentHelpWindow = nil then
  begin
    WindowIndex := Windows.IndexOf( Window );
    Windows.Delete( WindowIndex );
  end
  else
  begin
    ParentWindow := Window.ParentHelpWindow;
    WindowIndex := ParentWindow.ChildWindows.IndexOf( Window );
    ParentWindow.ChildWindows.Delete( WindowIndex );
  end;
end;

Procedure TMainForm.OnWindowClose( Window: THelpWindow );
Begin
  if Navigating then
    exit;

  RemoveHelpWindowFromParent( Window );

  SaveNavigatePoint;
  EnableControls;
End;

Procedure TMainForm.BackButtonOnClick (Sender: TObject);
Begin
  NavigateBack;
End;

Procedure TMainForm.RTViewOnSetupShow (Sender: TObject);
Begin
End;

Procedure TMainForm.ExitMIOnClick (Sender: TObject);
Begin
  Close;
End;

Procedure TMainForm.CreateMRUMenuItems;
var
  MenuItem: TMenuItem;
  i: integer;
  FileName: string;
  FileNameIndex: longint;
  MRUText: string;
  MRUItem: TMRUItem;
begin
  DestroyListObjects( MRUMenuItems );
  MRUMenuItems.Clear;

  // if there are Most Recently Used files
  if Settings.MRUList.Count > 0 then
  begin
    // create a seperator after Exit
    MenuItem:= TMenuItem.Create( self );
    MenuItem.Name := 'MRUSeparatorMI';
    MenuItem.Caption:= '-';
    FileMenu.Add( MenuItem );
    MRUMenuItems.Add( MenuItem );
  end;

  // Add items for the MRU files
  for i:= 0 to Settings.MRUList.Count -1 do
  begin
    MRUItem := Settings.MRUList[ i ];

    MenuItem := TMenuItem.Create( self );

    MenuItem.Name := 'MRUItem' + IntToStr( i ) + 'MI';
    MRUText := MRUItem.Title;
    if Trim( MRUText ) = '' then
    begin
      // Take the filenames, less path, as caption...
      MRUText := '';
      for FileNameIndex := 0 to MRUItem.Filenames.Count - 1 do
      begin
        FileName := MRUItem.Filenames[ FileNameIndex ];
        FileName := ExtractFileName( FileName );
        FileName := ChangeFileExt( FileName, '' );// remove extension
        AddToListString( MRUText,
                         FileName,
                         '+' );

        // stop after 50 chars
        if Length( MRUText ) > 50 then
        begin
          MRUText := MRUText + '+ ...';
          break;
        end;
      end;
    end;

    MenuItem.Caption:= '~'
                       + IntToStr( i + 1 )
                       + '. '
                       + MRUText;
    if MRUItem.Filenames.Count = 1 then
      MenuItem.Hint := MRUItem.Filenames[ 0 ]
    else
      MenuItem.Hint := MRUItem.Title
                       + ' ('
                       + IntToStr( MRUItem.Filenames.Count )
                       + ' '
                       + MRUMultipleFilesHint
                       + ')';

    MenuItem.OnClick:= OnMRUMenuItemClick;
    MenuItem.Tag:= i;
    FileMenu.Add( MenuItem );
    MRUMenuItems.Add( MenuItem );
  end;
end;

procedure TMainForm.OnMRUMenuItemClick( Sender: TObject );
var
  Tag: longint;
  MenuItem: TMenuItem;
  MRUItem: TMRUItem;
begin
  MenuItem:= Sender as TMenuItem;
  Tag:= MenuItem.Tag;
  MRUItem := Settings.MRUList[ Tag ];
  if OpenFiles( MRUItem.FileNames, '', true ) then
  begin
    Startup_OwnHelpMode := false;
    ClearHelpManager;
  end;
end;

Procedure TMainForm.OnNavigateToMenuItemClick( Sender: TObject );
var
  MenuItem: TMenuItem;
  Tag: longint;
begin
  MenuItem:= Sender as TMenuItem;
  Tag:= MenuItem.Tag;
  NavigateToHistoryIndex( Tag );
end;

Procedure TMainForm.AddChildNodes( HelpFile: THelpFile;
                                   ParentNode: TNode;
                                   Level: longint;
                                   Var TopicIndex: longint );
var
  Topic: TTopic;
  Node: TNode;
begin
  assert( ParentNode <> nil );
  Node := nil;
  while TopicIndex < HelpFile.TopicCount do
  begin
    Topic:= HelpFile.Topics[ TopicIndex ];

    if Topic.ShowInContents then
    begin
      if Topic.ContentsLevel < Level then
        break;

      if Topic.ContentsLevel = Level then
      begin
        Node:= ParentNode.AddChild( Topic.Title,
                                    Topic );
        inc( TopicIndex );
      end
      else
      begin
        assert( Node <> nil );
        AddChildNodes( HelpFile,
                       Node,
                       Topic.ContentsLevel,
                       TopicIndex );
        Node := nil;
      end
    end
    else
    begin
      inc( TopicIndex );
    end;
  end;

end;

Procedure TMainForm.LoadContents;
var
  TopicIndex: longint;
  Topic: TTopic;
  Node: TNode;
  FileIndex: longint;
  HelpFile: THelpFile;
begin
  ContentsOutline.BeginUpdate;
  ProfileEvent( 'Load contents outline' );

  ContentsOutline.Clear;

  ProfileEvent( 'Loop files' );

  Node := nil;

  for FileIndex:= 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    ProfileEvent( 'File ' + IntToStr( FileIndex ) );
    TopicIndex:= 0;
    while TopicIndex < HelpFile.TopicCount do
    begin
      Topic:= HelpFile.Topics[ TopicIndex ];
      assert( Topic.ContentsLevel >= 0,
              'Topic contents level is ' + IntToStr( Topic.ContentsLevel ) );
      if Topic.ShowInContents then
      begin
        if Topic.ContentsLevel = 1 then
        begin
          Node:= ContentsOutline.AddChild( Topic.Title,
                                           Topic );
          inc( TopicIndex );
        end
        else
        begin
          // subnodes
          assert( Node <> nil, 'No level 1 topic for subnodes!' );
          AddChildNodes( HelpFile,
                         Node,
                         Topic.ContentsLevel,
                         TopicIndex );
          Node := nil;
        end;
      end
      else
      begin
        inc( TopicIndex );
      end;
    end;
  end;
  ProfileEvent( '  EndUpdate' );
  ContentsOutline.EndUpdate;
  ProfileEvent( '  Contents loaded' );

end;

Procedure TMainForm.SaveNotesForFile( HelpFile: THelpFile );
var
  NotesFileName: string;
  TopicIndex: longword;
  Note: THelpNote;
  NoteIndex: longint;

  NotesFile: HFile;
  OpenAction: ULong;
  rc: APIRET;
  CName: Cstring;
  FileNoteCount: integer;

begin
  NotesFileName:= ChangeFileExt( HelpFile.FileName, '.nte' );

  FileNoteCount:= 0;
  for  NoteIndex:= 0 to Notes.Count - 1 do
  begin
    Note:= Notes[ NoteIndex ];

    if Note.Topic.HelpFile = HelpFile then
      inc( FileNoteCount );
  end;

  if FileNoteCount = 0 then
  begin
    // no notes. delete notes file if it already exists.
    if FileExists( NotesFileName ) then
      DeleteFile( NotesFileName );
    exit;
  end;

  CName:= NotesFileName;
  rc:= DosOpen( CName,
                NotesFile,
                OpenAction,
                0, // file size
                0, // attrs
                OPEN_ACTION_CREATE_IF_NEW + OPEN_ACTION_REPLACE_IF_EXISTS,
                OPEN_SHARE_DENYREADWRITE + OPEN_ACCESS_WRITEONLY,
                nil ); // no eas
  if rc <> 0 then
  begin
    DoErrorDlg( SaveNotesTitle,
                SaveNotesError
                + EndLine
                + NotesFileName
                + EndLine
                + SysErrorMessage( rc ) );
    exit;
  end;

  for  NoteIndex:= 0 to Notes.Count - 1 do
  begin
    Note:= Notes[ NoteIndex ];

    if Note.Topic.HelpFile <> HelpFile then
      continue;

    TopicIndex:= HelpFile.IndexOfTopic( Note.Topic );

    MyWriteLn( NotesFile,
               IntToStr( TopicIndex ));
    MyWriteLn( NotesFile,
               IntToStr( Note.InsertPoint ) );

    MyWrite( NotesFile,
             Note.Text.AsPChar,
             Note.Text.Length );

    MyWriteLn( NotesFile,
               '' );
    MyWriteLn( NotesFile,
               '#ENDNOTE#' );

  end;

  DosClose( NotesFile );
end;

Procedure TMainForm.LoadNotes( HelpFile: THelpFile );
var
  NotesFileName: string;
  TopicIndex: longint;
  InsertPoint: longint;
  Note: THelpNote;

  NotesFile: HFile;
  OpenAction: ULong;
  rc: APIRET;
  CName: Cstring;

  Paragraph: TAString;
  NotEOF: boolean;
  NoteText: TAString;

begin
  NotesFileName := ChangeFileExt( HelpFile.FileName, '.nte' );

  if not FileExists( NotesFileName ) then
    // no notes
    exit;

  CName := NotesFileName;
  rc := DosOpen( CName,
                 NotesFile,
                 OpenAction,
                 0, // file size - irrelevant, not creating,
                 0, // attrs - ''
                 OPEN_ACTION_OPEN_IF_EXISTS,
                 OPEN_SHARE_DENYREADWRITE + OPEN_ACCESS_READONLY,
                 nil ); // no eas
  if rc <> 0 then
  begin
    DoErrorDlg( LoadNotesTitle,
                LoadNotesError
                + EndLine
                + NotesFileName
                + EndLine
                + SysErrorMessage( rc ) );
    exit;
  end;

  Paragraph := TAString.Create;
  NoteText := TAString.Create;

  NotEOF := true;

  while NotEOF do
  begin
    // Read contents index
    NotEOF := Paragraph.ReadParagraph( NotesFile );
    if not NotEOF then
      continue;
    try
      TopicIndex := StrToInt( Paragraph.AsString );
    except
      TopicIndex := -1;
    end;

    // Read insert point
    NotEOF := Paragraph.ReadParagraph( NotesFile );
    if not NotEOF then
      continue;
    try
      InsertPoint := StrToInt( Paragraph.AsString );
    except
      InsertPoint := -1;
    end;

    NoteText.Clear;

    while NotEOF do
    begin
      NotEOF := Paragraph.ReadParagraph( NotesFile );
      if Paragraph.SameAs( '#ENDNOTE#' ) then
      begin
        // found end of note
        if     ( TopicIndex >= 0 )
           and ( InsertPoint >= 0 ) then
        begin
          Note := THelpNote.Create;
          Note.Topic := HelpFile.Topics[ TopicIndex ];
          Note.InsertPoint := InsertPoint;

          // Remove the last end line
          Note.Text.Assign( NoteText );
          if Note.Text.Length > 2 then
            Note.Text.Delete( Note.Text.Length - 2, 2 );

          Notes.Add( Note );
        end;
        break;
      end;
      NoteText.Add( Paragraph );
      NoteText.AddString( #13 + #10 );
    end;

  end;
  DosClose( NotesFile );

  Paragraph.Destroy;
  NoteText.Destroy;

end;

Procedure TMainForm.UpdateNotesDisplay;
var
  NoteIndex: longint;
  Note: THelpNote;
  NoteTitle: string;
begin
  NotesListBox.Clear;
  for NoteIndex := 0 to Notes.Count - 1 do
  begin
    Note := Notes[ NoteIndex ];

    if Note.Topic > nil then
      NoteTitle := Note.Topic.Title
    else
      NoteTitle := StrLeft( Note.Text.AsString, 100 );
    NotesListBox.Items.AddObject( NoteTitle,
                                  Note );
  end;
  EnableNotesControls;
end;

Procedure TMainForm.EnableNotesControls;
var
  NoteSelected: boolean;
begin
  NoteSelected:= NotesListBox.ItemIndex <> -1;
  EditNoteButton.Enabled:= NoteSelected;
  GotoNoteButton.Enabled:= NoteSelected;
  DeleteNoteButton.Enabled:= NoteSelected;
  AddNoteButton.Enabled := Files.Count > 0;
end;

Procedure TMainForm.CloseFile;
var
  FileIndex: longint;
  HelpFile: THelpFile;
  M1: longint;
begin
  ProfileEvent( 'Set Caption' );
  MainTitle := '';
  SetMainCaption;

  ProfileEvent( 'Close Windows' );
  CloseWindows;
  ProfileEvent( 'Set selected node to nil' );

  ContentsOutline.SelectedNode:= Nil;

  M1:= MemAvail;

  ContentsOutline.Clear;

  ProfileEvent( 'Free contents: ' + IntToStr( MemAvail - M1 ) );
  M1:= MemAvail;

  DisplayedIndex.Clear;
  IndexListBox.Clear;
  ProfileEvent( 'Clear index ' + IntToStr( MemAvail - M1 ) );
  M1:= MemAvail;

  NotesListBox.Clear;
  SearchResultsListBox.Clear;

  ProfileEvent( 'Notes, search etc ' + IntToStr( MemAvail - M1 ) );
  M1:= MemAvail;

  ProfileEvent( 'Save bookmarks' );

  // First save notes and bookmarks.
  // It's important we do this first
  // since we scan all notes each time to find the ones
  // belonging to this file.
  SaveBookmarks;

  ProfileEvent( 'Save notes' );

  SaveNotes;

  ProfileEvent( 'Destroy helpfile objects' );

  // Now destroy help files
  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile := Files[ FileIndex ];
    HelpFile.Free;
  end;

  Files.Clear;

  ProfileEvent( 'Destroy helpfiles ' + IntToStr( MemAvail - M1 ) );
  M1 := MemAvail;

  ProfileEvent( 'Clear notes' );
  ClearNotes;

  ProfileEvent( 'Clear bookmarks' );
  ClearBookmarks;
  if Assigned( BookmarksForm ) then
    BookmarksForm.Hide;

  ClearPageHistory;

  ProfileEvent( 'Enable controls' );
  EnableControls;

  ProfileEvent( 'CloseFile done' );

end;

Function TMainForm.FindOpenHelpFile( FileName: string ): THelpFile;
var
  FileIndex: longint;
begin
  for FileIndex:= 0 to Files.Count - 1 do
  begin
    Result:= Files[ FileIndex ];
    if StringsSame( Result.Filename, FileName ) then
      // found
      exit;
  end;
  Result:= nil;
end;

// This rather horrendous looking bit of code simply:

// Gets the contents from each file
// Sorts it alphabetically.
// Merges all the sorted contents and indexes together,
// alphabetically.
type
  TListType = ( ltContents, ltIndex );

procedure TMainForm.LoadIndex;
var
  HelpFile: THelpFile;
  TextCompareResult: integer;

  FileIndex: longint;

  Contents: TList;
  ContentsLists: TList; // of tlist
  IndexLists: TList; // of tstringlist
  ContentsNextIndex: array[ 0..255 ] of longint;
  IndexNextIndex: array[ 0..255 ] of longint;
  Topic: TTopic;

  ListIndex: longint;

  pListEntry: pstring;
  pLowestEntry: pstring;
  pLastEntry: pstring;

  LowestEntryListIndex: longint;
  LowestEntryListType: TListType;
  LowestEntryTopic: TTopic;

  Index: TStringList;

  i : longint;
begin
  ProfileEvent( 'Create index' );
  ProfileEvent( '  Get/sort lists' );

  ProgressBar.Position:= 70;
  SetStatus( 'Building index... ' );

  ContentsLists := TList.Create;
  IndexLists := TList.Create;

  // collect the contents and index lists from the files
  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile := Files[ FileIndex ];
    ProgressBar.Position := 70 + 10 * FileIndex div Files.Count;

    if Settings.IndexStyle in [ isAlphabetical, isFull ] then
    begin
      Contents := TList.Create;

      // copy [contents] topic list
      for i := 0 to HelpFile.TopicCount - 1 do
      begin
        Topic := HelpFile.Topics[ i ];
        if Topic.ShowInContents then
          Contents.Add( Topic );
      end;

      // sort by title
      Contents.Sort( TopicTitleCompare );

      ContentsLists.Add( Contents );

      // initialise list index
      ContentsNextIndex[ ContentsLists.Count - 1 ] := 0;
    end;

    if Settings.IndexStyle in [ isFileOnly, isFull ] then
    begin
      IndexLists.Add( HelpFile.Index );
      IndexNextIndex[ IndexLists.Count - 1 ] := 0;
    end;
  end;

  DisplayedIndex.Clear;
  ProgressBar.Position := 80;

  ProfileEvent( '  Merge lists' );

  pLastEntry := NullStr;
  while true do
  begin
    pLowestEntry := NullStr;
    LowestEntryListIndex := -1;

    // Find alphabetically lowest (remaining) topic

    // first, look in contents lists
    for ListIndex := 0 to ContentsLists.Count - 1 do
    begin
      Contents := ContentsLists[ ListIndex ];
      if ContentsNextIndex[ ListIndex ] < Contents.Count then
      begin
        // list is not yet finished, get next entry
        Topic := Contents[ ContentsNextIndex[ ListIndex ] ];
        pListEntry := Topic.TitlePtr;

        if pLowestEntry^ <> '' then
          TextCompareResult := CompareText( pListEntry^, pLowestEntry^ )
        else
          TextCompareResult := -1;

        if TextCompareResult < 0 then
        begin
          // this index entry comes before the lowest one so far
          pLowestEntry := pListEntry;
          LowestEntryListIndex := ListIndex;
          LowestEntryListType := ltContents;
          LowestEntryTopic := Topic;
        end;
      end;
    end;

    // look in indices
    for ListIndex := 0 to IndexLists.Count - 1 do
    begin
      Index := IndexLists[ ListIndex ];
      if IndexNextIndex[ ListIndex ] < Index.Count then
      begin
        // list is not yet finished, get next entry
        pListEntry := Index.ValuePtrs[ IndexNextIndex[ ListIndex ] ];

        if pLowestEntry^ <> '' then
          TextCompareResult := CompareText( pListEntry^, pLowestEntry^ )
        else
          TextCompareResult := -1;

        if TextCompareResult < 0 then
        begin
          // this index entry comes before the lowest one so far
          pLowestEntry := pListEntry;
          LowestEntryListIndex := ListIndex;
          LowestEntryListType := ltIndex;
          LowestEntryTopic := TTopic( Index.Objects[ IndexNextIndex[ ListIndex ] ] );
        end;
      end;
    end;

    if LowestEntryListIndex = -1 then
      // we're out
      break;

    if ( pLowestEntry^ ) <> ( pLastEntry^ ) then
      // add, if different from last
      DisplayedIndex.AddObject( pLowestEntry^,
                                LowestEntryTopic );
    pLastEntry := pLowestEntry;

    if LowestEntryListType = ltContents then
    begin
      inc( ContentsNextIndex[ LowestEntryListIndex ] );
    end
    else
    begin
      // found in one of indices.
      // Check for subsequent indented strings
      Index := IndexLists[ LowestEntryListIndex ];

      i := IndexNextIndex[ LowestEntryListIndex ] + 1;
      while i < Index.Count do
      begin
        pListEntry := Index.ValuePtrs[ i ];
        if pListEntry^ = '' then
          break;

        if pListEntry^[ 1 ] <> ' ' then
          // not indented, stop looking
          break;

        // found one,
        Topic := Index.Objects[ i ] as TTopic;
        DisplayedIndex.AddObject( pListEntry^,
                                  Topic );
        inc( i );
      end;
      IndexNextIndex[ LowestEntryListIndex ] := i;
    end;
  end;

  ProgressBar.Position := 95;
  ProfileEvent( '  Display index (count = '
                + IntToStr( DisplayedIndex.Count )
                + ')' );

  // Now display the final index list
  IndexListBox.Items.Assign( DisplayedIndex );

  ProfileEvent( '  Tidy up' );

  IndexLists.Destroy;

  DestroyListAndObjects( ContentsLists );

  ProfileEvent( '  Done' );
end;

// Given a filename, which may or may not contain a path or extension,
// finds the actual file. This can involve searching
// the help and bookshelf paths.
Function TMainForm.FindHelpFile( FileName: string ): string;
var
  AlternativeFileName: string;
begin
  Result := '';

  AlternativeFileName := '';
  if ExtractFileExt( Filename ) = '' then
  begin
    Filename := ChangeFileExt( Filename, '.inf' );
    AlternativeFileName := ChangeFileExt( Filename, '.hlp' );
  end;

  if ExtractFilePath( FileName ) <> '' then
  begin
    // Path specified; just see if it exists
    if FileExists( Filename ) then
      Result := Filename
    else if FileExists( AlternativeFilename ) then
      Result := AlternativeFilename;

  end
  else
  begin
    // Path not specified; Search help paths
    if not SearchPath( HelpPathEnvironmentVar,
                       FileName,
                       Result ) then
    begin
      if not SearchPath( BookshelfEnvironmentVar,
                         FileName,
                         Result ) then
      begin
        // Didn't find as specified or as .inf, try .hlp
        if AlternativeFilename <> '' then
        begin
          if not SearchPath( HelpPathEnvironmentVar,
                             AlternativeFileName,
                             Result ) then
          begin
            if not SearchPath( BookshelfEnvironmentVar,
                               AlternativeFileName,
                               Result ) then
            begin
              Result := '';
            end;
          end;
        end;
      end;
    end;
  end;
end;

Procedure TMainForm.OnHelpFileLoadProgress( n, outof: integer;
                                            message: string );
var
  ThisFileSize: longint;
  ProgressOnFiles: longint;
  RelativeSizeOfFile: double;
  Filename: string;
  ProgressOnThisFile: longint;

begin
  Filename := LoadingFilenameList[ LoadingFileIndex ];

  ThisFileSize := longint( LoadingFilenameList.Objects[ LoadingFileIndex ] );

  ProgressOnFiles := round( 100 * LoadingSizeSoFar / LoadingTotalSize );
  RelativeSizeOfFile := ThisFileSize / LoadingTotalSize;

  ProgressOnThisFile := round( 100 * n / outof * RelativeSizeOfFile );

  ProgressBar.Position := ( ProgressOnFiles
                            + ProgressOnThisFile ) div 2;
  SetStatus( LoadingFileMsg
             + ExtractFileName( Filename )
              + ': '
              + message );
  ProgressBar.Show;
end;

// Load a single file.
Function TMainForm.OpenFile( const FileName: string;
                             const WindowTitle: string;
                             const SelectFirstContentsNode: boolean ): boolean;
var
  FileNames: TStringList;
begin
  FileNames := TStringList.Create;
  FileNames.Add( FileName );
  Result := OpenFiles( FileNames,
                       WindowTitle,
                       DisplayFirstTopic );
  FileNames.Destroy;
end;

// Look for any items that are actually specifiying environment
// variables, and expand them to the contents of the variables
Procedure TMainForm.TranslateIPFEnvironmentVars( Items: TStrings;
                                                 ExpandedItems: TStrings );
var
  i: longint;
  Item: string;
  EnvironmentVarValue: string;
begin
  ProfileEvent( 'Translating environment vars' );
  for i := 0 to Items.Count - 1 do
  begin
    Item := Items[ i ];

    Item := StrUnQuote( Item ); // remove single quotes
    Item := StrUnDoubleQuote( Item ); // remove double quotes

    ProfileEvent( '  Item: ' + Item );
    EnvironmentVarValue := GetEnv( Uppercase( Item ) );
    if DosError = 0 then
    begin
      // environment var exists - use it's value
      ProfileEvent( '    Translated: ' + EnvironmentVarValue );
      while EnvironmentVarValue <> '' do
      begin
         Item := ExtractNextValue( EnvironmentVarValue, '+' );
         ExpandedItems.Add( Item );
      end;
    end
    else
    begin
      // not an environment var
      ExpandedItems.Add( Item );
    end;
  end;
end;

Function TMainForm.OpenFiles( const FileNames: TStrings;
                              const WindowTitle: string;
                              const SelectFirstContentsNode: boolean ): boolean;
var
  HelpFiles: TList;
  HelpFile: THelpFile;
  FileIndex: longint;
  FileName: string;
  FullFilePath: string;
  FileSize: longint;

  FileHandlesAdjustNum: LONG;
  CurrentMaxFileHandles: ULONG;
  RequiredFileHandles: LONG;
begin
  ProfileEvent( 'OpenFiles' );

  PageHistory.Clear;
  CurrentHistoryIndex := -1;
  Cursor := crHourGlass;

  HelpFiles := TList.Create;

  LoadingFilenameList := TStringList.Create;

  TranslateIPFEnvironmentVars( FileNames, LoadingFilenameList );

  LoadingTotalSize := 0;

  ProfileEvent( 'Finding files' );

  ProgressBar.Show;

  // now find full file paths,
  // and also the total file size for progress display
  for FileIndex := 0 to LoadingFilenameList.Count - 1 do
  begin
    FileName := LoadingFilenameList[ FileIndex ];
    ProfileEvent( '  File: ' + FileName );

    // Find the help file, if possible
    FullFilePath := FindHelpFile( Filename );
    FileSize := 0;
    if FullFilePath <> '' then
    begin
      // found it
      FileSize := GetFileSize( Filename ); // see how big it is (for progress bar)
      inc( LoadingTotalSize, FileSize ); // add to total
    end
    else
    begin
      FullFilePath := FileName; // we'll complain later.
    end;
    ProfileEvent( '    Full path: ' + FullFilePath );
    ProfileEvent( '    Size: ' + IntToStr( FileSize ) );

    // save the file size
    LoadingFilenameList[ FileIndex ] := FullFilePath;
    LoadingFilenameList.Objects[ FileIndex ] := TObject( FileSize );
  end;

  // Make sure we have enough file handles

  FileHandlesAdjustNum := 0;
  DosSetRelMaxFH( FileHandlesAdjustNum, // 0 queries current
                  CurrentMaxFileHandles );

  RequiredFileHandles := Files.Count // already opened
                         + LoadingFilenameList.Count // new ones
                         + 40; // some spares.
  if CurrentMaxFileHandles < RequiredFileHandles then
  begin
    // need some more
    FileHandlesAdjustNum := RequiredFileHandles - CurrentMaxFileHandles;
    DosSetRelMaxFH( FileHandlesAdjustNum,
                    CurrentMaxFileHandles );
  end;

  // Now actually load the files
  LoadingSizeSoFar := 0;
  for FileIndex := 0 to LoadingFilenameList.Count - 1 do
  begin
    Filename := LoadingFilenameList[ FileIndex ];
    ProfileEvent( '  Loading: ' + Filename );
    try
      LoadingFileIndex := FileIndex;

      // load the file
      HelpFile := THelpFile.Create( FileName );
      inc( LoadingSizeSoFar,
           longint( LoadingFilenameList.Objects[ FileIndex ] ) );
      HelpFiles.Add( HelpFile );

    except
      on E: Exception do
      begin
        DoErrorDlg( FileOpenTitle,
                    HelpFileError
                    + FileName
                    + ': '
                    + E.Message );

        // back out of the load process
        Cursor := crDefault;
        Result := false;

        DestroyListAndObjects( HelpFiles );

        LoadingFilenameList.Destroy;
        ResetProgress;
        exit;
      end
    end;
  end;

  // Now that we have successfully loaded the new help file(s)
  // close the existing one.
  CloseFile;

  AssignList( HelpFiles, Files );

  ProgressBar.Position := 50;
  SetStatus( LoadingStatusDisplaying );

  Result := true;

  if WindowTitle = '' then
    MainTitle := THelpFile( Files[ 0 ] ).Title
  else
    MainTitle := WindowTitle;

  SetMainCaption;

  // update most-recently-used file list
  AddToMRUList( THelpFile( Files[ 0 ] ).Title,
                LoadingFilenameList );

  // recreate menu
  CreateMRUMenuItems;

  LoadingFilenameList.Destroy;
  HelpFiles.Destroy;

  // Now load the various parts of the file(s)
  // into the user interface
  ProgressBar.Position := 51;
  SetStatus( LoadingStatusNotesAndBookmarks );

  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile := Files[ FileIndex ];
    LoadNotes( HelpFile );
    LoadBookmarks( HelpFile );
  end;

  ProgressBar.Position := 55;
  SetStatus( LoadingStatusContents );

  LoadContents;

  if ContentsOutline.ChildCount = 1 then
  begin
    ProfileEvent( '  Expand first node' );
    // Contents has only one top level node... expand it
    ContentsOutline.Children[ 0 ].Expand;
  end;

  // Select first contents node if there is one
  if ContentsOutline.ChildCount > 0 then
    ContentsOutline.SelectedNode := ContentsOutline.Children[ 0 ];

  ProgressBar.Position := 75;
  SetStatus( LoadingStatusIndex );

  LoadIndex;

  SearchResultsListBox.Clear;

  Cursor:= crDefault;

  ProgressBar.Position:= 100;
  SetStatus( LoadingStatusDone );

  ResetProgress;

  EnableControls;

  UpdateNotesDisplay;

  BuildBookmarksMenu;
  UpdateBookmarksForm;

  if Settings.OpenWithExpandedContents then
    ContentsOutline.ExpandAll;

  if DisplayFirstTopic then
    DisplaySelectedContentsTopic;

  ProfileEvent( 'OpenFiles complete' );
end;

Procedure TMainForm.OpenMIOnClick (Sender: TObject);
Begin
  FileOpen;
end;

procedure TMainForm.FileOpen;
var
  Filenames: TStringList;
begin
  if Settings.UseOriginalDialogs then
  begin
    SystemOpenDialog.Filename := Settings.LastOpenDirectory;
    if SystemOpenDialog.Execute then
    begin
      Settings.LastOpenDirectory := ExtractFilePath( SystemOpenDialog.Filename );
      // note - sibyl's encapsulation doesn't allow multi-select
      if OpenFile( SystemOpenDialog.FileName, '', true ) then
      begin
        Startup_OwnHelpMode := false;
        ClearHelpManager;
      end;
    end;
  end
  else
  begin
    Filenames := TStringList.Create;
    if DoOpenMultiFileDialog( FileOpenTitle,
                              HelpFilesDesc
                              + '|*.inf;*.hlp|'
                              + AllFilesDesc
                              + '|*.*',
                              '*.hlp;*.inf',
                              Settings.LastOpenDirectory,
                              Filenames ) then
    begin
      if OpenFiles( FileNames, '', true ) then
      begin
        Startup_OwnHelpMode := false;
        ClearHelpManager;
      end;
    end;
    Filenames.Destroy;
  end;
End;

Procedure TMainForm.CloseWindows;
Begin
  DestroyListObjects( Windows );
  Windows.Clear;
end;

// Help manager mode

Procedure TMainForm.NHMDisplayIndex( Var Msg: TMessage );
begin
  DisplayIndex;
end;

Procedure TMainForm.NHMDisplayContents( Var Msg: TMessage );
begin
  DisplayContents;
end;

Procedure TMainForm.NHMTopicByResourceID( Var Msg: TMessage );
begin
  DisplayTopicByResourceID( Msg.Param1 );
end;

Procedure TMainForm.NHMTopicByPanelName( Var Msg: TMessage );
var
  pMessageMem: pchar;
begin
  pMessageMem := pchar( Msg.Param1 );
  DoSearch( StrPas( pMessageMem ) );

  MessageMemory.Free( pMessageMem );
end;

Procedure TMainForm.NHMTest( Var Msg: TMessage );
var
  ps: pstring;
begin
  ps := PString( Msg.Param1 );
  ShowMessage( 'Got test message: ' + ps^ );
  MessageMemory.Free( ps );
end;

Initialization
  RegisterClasses ([TMainForm, TSplitBar,
    TNoteBook,
    TEdit, TListBox,
    TTabSet, TRichTextView, TCoolBar2, TMainMenu, TMenuItem,
    TImageList, TPanel, TButton,
    TSystemOpenDialog, TOutline2, TCustomListBox, TPopupMenu, TSpeedButton
   , TProgressBar]);
End.
