Unit SettingsUnit;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// Defines settings (options) in a record and contains functions
// for loading and saving them to ini file.

Uses
  Graphics, Forms, Classes;

Const
  ContentsBackgroundColorIndex          = 0;
  ContentsTextColorIndex                = 1;
  ContentsLinesColorIndex               = 2;
  IndexBackgroundColorIndex             = 3;
  IndexTextColorIndex                   = 4;
  SearchBackgroundColorIndex            = 5;
  SearchTextColorIndex                  = 6;
  NotesListBackgroundColorIndex         = 7;
  NotesListTextColorIndex               = 8;
  TopicBackgroundColorIndex             = 9;
  NotesTextColorIndex                   = 10;
  SearchHighlightTextColorIndex         = 11;
  NumColorSettings                      = 12;

  clLightYellow = $ffffc0;
  clLightBlue = $e0e0ff;
  clLightCyan = $c0ffff;
  clLightGreen = $e0ffe0;

  DefaultColors: array[ 0 .. NumColorSettings - 1 ] of TColor
   = ( clLightCyan,
       clBlack,
       clBlue,
       clLightGreen,
       clBlack,
       clLightBlue,
       clBlack,
       clWhite,
       clBlack,
       clWhite,
       clGreen,
       clYellow );

  DefaultTopicFontName = 'Helv';
  DefaultTopicFontSize = 8;

  DefaultTopicFixedFontName = 'Courier';
  DefaultTopicFixedFontSize = 8;

Type
  TIndexStyle = ( isFileOnly, isAlphabetical, isFull );
  TToolbarStyle = ( tsNone, tsImages, tsText, tsImagesAndText );

  TMRUItem = class
    Title: string;
    Filenames: TStringList;
    constructor Create;
    destructor Destroy; override;
  end;

  TSettings = record
    MRUList: TList;

    LastOpenDirectory: string;
    LastSaveDirectory: string;

    StartupHelp: boolean;

    LeftPanelWidth: longint;
    LeftPanelVisible: boolean;

    FileDialogSplit: real;

    // COlours
    Colors: array[ 0..NumColorSettings - 1 ] of TColor;

    NormalFont: TFont;
    FixedFont: TFont;

    IndexStyle: TIndexStyle;

    SmoothScrolling: boolean;
    UseOriginalDialogs: boolean;
    OpenWithExpandedContents: boolean;

    ToolbarBackgroundImageFilename: string;
    ToolbarStyle: TToolbarStyle;
  end;

  Procedure WriteWindowPos( Form: TForm );
  Procedure ReadWindowPos( Form: TForm );
  Procedure LoadSettings;
  Procedure SaveSettings;

Procedure AddToMRUList( const Title: string;
                        Filenames: TStrings );

Var
  Settings: TSettings;

Implementation

Uses
  SysUtils, IniFiles, Dos, Forms,
  ACLFileUtility, ACLUtility, ACLStringUtility, ACLDialogs;

Const
  IniFileName = 'NewView.ini';
  GeneralSection = 'General';
  FontsSection = 'Fonts';
  ColoursSection = 'Colours';
  MRUSection = 'RecentFiles';
  MRUItemBaseSection = 'RecentFile';

  DefaultWidth = 620;
  DefaultHeight = 460;

  MaxMRUListEntries = 9;

constructor TMRUItem.Create;
begin
  Title := '';
  Filenames := TStringList.Create;
end;

destructor TMRUItem.Destroy;
begin
  Filenames.Destroy;
end;

Function IniFilePath: string;
Var
  Dir: string;
  Name, Ext: string;
  UserIniFile: string;
Begin
 // Put it in the same place as the application
  UserIniFile := GetApplicationDir;
  // Note: do NOT combine these two steps. Caused an obscure bug!
  FSplit( UserIniFile, Dir, Name, Ext );
  Result := AddSlash( Dir ) + IniFileName;
End;

// Note: since Sibyl SPCC 2.1.8.3
// the Ascii ini file constructor does not throw an exception
procedure GetIniFile( Var IniFile: TMyIniFile );
begin
  IniFile := TMyIniFile.Create( IniFilePath );
end;

procedure CloseIniFile( Var IniFIle: TMyIniFile );
begin
  try
    IniFile.Destroy;
  except
    // don't care if we can't save settings.
  end;
end;

Procedure LoadSettings;
Var
  IniFile: TMyIniFile;
  ColorIndex: longint;
  IndexStyleString: string;
  ToolbarStyleString: string;

  MRUItem: TMRUItem;
  MRUItemSection: string;
  MRUItemFileCount: longint;
  MRUItemFileIndex: longint;
  MRUIndex: longint;
  MRUCount: longint;
  MRUFilename: string;
  MRUFileTitle: string;
Begin
  GetIniFile( IniFile );
  with IniFile do
  begin
    IniFile.EraseSection( 'Windows' );
    with Settings do
    begin
      LastOpenDirectory := ReadString( GeneralSection, 'LastOpenDirectory', '' );
      LastSaveDirectory := ReadString( GeneralSection, 'LastSaveDirectory', '' );

      // Read split points, as units of 0.1%
      LeftPanelWidth := ReadInteger( GeneralSection, 'LeftPanelWidth', 225 );
      FileDialogSplit := ReadInteger( GeneralSection, 'FileDialogSplit', 500 ) / 1000;

      //Colours
      for ColorIndex := 0 to High( Colors ) do
        Colors[ ColorIndex ] := ReadInteger( ColoursSection,
                                             'Color' + IntToStr( ColorIndex ),
                                             DefaultColors[ ColorIndex ] );

      // Most Recently Used files list...

      MRUCount := ReadInteger( MRUSection,
                               'Count',
                               0 );

      // Old style MRU list
      for MRUIndex := 0 to MRUCount - 1 do
      begin
        MRUFilename := ReadString( MRUSection,
                                   'File' + IntToStr( MRUIndex ),
                                   '' );
        MRUFileTitle := ReadString( MRUSection,
                                    'Title' + IntToStr( MRUIndex ),
                                    '' );
        if MRUFilename <> '' then
        begin
          MRUItem := TMRUItem.Create;
          MRUItem.Title := MRUFileTitle;
          StringToList( MRUFilename,
                        MRUItem.Filenames,
                        '+' );
          MRUList.Add( MRUItem );
        end;
      end;

      // New style
      for MRUIndex := 0 to MRUCount - 1 do
      begin
        MRUItemSection := MRUItemBaseSection + IntToStr( MRUIndex );

        MRUItem := TMRUItem.Create;

        MRUItem.Title := ReadString( MRUItemSection,
                                     'Title',
                                     '' );

        MRUItemFileCount := ReadInteger( MRUItemSection,
                                         'FileCount',
                                         0 );
        for MRUItemFileIndex := 0 to MRUItemFileCount - 1 do
        begin
          MRUFilename := ReadString( MRUItemSection,
                                     'File' + IntToStr( MRUItemFileIndex ),
                                     '' );
          if MRUFilename <> '' then
          begin
            MRUItem.Filenames.Add( MRUFilename );
          end;
        end;

        if     ( MRUItem.Title <> '' )
           and ( MRUItem.Filenames.Count > 0 ) then
        begin
          // valid MRU item
          MRUList.Add( MRUItem );
        end
        else
        begin
          // not valid
          MRUItem.Destroy;
          MRUItem := nil;
        end;

      end;

      // Fonts
      NormalFont := Screen.GetFontFromPointSize(
        ReadString( FontsSection, 'NormalFont', DefaultTopicFontName ),
        ReadInteger( FontsSection, 'NormalFontSize', DefaultTopicFontSize ) );
      FixedFont := Screen.GetFontFromPointSize(
        ReadString( FontsSection, 'FixedFont', DefaultTopicFixedFontName ),
        ReadInteger( FontsSection, 'FixedFontSize', DefaultTopicFixedFontSize ) );

      // Index style
      IndexStyleString := ReadString( GeneralSection, 'IndexStyle', 'Full' );
      if IndexStyleString = 'FileOnly' then
        IndexStyle := isFileOnly
      else if IndexStyleString = 'Alphabetical' then
        IndexStyle := isAlphabetical
      else
        IndexStyle := isFull;

      StartupHelp := ReadBool( GeneralSection, 'StartupHelp', true );

      SmoothScrolling := ReadBool( GeneralSection, 'SmoothScrolling', true );
      UseOriginalDialogs := ReadBool( GeneralSection, 'UseOriginalDialogs', false );
      OpenWithExpandedContents := ReadBool( GeneralSection, 'OpenWithExpandedContents', false );

      ToolBarBackgroundImageFilename := ReadString( GeneralSection, 'ToolbarBackground', '' );
      IndexStyleString := ReadString( GeneralSection, 'IndexStyle', 'Full' );

      ToolbarStyleString := ReadString( GeneralSection, 'ToolbarStyle', 'ImagesAndText' );

      if ToolbarStyleString = 'None' then
        ToolbarStyle := tsNone
      else if ToolbarStyleString = 'Images' then
        ToolbarStyle := tsImages
      else if ToolbarStyleString = 'Text' then
        ToolbarStyle := tsText
      else
        ToolbarStyle := tsImagesAndText;
    end;
  end;
  CloseIniFile( IniFile );
End;

Procedure SaveSettings;
Var
  IniFile: TMyIniFile;
  ColorIndex: longint;
  MRUIndex: longint;
  MRUItemFileIndex: longint;
  IndexStyleString: string;
  ToolbarStyleString: string;
  MRUItem: TMRUItem;
  MRUItemSection: string;
Begin
  GetIniFile( IniFile );
  with IniFile do
  begin
    with Settings do
    begin
      WriteString( GeneralSection, 'LastOpenDirectory', LastOpenDirectory );
      WriteString( GeneralSection, 'LastSaveDirectory', LastSaveDirectory );

      WriteInteger( GeneralSection, 'LeftPanelWidth', LeftPanelWidth );
      // Write split points, as units of 0.1%
      WriteInteger( GeneralSection, 'FileDialogSplit', Round( FileDialogSplit * 1000 ) );
      // COlours
      for ColorIndex := 0 to High( Colors ) do
        WriteInteger( ColoursSection,
                      'Color' + IntToStr( ColorIndex ),
                      Colors[ ColorIndex ] );

      EraseSection( MRUSection ); // get rid of old style MRU
      WriteInteger( MRUSection,
                    'Count',
                    MRUList.Count );

      for MRUIndex := 0 to MRUList.Count - 1 do
      begin
        MRUItem := MRUList[ MRUIndex ];
        MRUItemSection := MRUItemBaseSection + IntToStr( MRUIndex );
        EraseSection( MRUItemSection );

        WriteString( MRUItemSection,
                     'Title',
                     MRUItem.Title );

        WriteInteger( MRUItemSection,
                      'FileCount',
                      MRUItem.Filenames.Count );
        for MRUItemFileIndex := 0 to MRUItem.Filenames.Count - 1 do
          WriteString( MRUItemSection,
                       'File' + IntToStr( MRUItemFileIndex ),
                       MRUItem.Filenames[ MRUItemFileIndex ] );
      end;

      // clear unused sections
      for MRUIndex := MRUList.Count to MaxMRUListEntries do
      begin
        MRUItemSection := MRUItemBaseSection + IntToStr( MRUIndex );
        EraseSection( MRUItemSection );
      end;

      WriteString( FontsSection, 'NormalFont', NormalFont.FaceName );
      WriteInteger( FontsSection, 'NormalFontSize', NormalFont.PointSize );
      WriteString( FontsSection, 'FixedFont', FixedFont.FaceName );
      WriteInteger( FontsSection, 'FixedFontSize', FixedFont.PointSize );

      case IndexStyle of
        isFileOnly:
          IndexStyleString := 'FileOnly';
        isAlphabetical:
          IndexStyleString := 'Alphabetical';
        isFull:
          IndexStyleString := 'Full';
      end;

      WriteString( GeneralSection, 'IndexStyle', IndexStyleString );

      WriteBool( GeneralSection, 'StartupHelp', StartupHelp );

      WriteBool( GeneralSection, 'SmoothScrolling', SmoothScrolling );
      WriteBool( GeneralSection, 'UseOriginalDialogs', UseOriginalDialogs );
      WriteBool( GeneralSection, 'OpenWithExpandedContents', OpenWithExpandedContents );

      WriteString( GeneralSection, 'ToolbarBackground', ToolbarBackgroundImageFilename );

      case ToolbarStyle of
        tsNone:
          ToolbarStyleString := 'None';
        tsImages:
          ToolbarStyleString := 'Images';
        tsText:
          ToolbarStyleString := 'Text';
        tsImagesAndText:
          ToolbarStyleString := 'ImagesAndText';
      end;

      WriteString( GeneralSection, 'ToolbarStyle', ToolbarStyleString );
    end;
  end;

  CloseIniFile( IniFile );

End;

Procedure AddToMRUList( const Title: string;
                        Filenames: TStrings );
var
  MRUIndex: longint;
  PreviousMRUIndex: longint;
  MRUItem: TMRUItem;
begin
  PreviousMRUIndex := -1;
  for MRUIndex := 0 to Settings.MRUList.Count - 1 do
  begin
    MRUItem := Settings.MRUList[ MRUIndex ];

    if     ( MRUItem.Title = Title )
       and ( MRUItem.Filenames.Equals( Filenames ) ) then
    begin
      // found identical entry in the list already.
      PreviousMRUIndex := MRUIndex;
      break;
    end;
  end;

  if PreviousMRUIndex > -1 then
  begin
    // is already in list, move to top of list
    MRUItem := Settings.MRUList[ PreviousMRUIndex ];
    Settings.MRUList.Delete( PreviousMRUIndex );
  end
  else
  begin
    // not yet in list. Create new
    MRUItem := TMRUItem.Create;
    MRUItem.Title := Title;
    MRUItem.Filenames.Assign( Filenames );
  end;

  Settings.MRUList.Insert( 0, MRUItem );

  while Settings.MRUList.Count > MaxMRUListEntries do
  begin
    MRUItem := Settings.MRUList[ MaxMRUListEntries ];
    Settings.MRUList.Delete( MaxMRUListEntries );
    MRUItem.Destroy;
  end;
end;

Procedure WriteWindowPos( Form: TForm );
Var
  IniFile: TMyIniFile;
Begin
  GetIniFile( IniFile );
  SaveFormSizePosition( Form, IniFile );
  CloseIniFile( IniFile );
End;

Procedure ReadWindowPos( Form: TForm );
Var
  IniFile: TMyIniFile;
Begin
  GetIniFile( IniFile );
  LoadFormSizePosition( Form, IniFile );
  CloseIniFile( IniFile );
End;

Initialization
  Settings.NormalFont:= Screen.GetFontFromPointSize( 'Helv',
                                                     8 );
  Settings.FixedFont:= Screen.GetFontFromPointSize( 'Courier',
                                                     8 );
End.
