
/* Various magic numbers.
 */

#define MAXFD				20
#define DEFAULT_PORT			26000
#define DEFAULT_RETRIES			3
#define DEFAULT_RETRY_INTERVAL		500	/* milli-seconds */
#define MAXIP				4

/* Structures for keeping information about Quake servers, server
 * rules, and players.
 */

struct player;

#define Q_UNKNOWN_TYPE 0
#define Q_SERVER 1
#define QW_SERVER 2
#define QW_MASTER 4

#define F_USERS_BACKSLASH	0x1
#define F_SEEN_BACKSLASH	0x2

struct qserver {
    char *arg;
    char *host_name;
    struct in_addr ip[MAXIP];
    int type;
    int fd;
    unsigned short port;
    int retry1;
    int retry2;
    int n_retries;
    struct timeval packet_time1;
    struct timeval packet_time2;
    int ping_total;		/* average is ping_total / n_requests */
    int n_requests;
    int n_packets;

    char *uid;			/* login info for QW master */
    char *password;
    int n_servers;
    char *error;
    char *qw_users;
    char *qw_users_cursor;
    char *qw_seen;
    char *qw_seen_cursor;
    char list_delim;

    char *server_name;
    char *address;
    char *map_name;
    int max_players;
    int num_players;
    int protocol_version;

    int next_player_info;
    int n_player_info;
    struct player *players;

    char *next_rule;
    int n_rules;
    struct rule *rules;
    int missing_rules;

    struct qserver *next;
};

struct player  {
    int number;
    char *name;
    int frags;
    int connect_time;
    int shirt_color;
    int pants_color;
    char *address;
    int ping;
    char *skin;
    struct player *next;
};

struct rule  {
    char *name;
    char *value;
    struct rule *next;
};

struct qw_user  {
    char *uid;
    char *name;
    char *efficiency;
    char *frags;
    char *rank;
    char *deaths;
    char *skill;
    int n_info;
    struct  {
	char *key;
	char *value;
    } info[20];
    char *error;
};

/* Definitions for the original Quake network protocol.
 */

#define PACKET_LEN 1600

/* Quake packet formats and magic numbers
 */
struct qheader  {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
};

struct qpacket  {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    unsigned char data[1500];
};

#define Q_FLAG1			0x80
#define Q_FLAG2			0x00
#define Q_NET_PROTOCOL_VERSION	3
#define Q_HEADER_SIZE		5

#define Q_CCREQ_CONNECT		0x01
#define Q_CCREP_ACCEPT		0x81
#define Q_CCREP_REJECT		0x82

#define Q_CCREQ_SERVER_INFO	0x02
#define Q_CCREP_SERVER_INFO	0x83

#define Q_CCREQ_PLAYER_INFO	0x03
#define Q_CCREP_PLAYER_INFO	0x84

#define Q_CCREQ_RULE_INFO	0x04
#define Q_CCREP_RULE_INFO	0x85

#define Q_DEFAULT_SV_MAXSPEED	"320"
#define Q_DEFAULT_SV_FRICTION	"4"
#define Q_DEFAULT_SV_GRAVITY	"800"
#define Q_DEFAULT_NOEXIT	"0"
#define Q_DEFAULT_TEAMPLAY	"0"
#define Q_DEFAULT_TIMELIMIT	"0"
#define Q_DEFAULT_FRAGLIMIT	"0"


/* Definitions for the QuakeWorld network protocol
 */

#define QW_DEFAULT_PORT			27500
#define QW_MASTER_DEFAULT_PORT		27000

#define QW_GET_SERVERS    'c'
#define QW_SERVERS        'd'
#define QW_GET_USERINFO   'o'
#define QW_USERINFO       'p'
#define QW_GET_SEENINFO   'u'
#define QW_SEENINFO       'v'
#define QW_NACK           'm'
#define QW_NEWLINE        '\n'
#define QW_RULE_SEPARATOR '\\'

#define QW_REQUEST_LENGTH 20
#define QW_REQUEST_PREFIX '\377', '\377', '\377', '\377'

/* Output and formatting functions
 */
 
void display_server( struct qserver *server);
void display_qwmaster( struct qserver *server);
void display_server_rules( struct qserver *server);
void display_player_info( struct qserver *server);
void display_q_player_info( struct qserver *server);
void display_qw_player_info( struct qserver *server);
 
void raw_display_server( struct qserver *server);
void raw_display_server_rules( struct qserver *server);
void raw_display_player_info( struct qserver *server);
void raw_display_q_player_info( struct qserver *server);
void raw_display_qw_player_info( struct qserver *server);
void raw_display_qw_seen_info( struct qserver *server, char *seen);
void raw_display_qw_user_info( struct qserver *server, struct qw_user *user);
 
int is_default_rule( struct rule *rule);
char *escape(unsigned char*);
char *quake_color( int color);
char *play_time( int seconds, int show_seconds);
char *ping_time( int ms);


/* Query status and packet handling functions
 */

int cleanup_qserver( struct qserver *server, int force);
 
void deal_with_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_q_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_qw_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_qwmaster_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_qw_userinfo_packet( struct qserver *server, char *pkt, int pktlen);
void deal_with_qw_seeninfo_packet( struct qserver *server, char *pkt, int pktlen);
int server_info_packet( struct qserver *server, struct qpacket *pkt,
        int datalen);
int player_info_packet( struct qserver *server, struct qpacket *pkt,
        int datalen);
int rule_info_packet( struct qserver *server, struct qpacket *pkt, int datalen);
 
int time_delta( struct timeval *later, struct timeval *past);
char * strherror( int h_err);
int connection_refused();
 
void add_file( char *filename);
int add_qserver( char *arg, int type, char *qw_users, char *qw_seen);
int add_qserver_byaddr( unsigned long ipaddr, unsigned short port, int type);
void init_qserver( struct qserver *server);
int bind_qserver( struct qserver *server);
void bind_sockets();
void send_packets();
int send_server_request_packet( struct qserver *server);
int send_qserver_request_packet( struct qserver *server);
int send_qwmaster_request_packet( struct qserver *server);
int send_player_request_packet( struct qserver *server);
int send_rule_request_packet( struct qserver *server);
int send_qwuser_request_packet( struct qserver *server);
int send_qwseen_request_packet( struct qserver *server);
void next_qw_user( struct qserver *server);
void next_qw_seen( struct qserver *server);

 
void set_fds( fd_set *fds);
void get_next_timeout( struct timeval *timeout);


