/*
 * command.c - Command Processing Module for Quick Pager
 */

#define	INCL_BASE

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>

#include "qpager.h"
#include "screen.h"

#define SLACK   2
#define NPAGE   (nLines - SLACK)
#define NHALF   (nLines / 2)

/*
 * cmdDispAgain - Redraw Current Page
 */

FILEPTR cmdDispAgain(FILEPTR fptr)
{
    if (fptr == NULL) {
        return ;
    }
    scrClear() ; dispPage(fptr) ;
    return fptr ;
}

/*
 * cmdHelp - Invoke QP to display Key Help
 */

FILEPTR cmdHelp(FILEPTR fptr)
{
    UCHAR   key[64] ;
    UCHAR   fname[256] ;
    UCHAR   cmd[256]   ;
    
    sprintf(key, "%s.key", progName) ;

    if (DosSearchPath(0x03, "HOME", key, fname, 256) != 0) {
        if (DosSearchPath(0x03, "PATH", key, fname, 256) != 0) {
	    return fptr ;
        }
    }
    
    sprintf(cmd, "%s %s", progName, fname) ;
    
    system(cmd) ;
    
    return cmdDispAgain(fptr) ;
}

/*
 * cmdFileTop - Jump to First File
 */

FILEPTR cmdFileTop(FILEPTR fptr)
{
    FILEPTR nptr ;
    
    if (fptr == NULL) {
        return ;
    }
    if ((nptr = filesTop()) == NULL) {
        return fptr ;
    }
    if (fptr == nptr) {
        return fptr ;
    }
    dispPage(nptr) ;
    return nptr ;
}

/*
 * cmdFileBot - Jump to Last File
 */

FILEPTR cmdFileBot(FILEPTR fptr)
{
    FILEPTR nptr ;
    
    if (fptr == NULL) {
        return ;
    }
    if ((nptr = filesBot()) == NULL) {
        return fptr ;
    }
    if (fptr == nptr) {
        return fptr ;
    }
    dispPage(nptr) ;
    return nptr ;
}

/*
 * cmdFilePrv - Jump to Previous File
 */

FILEPTR cmdFilePrv(FILEPTR fptr)
{
    FILEPTR nptr ;

    if (fptr == NULL) {
        return ;
    }
    if ((nptr = filesPrev(fptr)) == NULL) {
        return fptr ;
    }
    if (fptr == nptr) {
        return fptr ;
    }
    dispPage(nptr) ;
    return nptr ;
}

/*
 * cmdFileNxt - Jump to Next File
 */

FILEPTR cmdFileNxt(FILEPTR fptr)
{
    FILEPTR nptr ;

    if (fptr == NULL) {
        return ;
    }
    if ((nptr = filesNext(fptr)) == NULL) {
        return fptr ;
    }
    if (fptr == nptr) {
        return fptr ;
    }
    dispPage(nptr) ;
    return nptr ;
}

/*
 * cmdTextTop - Jump to First Page of Current File
 */

FILEPTR cmdTextTop(FILEPTR fptr)
{
    if (fptr == NULL) {
        return ;
    }
    fptr->curln = linesTop(fptr) ;
    dispPage(fptr) ;
    return fptr ;
}

/*
 * cmdTextBot - Jump to Last Page of Current File
 */


FILEPTR cmdTextBot(FILEPTR fptr)
{
    int     n ;
    LINEPTR p ;

    if (fptr == NULL) {
        return ;
    }
    if (fptr->curln == NULL) {
        fptr->curln = linesTop(fptr) ;
    }
    if (dispEof(fptr)) {
        return fptr ;
    }

    p = linesBot(fptr) ;

    for (n = 0 ; p != NULL && n < NPAGE ; ) {
        fptr->curln = p ;
        n += dispLine(0, p, FALSE) ;
	p = linesPrev(fptr, p) ;
    }
    dispPage(fptr) ;

    return fptr ;
}

/*
 * cmdPageUp - Move Page Up
 */
    
FILEPTR cmdPageUp(FILEPTR fptr)
{
    int     n ;
    LINEPTR p ;

    if (fptr == NULL) {
        return ;
    }
    if (fptr->curln == NULL) {
        fptr->curln = linesTop(fptr) ;
    }
    if (fptr->curln == linesTop(fptr)) {
        return fptr ;
    }
    
    p = linesPrev(fptr, fptr->curln) ;

    for (n = 0 ; p != NULL && n < NPAGE ; ) {
        fptr->curln = p ;
	n += dispLine(0, p, FALSE) ;
	p = linesPrev(fptr, p) ;
    }
    dispPage(fptr) ;

    return fptr ;
}

/*
 * cmdPageDn - Move Page Down
 */
 
FILEPTR cmdPageDn(FILEPTR fptr)
{
    int     n ;
    LINEPTR p ;

    if (fptr == NULL) {
        return ;
    }
    if (fptr->curln == NULL) {
        fptr->curln = linesTop(fptr) ;
    }
    if (dispEof(fptr)) {
        return fptr ;
    }

    p = linesNext(fptr, fptr->curln) ;
    
    for (n = 0 ; p != NULL && n < NPAGE ; ) {
        fptr->curln = p ;
	n += dispLine(0, p, FALSE) ;
	p = linesNext(fptr, p) ;
    }
    dispPage(fptr) ;

    return fptr ;
}

/*
 * cmdHalfUp - Move Half Page Up
 */
 
FILEPTR cmdHalfUp(FILEPTR fptr)
{
    int     n ;
    LINEPTR p ;

    if (fptr == NULL) {
        return ;
    }
    if (fptr->curln == NULL) {
        fptr->curln = linesTop(fptr) ;
    }
    if (fptr->curln == linesTop(fptr)) {
        return fptr ;
    }
    
    p = linesPrev(fptr, fptr->curln) ;

    for (n = 0 ; p != NULL && n < NHALF ; ) {
        fptr->curln = p ;
	n += dispLine(0, p, FALSE) ;
	p = linesPrev(fptr, p) ;
    }
    dispPage(fptr) ;

    return fptr ;
}

/*
 * cmdHalfDn - Move Half Page Down
 */
 
FILEPTR cmdHalfDn(FILEPTR fptr)
{
    int     n ;
    LINEPTR p ;

    if (fptr == NULL) {
        return ;
    }
    if (fptr->curln == NULL) {
        fptr->curln = linesTop(fptr) ;
    }
    if (dispEof(fptr)) {
        return fptr ;
    }

    p = linesNext(fptr, fptr->curln) ;
    
    for (n = 0 ; p != NULL && n < NPAGE ; ) {
        fptr->curln = p ;
	n += dispLine(0, p, FALSE) ;
	p = linesNext(fptr, p) ;
    }
    dispPage(fptr) ;

    return fptr ;
}

/*
 * cmdLineUp - Move One Line Up
 */

FILEPTR cmdLineUp(FILEPTR fptr)
{
    LINEPTR p ;
    
    if (fptr == NULL) {
        return ;
    }
    if (fptr->curln == NULL) {
        fptr->curln = linesTop(fptr) ;
    }
    if (fptr->curln == linesTop(fptr)) {
        return fptr ;
    }
    
    if ((p = linesPrev(fptr, fptr->curln)) != NULL) {
        fptr->curln = p ;
    }
    dispPage(fptr) ;

    return fptr ;
}

/*
 * cmdLineDn - MoveOne Line Down
 */

FILEPTR cmdLineDn(FILEPTR fptr)
{
    LINEPTR p ;
    
    if (fptr == NULL) {
        return ;
    }
    if (fptr->curln == NULL) {
        fptr->curln = linesTop(fptr) ;
    }
    if (dispEof(fptr)) {
        return fptr ;
    }
    
    if ((p = linesNext(fptr, fptr->curln)) != NULL) {
        fptr->curln = p ;
    }
    dispPage(fptr) ;

    return fptr ;
}

