#include <genesis.h>

#include "main.h"

#include "gfx.h"
#include "music.h"


#define MAN_POS_MAX             15
#define RABBIT_POS_MAX          15
#define EGG_POSX_MAX            5
#define EGG_POSY_MAX            10

#define EGG_POSY_FAIL           172

#define SCORE_DIGIT             4
#define MAX_LIFE                3
#define MAX_SCORE               4
#define MAX_EGG                 5

#define NUM_SPRITE              (MAX_LIFE + SCORE_DIGIT + MAX_EGG + 16)

#define MAN_BASE_POSX           126
#define MAN_BASE_POSY           130
#define RABBIT_BASE_POSX        128
#define RABBIT_BASE_POSY        48
#define EGG_BASE_POSX           136
#define EGG_BASE_POSY           70

#define TRUCK_POSX              572
#define TRUCK_POSY              110
#define SMOKE_POSX              540
#define SMOKE_POSY              60

#define LIFE_POSX               84
#define LIFE_POSY               40
#define BATTERY_POSX            140
#define BATTERY_POSY            32
#define SCORE_POSX              450
#define SCORE_POSY              32

#define BATTERY_BASE_COUNT      50

#define RABBIT_TIMER_POS_INIT   1000
#define RABBIT_TIMER_EGG_INIT   5000
#define UPDATE_TIMER_INIT       10000
#define SPEED_TIMER_DEFAULT     250


#define SFX_POUET               pouet_sfx
#define SFX_POUET_BAS           pouet_bas_sfx
#define SFX_DPOUET              dpouet_sfx
#define SFX_LOST                lost_sfx
#define SFX_DANLKU              danlku_sfx
#define SFX_EVENT               event_sfx
#define SFX_VICTORY             victory_sfx


#define abs(X) (((X) < 0)?(-(X)):(X))


static const u16 manOffsetX[MAN_POS_MAX] =
{
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0
};
static const u16 manOffsetY[MAN_POS_MAX] =
{
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0
};

static const u16 manEggOffsetX[MAN_POS_MAX] =
{
    -4, 2, -1, 12,
    -5, 10, 13, 18,
    -8, 0, 0, -3,
    0, 10, 0
};
static const u16 manEggOffsetY[MAN_POS_MAX] =
{
    10, 8, -7, 2,
    7, 12, -1, -6,
    22, 0, 0, 30,
    0, -4, 0
};

static const u16 rabbitOffsetX[RABBIT_POS_MAX] =
{
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0
};
static const u16 rabbitOffsetY[RABBIT_POS_MAX] =
{
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0
};
static const u16 eggOffsetX[EGG_POSX_MAX] =
{
    0, 0, 0, 0, 0
};
static const u16 eggOffsetY[EGG_POSY_MAX] =
{
    0, 0, 0, 0, 0
};



// forward
static void showLogo();

static void init();
static void handleInput();
static void joyEvent(u16 joy, u16 changed, u16 state);

static u16 getRabbitPosition();
static u16 getManPosition();
static void updateMan();
static void updateManLogic();
static void updateRabbitLogic();
static void addEgg(u16 pos);
static void updateFrame();
static void updateLogic();
static void lifeLost();
static void updateEgg(Egg* egg);
static void updateEggLogic(Egg* egg);
static void updateGui();
static u16 getPlanPos();
static u16 updateCameraSoft();
static u16 updateCameraHard();
static void playSFX(u8 *sfx, u32 len);

// sprites structure (Egg + Man + Rabbit + Truck + Life + Score + Battery)
Sprite sprites[NUM_SPRITE];

Man man;
Egg eggs[MAX_EGG];
Rabbit rabbit;
Truck truck;
Score score;
Life life;
Battery battery;

u16 camPosX;
u16 activeEgg;
s16 updateSpeed;
s16 updateTimer;
u16 sprNum;

u16 paused;
u16 ended;


int main()
{
    u16 palette[64];
    u16 ind;

    // disable interrupt when accessing VDP
    SYS_disableInts();
    // initialization
    VDP_setScreenWidth320();
    VDP_setPlanSize(128, 32);
    // start music
    //SND_startPlay_XGM(sonic_music);

    showLogo();

    // init sprites engine
    SPR_init(264);

    // set all palette to black
    VDP_setPaletteColors(0, palette_black, 64);

    // load background
    ind = TILE_USERINDEX;
    VDP_drawImageEx(APLAN, &bga_image, TILE_ATTR_FULL(PAL1, TRUE, FALSE, FALSE, ind), 0, 0, FALSE, TRUE);
    ind += bga_image.tileset->numTile;
    VDP_drawImageEx(BPLAN, &bgb_image, TILE_ATTR_FULL(PAL0, FALSE, FALSE, FALSE, ind), 0, 0, FALSE, TRUE);
    ind += bgb_image.tileset->numTile;

    VDP_setVerticalScroll(PLAN_B, -28);
    VDP_setVerticalScroll(PLAN_A, 0);

    // VDP process done, we can re enable interrupts
    SYS_enableInts();

    // init engine
    init();

    // prepare palettes
    memcpy(&palette[0], bgb_image.palette->data, 16 * 2);
    memcpy(&palette[16], bga_image.palette->data, 16 * 2);
    memcpy(&palette[32], man_sprite.palette->data, 16 * 2);

    palette[0] = 0;

    // fade in
    VDP_fadeIn(0, (3 * 16) - 1, palette, 20, TRUE);

    // wait until camera is correctly positionned
    while(TRUE)
    {
        u16 done;

        SYS_disableInts();
        done = !updateCameraSoft();
        updateCameraHard();
        SYS_enableInts();

        if (done) break;
    }

    // wait fade completion
    VDP_waitFadeCompletion();

    JOY_setEventHandler(joyEvent);

    while(TRUE)
    {
        SYS_disableInts();

        handleInput();
        updateFrame();
        SPR_update(sprites, sprNum);

        SYS_enableInts();

        VDP_waitVSync();
        updateCameraHard();
    }

    return 0;
}


static void init()
{
    u16 i;

    paused = 0;
    ended = 0;

    camPosX = 120;
    activeEgg = 0;
    updateTimer = UPDATE_TIMER_INIT;
    updateSpeed = SPEED_TIMER_DEFAULT;

    sprNum = 0;

    // movable sprites
    man.pos = 0;
    man.egg = NULL;
    man.eggType = 0;
    man.sprite = &sprites[sprNum++];
    man.eggSprite = &sprites[sprNum++];

    SPR_initSprite(man.sprite, &man_sprite, 0, 0, TILE_ATTR(PAL2, TRUE, FALSE, FALSE));
    SPR_initSprite(man.eggSprite, &egg_sprite, 0, 0, TILE_ATTR(PAL2, TRUE, FALSE, FALSE));
    // egg not visible by default
    SPR_setNeverVisible(man.eggSprite, TRUE);

    rabbit.pos = 0;
    rabbit.mov = 1;
    rabbit.waitEgg = 0;
    rabbit.sprite = &sprites[sprNum++];

    SPR_initSprite(rabbit.sprite, &rabbit_sprite, 0, 0, TILE_ATTR(PAL2, FALSE, FALSE, FALSE));

    truck.state = 0;
    truck.eggType = 0;
    truck.numEgg = 0;
    truck.sprite = &sprites[sprNum++];
//    truck.smokeSprite = &sprites[sprNum++];

    SPR_initSprite(truck.sprite, &truck_sprite, 0, 0, TILE_ATTR(PAL2, FALSE, FALSE, FALSE));
//    SPR_initSprite(truck.smokeSprite, &smoke_sprite, 0, 0, TILE_ATTR(PAL2, TRUE, FALSE, FALSE));
    // smoke not visible by default
    SPR_setNeverVisible(truck.sprite, TRUE);
//    SPR_setNeverVisible(truck.smokeSprite, TRUE);

    for(i = 0; i < MAX_EGG; i++)
    {
        Egg* egg = &eggs[i];

        egg->state = 0;
        egg->type = 0;
        egg->pos = 0;
        egg->y = 0;
        egg->sprite = &sprites[sprNum++];

        SPR_initSprite(egg->sprite, &egg_sprite, 0, 0, TILE_ATTR(PAL2, TRUE, FALSE, FALSE));
        // egg not visible by default
        SPR_setNeverVisible(egg->sprite, TRUE);
    }

    // fixed sprites
    life.value = 3;
    for(i = 0; i < MAX_LIFE; i++)
    {
        life.sprite[i] = &sprites[sprNum++];
        SPR_initSprite(life.sprite[i], &life_sprite, LIFE_POSX + (i*16), LIFE_POSY, TILE_ATTR(PAL2, TRUE, FALSE, FALSE));
    }

    score.value = 0;
    for(i = 0; i < SCORE_DIGIT; i++)
    {
        score.sprite[i] = &sprites[sprNum++];
        SPR_initSprite(score.sprite[i], &number_sprite, SCORE_POSX + (i*16), SCORE_POSY, TILE_ATTR(PAL2, TRUE, FALSE, FALSE));
    }

    battery.level = 0;
    battery.cnt = BATTERY_BASE_COUNT;
    battery.sprite = &sprites[sprNum++];
    SPR_initSprite(battery.sprite, &battery_sprite, BATTERY_POSX, BATTERY_POSY, TILE_ATTR(PAL2, TRUE, FALSE, FALSE));
}

static void updateFrame()
{
    u16 i;
    s16 pos;

    updateCameraSoft();
    updateMan();

    if (!paused && !ended)
    {
        updateTimer -= updateSpeed;
        if (updateTimer <= 0)
        {
            updateTimer += UPDATE_TIMER_INIT;
            updateSpeed += 1;

            // do cycle update
            updateLogic();
        }
    }

    // update rabbit pos
    pos = RABBIT_BASE_POSX + getRabbitPosition() + (rabbitOffsetX[rabbit.pos] - getPlanPos());
    if ((pos < 0) || (pos > 320))
        SPR_setNeverVisible(rabbit.sprite, TRUE);
    else
    {
        SPR_setAlwaysVisible(rabbit.sprite, TRUE);
        SPR_setPosition(rabbit.sprite, pos, RABBIT_BASE_POSY + rabbitOffsetY[rabbit.pos]);
    }

    if (!paused && !ended)
    {
        // update egg state
        for(i = 0; i < MAX_EGG; i++)
        {
            Egg* egg = &eggs[i];

            if (egg->state) updateEgg(egg);
        }
    }

    // update eggs pos
    for(i = 0; i < MAX_EGG; i++)
    {
        Egg* egg = &eggs[i];

        if (egg->state)
        {
            Sprite* sprite = egg->sprite;

            pos = EGG_BASE_POSX + (egg->pos * 27) - getPlanPos();

            if ((pos < 0) || (pos > 320))
                SPR_setNeverVisible(sprite, TRUE);
            else
            {
                SPR_setAlwaysVisible(sprite, TRUE);

                // egg missed
                if (egg->y == EGG_POSY_MAX)
                    SPR_setPosition(sprite, pos, EGG_POSY_FAIL);
                else
                    SPR_setPosition(sprite, pos, EGG_BASE_POSY + (egg->y * 10));
            }
        }
    }

    // update truck pos
    pos = TRUCK_POSX - getPlanPos();
    if ((pos < 0) || (pos > 320))
        SPR_setNeverVisible(truck.sprite, TRUE);
    else
    {
        SPR_setAlwaysVisible(truck.sprite, TRUE);
        SPR_setPosition(truck.sprite, pos, TRUCK_POSY);
    }

    updateGui();
}

static void updateLogic()
{
    static u16 snd;
    u16 i;

    if (snd & 1)
        SND_startPlay_2ADPCM(tempo1_sfx, sizeof(tempo1_sfx), SOUND_PCM_CH1, FALSE);
    else
        SND_startPlay_2ADPCM(tempo2_sfx, sizeof(tempo1_sfx), SOUND_PCM_CH1, FALSE);
    snd++;

    // update rabbit state
    updateRabbitLogic();

    // update egg state
    for(i = 0; i < MAX_EGG; i++)
    {
        Egg* egg = &eggs[i];

        if (egg->state) updateEggLogic(egg);
    }

    // update battery
    if (battery.cnt-- < 0)
    {
        if (battery.level < 7)
            battery.level++;
        else ended = TRUE;

        battery.cnt = BATTERY_BASE_COUNT;
    }

    // inter level ?
    if (battery.level & 1)
    {
        if (battery.cnt & 1)
            SPR_setFrame(battery.sprite, (battery.level >> 1));
        else
            SPR_setFrame(battery.sprite, (battery.level >> 1) + 1);
    }
    else
        SPR_setFrame(battery.sprite, (battery.level >> 1));
}

static u16 getRabbitPosition()
{
    return rabbit.pos * 27;
}

static u16 getManPosition()
{
    return man.pos * 27;
}

static void updateMan()
{
    // update man logic
    updateManLogic();

    // update man sprite
    SPR_setFrame(man.sprite, man.pos);
    SPR_setPosition(man.sprite, MAN_BASE_POSX + getManPosition() + (manOffsetX[man.pos] - getPlanPos()), MAN_BASE_POSY + manOffsetY[man.pos]);
    // update man egg position
    if (man.egg)
        SPR_setPosition(man.eggSprite, MAN_BASE_POSX + getManPosition() + (manEggOffsetX[man.pos] - getPlanPos()), MAN_BASE_POSY + manEggOffsetY[man.pos]);
}

static void updateManLogic()
{
    Egg* egg = man.egg;

    // update man sprite
    if (egg)
    {
        u8 ok = FALSE;

        // trash pos and dark egg --> ok
        if ((man.pos == 0) && man.eggType)
        {
            ok = TRUE;
            // play sfx
            SND_startPlay_2ADPCM(egg_trash_sfx, sizeof(egg_trash_sfx), SOUND_PCM_CH1, FALSE);
        }
        // truck pos and white egg --> ok
        if ((man.pos >= MAN_POS_MAX) && (man.eggType == 0))
        {
            ok = TRUE;
            // play sfx
            SND_startPlay_2ADPCM(egg_truck_sfx, sizeof(egg_truck_sfx), SOUND_PCM_CH1, FALSE);
        }

        if (ok)
        {
            // remove man egg
            man.egg = NULL;
            SPR_setNeverVisible(man.eggSprite, TRUE);
            // increment score
            score.value++;
        }
    }
}

static void updateRabbitLogic()
{
    u16 action = random();

    // go left if possible
    if (rabbit.pos == 0) rabbit.mov = 1;
    else if (rabbit.pos >= (RABBIT_POS_MAX - 1)) rabbit.mov = -1;
    else if ((action & 0x0007) == 0x0007)
        rabbit.mov = -rabbit.mov;

    // change rabbit position
    rabbit.pos += rabbit.mov;

    // change animation frame
    SPR_setFrame(rabbit.sprite, random() % 6);

    if (rabbit.waitEgg > 0)
        rabbit.waitEgg--;
    else
    {
        // test if rabbit position is ok
        if ((rabbit.pos > 0) && (rabbit.pos < (RABBIT_POS_MAX - 1)))
        {
            // small random to throw an egg
            if ((action & 0x0001) == 0x0001)
            {
                addEgg(rabbit.pos);
                rabbit.waitEgg = 4;
            }
        }
    }
}

static void updateEgg(Egg* egg)
{
    // egg raising floor
    if ((egg->y >= (EGG_POSY_MAX - 4)) && (egg->y < EGG_POSY_MAX))
    {
        // man is on same pos with free hands ?
        if ((egg->pos == man.pos) && (man.egg == NULL))
        {
            // egg catched
            man.egg = egg;
            man.eggType = egg->type;
            // display man egg sprite
            SPR_setFrame(man.eggSprite, egg->type);
            SPR_setPosition(man.eggSprite, MAN_BASE_POSX + getManPosition() + (manEggOffsetX[man.pos] - getPlanPos()), MAN_BASE_POSY + manEggOffsetY[man.pos]);
            SPR_setAlwaysVisible(man.eggSprite, TRUE);

            // play sfx
            SND_startPlay_2ADPCM(egg_catched_sfx, sizeof(egg_catched_sfx), SOUND_PCM_CH1, FALSE);

            // egg life done
            egg->state = 0;
            SPR_setNeverVisible(egg->sprite, TRUE);

            // done
            return;
        }
    }
}

static void updateEggLogic(Egg* egg)
{
    egg->y++;

    // egg missed ?
    if (egg->y == EGG_POSY_MAX)
    {
        // set missed frame
        SPR_setFrame(egg->sprite, egg->type + 2);
        // play sfx for lost now
        SND_startPlay_2ADPCM(miss_sfx, sizeof(miss_sfx), SOUND_PCM_CH2, FALSE);
    }
    else if (egg->y > EGG_POSY_MAX)
    {
        // life lost
        lifeLost();

        // egg life done
        egg->state = 0;
        SPR_setNeverVisible(egg->sprite, TRUE);
    }
}

static void addEgg(u16 pos)
{
    u16 i;

    for(i = 0; i < MAX_EGG; i++)
    {
        Egg* egg = &eggs[i];

        // not active ?
        if (!egg->state)
        {
            egg->state = 1;
            egg->type = random() & 1;
            egg->pos = pos;
            egg->y = -1;
            SPR_setFrame(egg->sprite, egg->type);
            break;
        }
    }
}

static void lifeLost()
{
    life.value--;
    if (life.value <= 0) ended = TRUE;
}

static void updateGui()
{
    u16 i;

    // update life
    for(i = 0; i < MAX_LIFE; i++)
    {
        Sprite* sprite = life.sprite[i];

        if (life.value >= (i + 1))
            SPR_setAlwaysVisible(sprite, TRUE);
        else
            SPR_setNeverVisible(sprite, TRUE);

        SPR_setPosition(sprite, (LIFE_POSX + (i * 16)) - camPosX, LIFE_POSY);
    }

    // update score
    SPR_setFrame(score.sprite[0], (score.value / 1000) % 10);
    SPR_setFrame(score.sprite[1], (score.value / 100) % 10);
    SPR_setFrame(score.sprite[2], (score.value / 10) % 10);
    SPR_setFrame(score.sprite[3], score.value % 10);

    for(i = 0; i < SCORE_DIGIT; i++)
    {
        Sprite* sprite = score.sprite[i];

        SPR_setPosition(sprite, (SCORE_POSX + (i * 16)) - camPosX, SCORE_POSY);
     }

    // update battery
    SPR_setPosition(battery.sprite, BATTERY_POSX - camPosX, BATTERY_POSY);
}

static u16 getPlanPos()
{
    return camPosX + (camPosX >> 2);
}

static u16 updateCameraSoft()
{
    u16 wantedX;
    u16 manPos;

    manPos = getManPosition();
    wantedX = manPos - (manPos >> 2);

    if (camPosX != wantedX)
    {
        s16 delta = wantedX - camPosX;

        if (abs(delta) < 8)
        {
            if (delta < 0) camPosX--;
            else camPosX++;
        }
        else camPosX += delta >> 3;

        return TRUE;
    }

    return FALSE;
}

static u16 updateCameraHard()
{
    VDP_setHorizontalScroll(PLAN_B, 20 - getPlanPos());
    VDP_setHorizontalScroll(PLAN_A, -camPosX);
}


static void handleInput()
{
    u16 value = JOY_readJoypad(JOY_1);

//    if (value & BUTTON_UP) yorder = -1;
//    else if (value & BUTTON_DOWN) yorder = +1;
//    else yorder = 0;
//
//    if (value & BUTTON_LEFT) xorder = -1;
//    else if (value & BUTTON_RIGHT) xorder = +1;
//    else xorder = 0;
}

static void joyEvent(u16 joy, u16 changed, u16 state)
{
    // START button state changed
    if (changed & BUTTON_START)
    {

    }

    if (changed & state & BUTTON_LEFT)
    {
        if (man.pos > 0)
        {
            man.pos--;
            SND_startPlay_2ADPCM(move_sfx, sizeof(move_sfx), SOUND_PCM_CH1, FALSE);
        }
    }
    else if (changed & state & BUTTON_RIGHT)
    {
        if (man.pos < MAN_POS_MAX)
        {
            man.pos++;
            SND_startPlay_2ADPCM(move_sfx, sizeof(move_sfx), SOUND_PCM_CH1, FALSE);
        }
    }

//    if (changed & state & BUTTON_C)
//    {
//        switch(random() % 7)
//        {
//            case 0:
//                playSFX((u8*) SFX_DANLKU, sizeof(SFX_DANLKU));
//                break;
//
//            case 1:
//                playSFX((u8*) SFX_DPOUET, sizeof(SFX_DPOUET));
//                break;
//
//            case 2:
//                playSFX((u8*) SFX_EVENT, sizeof(SFX_EVENT));
//                break;
//
//            case 3:
//                playSFX((u8*) SFX_LOST, sizeof(SFX_LOST));
//                break;
//
//            case 4:
//                playSFX((u8*) SFX_POUET, sizeof(SFX_POUET));
//                break;
//
//            case 5:
//                playSFX((u8*) SFX_POUET_BAS, sizeof(SFX_POUET_BAS));
//                break;
//
//            case 6:
//                playSFX((u8*) SFX_VICTORY, sizeof(SFX_VICTORY));
//                break;
//
//        }
//    }
}


static void showLogo()
{
    u16 palette[16];

    // set all palette to black
    VDP_setPaletteColors(0, &palette_black, 64);

    VDP_drawImageEx(VDP_PLAN_A, &logo_image, TILE_ATTR_FULL(PAL0, FALSE, FALSE, FALSE, TILE_USERINDEX), 6, 9, FALSE, FALSE);

    VDP_drawText("PRESS START", 14, 18);

    memcpy(palette, logo_image.palette->data, 16 * 2);
    palette[0] = 0x0EEE;

    VDP_fadeIn(0, 15, palette, 30, FALSE);

    SND_startPlay_2ADPCM(logo_start_sfx, sizeof(logo_start_sfx), SOUND_PCM_CH1, FALSE);
    waitMs(1000);

    // wait Start pressed
    JOY_waitPress(JOY_1, BUTTON_START);

    VDP_fadeOut(0, 15, 30, FALSE);

    VDP_clearPlan(VDP_PLAN_A, FALSE);
}


static void playSFX(u8 *sfx, u32 len)
{
    SND_startPlay_4PCM_ENV(sfx, len, SOUND_PCM_CH1, FALSE);
}
