let vert_shader_text = `attribute vec2 coords; void main(void){ gl_Position = vec4(coords, 0.0, 1.0); }`;
let frag_shader_text = `

#ifdef GL_ES
precision mediump float;
#endif

#define eps 0.01
#define numsteps 48.0          // Typical number of marching steps (must match abbrevsteps in demo.html)
#define beatsteps 18.0         // Minimum steps to march when a beat hits (for visual effect)
#define light vec3(-2,3,1)
#define ambient_light 0.4
#define PI 3.14159265359
#define INVRT3 0.57735026919   // 1/sqrt(3)
#define beat 0.3409090909      // 60.0/176.0 (the BPM)
#define sum(x) dot(x,vec3(1))

// Follow Shadertoy conventions
uniform float abbrevsteps;
uniform int text[161];
uniform float iTime;
uniform vec2 iResolution;
uniform float beatAmt;
uniform float E0, E1, E2, E3, E4, E5, E6, G, T0, T_ripple, T5;  // Effect timing

bool in_text(int idx){
    for(int i=0;i<161;i++){
        if(text[i]==idx){ return true; }
    }
    return false;
}

// Adapted from https://en.wikipedia.org/wiki/HSL_and_HSV
// h in [0,360); s,ell in [0,1]
vec3 hsl_to_rgb(float h, float s, float ell){
    h /= 60.0;
    float c = s * (1.0 - abs(2.0*ell - 1.0));  // chroma
    float x = c * (1.0 - abs(mod(h,2.0) - 1.0));
    float m = ell - c/2.0;
    // Want to return:
    // (c,x,0) if h in [0,1)
    // (x,c,0) if h in [1,2)
    // (0,c,x) if h in [2,3)
    // (0,x,c) if h in [3,4)
    // (x,0,c) if h in [4,5)
    // (c,0,x) if h in [5,6)
    // ... + (m,m,m)
    return vec3(c*floor(mod(h+5.0,6.0)/4.0) + x*floor(mod(h+1.0,3.0)/2.0),
                c*floor(mod(h+3.0,6.0)/4.0) + x*floor(mod(h+2.0,3.0)/2.0),
                c*floor(mod(h+1.0,6.0)/4.0) + x*floor(mod(h,3.0)/2.0)) + m;
}

// Force things greater than 0.5 toward 1 and less than 0.5 toward 0; 'darken' adjusts the threshold away from 0.5.
float sigmoid(float x, float alpha, float darken) {
    return 1.0 / (1.0 + exp(-alpha*(x-0.5-darken)));
}

// Return 1 at x==0 and diminish away from 0, more quickly the larger "steep" is.
float bump(float x, float steep){
    return 1.0/(1.0+abs(x*x*steep));
}

// https://bmtechjournal.wordpress.com/2020/05/27/super-fast-quadratic-sinusoid-approximation/
float fsin(float x){
    x = mod(x/PI + 1.0, 2.0) - 1.0;
    return 4.0 * x * (1.0 - abs(x));
}
float fcos(float x){ return fsin(x + PI/2.0); }
vec3 fsin(vec3 a){ return vec3(fsin(a.x), fsin(a.y), fsin(a.z)); }
vec3 fcos(vec3 a){ return vec3(fcos(a.x), fcos(a.y), fcos(a.z)); }

float min3(float a, float b, float c){ return min(min(a,b),c); }
float max3(float a, float b, float c){ return max(max(a,b),c); }
float min3(vec3 a){ return min(min(a.x,a.y),a.z); }
float max3(vec3 a){ return max(max(a.x,a.y),a.z); }
float mid3(float a, float b, float c){ return a + b + c - min(min(a,b),c) - max(max(a,b),c); }
float mid3(vec3 a){ return sum(a) - min(min(a.x,a.y),a.z) - max(max(a.x,a.y),a.z); }

// https://mathworld.wolfram.com/RodriguesRotationFormula.html
vec3 fixed_rot(vec3 p, float theta){  // rotate about (1,1,1)
    float c = cos(theta);
    float A = INVRT3*sin(theta);
    float B = (1.0 - c)/3.0;
    vec3 q = vec3(c+B,B-A,A+B);
    return vec3(dot(p,q.xyz), dot(p,q.zxy), dot(p,q.yzx));
}

float ripple_amt(vec3 p){
    if(T_ripple <= 0.0){ return 0.0; }
    float ripple = fsin(dot(p,vec3(3.93,3.712,4.1)) + 0.05*iTime) + 0.25*fsin(dot(p,vec3(8.11,8.05,8.59)) + 1.5 + 4.67*iTime);
    return T_ripple*sigmoid(ripple, 5.0, 1.0-T_ripple);
}

float glitch_amt(vec3 p){
    return G*0.006*(fract(16.0*p.x) + fract(16.0*p.y) + 6.0*fract(3.0*p.x) + 6.0*fract(2.0*(p.y - iTime)));
}

vec2 F_vanilla(vec3 p){
    float ripple = ripple_amt(p);
    float step_size = -min3(fsin(p + T0 + vec3(0,2,4)*PI/3.0)) - 0.35*beatAmt + 0.07*ripple + glitch_amt(p);
    return vec2(step_size, ripple);  // Return ripple amount as a second argument to be used for coloration
}

vec2 F0(vec3 p){  // No ripples or glitches in intro
    float step_size = -min3(fsin(p + T0 + vec3(0,2,4)*PI/3.0)) - 0.35*beatAmt;
    step_size += (T0-1.0)*(0.3 + 0.06*(fsin(1.5*p.x + 3.5*p.z) + fsin(dot(p,vec3(1,1,8)) + iTime*3.0)));

    // Eurion constellation
    float note = 0.0;
    float d_sq = (p.x + 3.6456)*(p.x + 3.6456) + (p.z - 2.2193)*(p.z - 2.2193);
    d_sq = min(d_sq, (p.x + 3.5298)*(p.x + 3.5298) + (p.z - 2.2640)*(p.z - 2.2640));
    d_sq = min(d_sq, (p.x + 3.5053)*(p.x + 3.5053) + (p.z - 2.1456)*(p.z - 2.1456));
    d_sq = min(d_sq, (p.x + 3.3474)*(p.x + 3.3474) + (p.z - 2.3044)*(p.z - 2.3044));
    d_sq = min(d_sq, (p.x + 3.5096)*(p.x + 3.5096) + (p.z - 2.4632)*(p.z - 2.4632));
    float d2 = pow( pow(abs(p.x + 3.5), 8.0) + pow(abs(p.z - 2.3), 8.0), 0.125);
    if(p.y < 2.0 && ((mod(iTime,0.6) < 0.54 && mod(iTime,0.44) > 0.04) || mod(p.x - iTime/3.0,0.2) < 0.03 || mod(p.x - iTime/4.0,0.3) < 0.05)){
        if((0.23 < d2 && d2 < 0.24) || (0.0001 < d_sq && d_sq < 0.0008)){
            note = -0.1;
        }
        else if(d2 < 0.22 && d_sq > 0.002 && mod(p.x + p.z,0.03) > 0.015){
            note = -100.0;
        }

        // Inércia
        if(-3.74 < p.x && p.x < -3.26 &&  1.95625 < p.z && p.z < 2.02375){
            int ix = int(64.0*(p.x + 3.74)/0.48);
            int iy = int(9.0*(2.02375 - p.z)/0.0675);
            if(in_text(ix + 64*iy)){
                note = -0.1;
            }
        }
    }

    return vec2(step_size, note);
}

vec2 F1(vec3 p){  // No glitches in effect 1
    vec3 a = fsin(p + T0 + vec3(0,2,4)*PI/3.0);
    float ripple = ripple_amt(p);
    float step_size = -min3(a) - 0.35*beatAmt + 0.07*ripple;
    step_size += E1 * (0.5 + fcos((112.0 - iTime/beat)*a.x/48.0 + a.y + a.z - min3(a)) - 0.1*length(cross(a,p)));
    step_size += 0.5*E1 * abs(step_size);
    return vec2(step_size, ripple);
}

vec2 F2(vec3 p){
    float ripple = ripple_amt(p);
    float step_size = -min3(fsin(p + T0 + vec3(0,2,4)*PI/3.0)) - 0.35*beatAmt + 0.07*ripple + glitch_amt(p);
    step_size += E2 * (1.2 + min(0.0,1.2*sum(fsin(p-p.yzx))));
    step_size -= 0.5*E2 * abs(step_size);
    return vec2(step_size, ripple);
}

vec2 F3(vec3 p){
    vec3 a = fsin(p + T0 + vec3(0,2,4)*PI/3.0);
    float ripple = ripple_amt(p);
    float step_size = -min3(a) - 0.35*beatAmt + 0.07*ripple + glitch_amt(p);
    step_size += E3 * (1.5 - sum(fcos(p-p.yzx)) - 0.25*beatAmt*max3(a)*abs(min3(p)-1.0));
    step_size -= 0.5*E3 * abs(step_size);
    return vec2(step_size, ripple);
}

vec2 F4(vec3 p){  // No glitches in effect 4
    vec3 a = fsin(p + T0 + vec3(0,2,4)*PI/3.0);
    float ripple = ripple_amt(p);
    float step_size = -min3(a) - 0.35*beatAmt + 0.07*ripple;
    step_size += E4 * (-0.75 + 1.0*max3(a));
    return vec2(step_size, ripple);
}

vec2 F5(vec3 p){  // No glitches in effect 5
    float ripple = ripple_amt(p);
    float step_size = -min3(fsin(p + T0 + vec3(0,2,4)*PI/3.0)) - 0.35*beatAmt + 0.07*ripple;
    step_size += E5 * (1.0 - 0.25*fcos(iTime) + min(0.0,1.2*sum(fcos(2.0*(p-p.yzx)))));
    return vec2(step_size, ripple);
}

vec2 F6(vec3 p){  // No glitches in effect 6
    vec3 a = fsin(p + T0 + vec3(0,2,4)*PI/3.0);
    vec3 b = E6 * fcos(p - vec3(4.0*T0,0,0) + vec3(0,2,4)*PI/3.0) + smoothstep(0.5,0.6,E6);
    float ripple = ripple_amt(p);
    float step_size = -min(min3(a),max3(b)) - 0.35*beatAmt + 0.07*ripple;
    return vec2(step_size, ripple);
}

vec2 ray_march(vec3 p, vec3 ray, float t5, float t6){
    // Spin faster during effect 5 (beats 200-232) in a way that avoids "backspin" upon stopping
    p = fixed_rot(p, T0/3.0 + T5*(t5-t6));
    ray = fixed_rot(ray, T0/3.0 + T5*(t5-t6));

    vec2 dist_note = vec2(0);
    float realsteps = mix(abbrevsteps, beatsteps, beatAmt);
    realsteps = mix(realsteps, 4.0, smoothstep(0.0, 4.0*beat, iTime-248.0*beat));

    if(T0 < 1.0){
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F0(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    } else if(E1 > 0.0){
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F1(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    } else if(E2 > 0.0){
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F2(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    } else if(E3 > 0.0){
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F3(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    } else if(E4 > 0.0){
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F4(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    } else if(E5 > 0.0){
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F5(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    } else if(E6 > 0.0){
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F6(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    } else {
        for(float i=0.5; i<numsteps; i+=1.0){
            if(i > realsteps){ break; }
            dist_note += F_vanilla(p + dist_note.x*ray) * vec2(1, i/realsteps > 0.67 ? 1 : 0);
        }
    }

    return dist_note;
}

void mainImage(out vec4 fragColor, in vec2 fragCoord){
    vec2 uv = fragCoord/iResolution.xy;
    vec3 col = vec3(0);
    vec3 norm_light = normalize(light);

    float t_end = max(0.0, 0.15*(iTime-248.0*beat));
    // Parameters for color cycling during interval E5, used by ray_march()
    float t5 = iTime - 200.0*beat;
    float t6 = max(0.0, iTime-232.0*beat);

    // Set camera parameters
    vec3 cam_pos  = vec3((-PI/2.0)*E0, (3.0*PI/4.0)*E0, 0.0);
    vec3 cam_look = vec3(1.0-E0, 1.0-E0, 1);
	vec3 cam_up   = vec3(0, 1, 0);

    cam_look = normalize(cam_look);
    cam_up = normalize(cross(cam_look, cross(cam_up, cam_look)));
    vec3 cam_right = cross(cam_up, cam_look);

    if(E1 > 0.0){
        float prop = (iTime-64.0*beat) / (48.0*beat);
        cam_pos += E1 * fsin(4.0*PI*prop) * (3.0*cam_look - 0.5*cam_up + 0.8*cam_right);
    }
    else if(E2 > 0.0){
        float prop = (iTime-112.0*beat) / (16.0*beat);
        cam_pos += E2 * fsin((PI)*prop*prop) * (1.2*cam_up + 6.0*cam_right);
    }
    else if(E5 > 0.0){
        float prop = (iTime-200.0*beat)/(32.0*beat);
        cam_pos += E5 * (fsin(2.0*PI*prop)*(42.0*cam_look + 0.6*cam_right) - fcos(2.0*PI*prop)*0.6*cam_up);
    }
    else if(iTime > 240.0*beat){
        float T6 = smoothstep(0.0, 2.0*beat, iTime-240.0*beat);
        cam_pos += T6 * (iTime-240.0*beat) * (1.0*cam_look + 1.6*cam_right);
    }

    float fov = 120.0*(1.0-E2-E5) + 90.0*E2 + 20.0*E5;
    float aspect_ratio = iResolution.x / iResolution.y;
    float cam_w = tan(fov * PI/360.0) * 2.0;
	float cam_h = cam_w / aspect_ratio;

    float glitchx = 0.0;

    if(E0 > 0.0){
        cam_pos.xy += E0 * vec2(0.1*fcos(0.5*iTime), 0.2*fsin(1.1*iTime)); // camera bob

        float mt = mod(iTime/3.0, 1.0);
        glitchx =  bump(0.5 - mod(uv.y - uv.x/12.0 - mod(iTime/12.0,1.0), 1.0), 1e6);
        glitchx += bump(0.5 - mod(uv.y - uv.x/12.0 - mt, 1.0), 1e8);
        glitchx += bump(0.5 - mod(uv.y - uv.x/12.0 - mt - 0.2, 1.0), 1e8);
        glitchx += bump(0.5 - mod(uv.y - uv.x/12.0 - mt - 0.33, 1.0), 1e8);
        glitchx += bump(0.5 - mod(uv.y - uv.x/12.0 - mt - 0.65, 1.0), 1e8);
        glitchx *= E0;
        glitchx *= 14.0*fsin(iTime/12.0) + 8.0*fsin(1.1 + iTime/12.0);
    }

    // Get ray
    float right_amt = -cam_w/2.0 + cam_w*(uv.x + glitchx);
    float up_amt = -cam_h/2.0 + cam_h*(uv.y);
    vec3 ray = normalize(cam_look + cam_right*right_amt + cam_up*up_amt);
    vec3 cur_right = cam_right;
    vec3 cur_up = cam_up;
    vec3 p = cam_pos;

    float contribute_amt = 1.0;  // Diminishes upon reflection
    vec3 fade_col = (1.0-E0-E1-E5) * vec3(1,.99,.97)
                    + E0 * vec3(.95)
                    + E1 * vec3(.9,.9,1)
                    + E5 * (1. - 2.*dot(uv-.5,uv-.5)) * vec3(.94+.06*fsin(iTime), 1, .94+.06*fcos(iTime));

    for(int i=0;i<2;i++){
        vec2 result =     ray_march(p,                 ray, t5, t6);
        float result_dx = ray_march(p + eps*cur_right, ray, t5, t6).x;
        float result_dy = ray_march(p + eps*cur_up,    ray, t5, t6).x;

        vec3 p0 = p + result.x*ray;
        vec3 p1 = p + eps*cur_right + result_dx*ray;
        vec3 p2 = p + eps*cur_up    + result_dy*ray;
        vec3 norm = normalize(cross(p2-p0, p1-p0));

        float ell = max(dot(norm, norm_light), 0.0);
        float s = (result.y < 0.5 ? 0.05 : 1.0) * (0.4 + 0.4*ell);
        ell = mix(ell, 1.0, ambient_light) + t_end;  // Fade out at the end
        ell = clamp(sigmoid(ell,7.0,0.1), 0.0, 1.0);
        float h = 80.0 + dot(norm, vec3(20,-40,10)) - 3.15*result.x - 100.0*dot(norm, cam_look) + T5*(t5-t6)*35.0;

        // Tweaks for Eurion constellation
        if(result.y < -1e2){ h = 0.0; s += 0.5; ell *= 1.0; }
        else if(result.y < 0.0){ ell *= 0.1; }

        float fade_limit = 25.0 * (1.0 - 0.2*E1 - 0.2*E2 - 2.0*E3 + 0.8*E5);
        float fade_amt = smoothstep(0.5*fade_limit, fade_limit, result.x);  // Fade in the distance

        col += contribute_amt * mix(hsl_to_rgb(h,s,ell), fade_col, fade_amt);  // Accumulate color in col[]

        // Set up reflection
        ray = normalize(ray - 2.0*dot(ray,norm)*norm);
        p = p0 + eps*ray;  // Move away from the surface a little
        cur_up = normalize(p2-p0);
        cur_right = normalize(p1-p0);
        contribute_amt *= 0.4 * (1.0-fade_amt) * (1.0-s);
    }

    // Major palette adjustment
    float gr = max(0.0, col.g-col.r);
    float gb = max(0.0, col.g-col.b);
    col += vec3(0.75*gb, -2.0*min(gr,gb), 0.5*gr);

    fragColor = vec4(col,1.0);
}

void main() {
    mainImage(gl_FragColor, gl_FragCoord.xy);
}

`;