/*
 * WHAT: Transfer Activities; for DayDream BBS/Linux.
 * Coding by: bObO/mYSTiC (east@canit.se)
 *
 * You may always reach me at east@canit.se, or on IRCNET (nick: east).
 *
 * As usual, I'm always open for suggestions. Feel free to mail me at 
 * any time.
 *
 * Adios!
 *  - bObO/mYSTiC (east bbs)
 */

#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <dd.h>

#include <DDDIUserbase.h>  /* support for DDDIUserbase class */
#include "what.h"

/* initialize our door object */
what theDoor;

what::what()
{
  /* initialize data */
  mTotalUlBytes = mTotalUlFiles = 0;
  mTotalDlBytes = mTotalDlFiles = 0;

  /* default options/settings for what */
  mShowNode = 0;              /* 0 = show all nodes */
  mFirstCnt = mLastCnt = 0;   /* 0 = everything */

  mShowUploadsOnly = mShowDownloadsOnly = false;
  mShowExtended    = mShowNonstop       = false;
}

what::~what()
{
}

/* we start here */
void what::DDDIMain(int argc, String* argv)
{
  /* total number of activities (upload/download) going on. */
  int totactivities = 0;

  /* daydream temporary directory; always append ending '/' */
  const String ddtmp = "/tmp/dd/";

  /* nodeinfo filename (search pattern) for daydream; located in ddtmp[].
     filename: nodeinfo<node>.dat */
  const String ddnodeinfostr = "nodeinfo";

  /* upload/download search-strings in activity field; a empty string ends
     the list. if a user activity field matches on of these strings, "WHAT" 
     suspects that a transfer is going on and will check the logs to make sure */
  const String transferstr[] = { "upload", "download", "ul:", "dl:", "" };

  /* upload/download string in activity field that indicates the current
     file being transfered i.e. ul: filename */
  const String dduploadingstr = "ul:";
  const String dddownloadingstr = "dl:";

  /* change node status */
  ChangeNodeStatus("WHAT");

  /* write a log entry */
  WriteLog("WHAT: Transfer Activities\n");

  /* if the first argument is -h, the user wants help */
  if (argc >= 1 && argv[0] == "-h")
    {
      PrintHelp();
      return;
    }

  /* analyze arguments supplied */
  AnalyzeArguments(argc, argv);

  /* print header */
  SendString("\n");
  PrintHeader();

  /* scan dd-tmp directory for nodeinfo-files so we can see who's online 
     and what they're doing. i.e. /tmp/dd/nodeinfo%d.dat */
  DIR* mydir;
  struct dirent* myfile;

  /* open ddtmp directory */
  mydir = opendir(ddtmp);
  if (mydir == NULL)
    {
      SendString("error: opendir() on %s failed.\n", (char *)ddtmp);
      return;
    }

  struct DayDream_NodeInfo mynodeinfo;
  String tempStr;
  FILE* fp;
  int node;

  /* loop through the directory */
  while ((myfile = readdir(mydir)) != NULL)
    {
      /* skip hidden files and ./.. directories */
      if (myfile->d_name[0] == '.') continue;

      /* we're searching for files called nodeinfo%d.data. */
      if (strncmp(myfile->d_name, ddnodeinfostr, ddnodeinfostr.GetLength()) == 0)
	{
	  /* ok we found a file matching our pattern. we need to find out 
	     what node it is.*/
	  node = atoi(&myfile->d_name[ddnodeinfostr.GetLength()]);

	  /* invalid filename/node. continue and try again */
	  if (node <= 0)
	    continue;

	  /* open the nodedata file (ddtmp + d_name = /tmp/dd/nodeinfo%d.dat) */
	  tempStr = (ddtmp + myfile->d_name);
	  fp = fopen(tempStr, "r");

	  /* for some strange reason, we couldn't open the nodeinfo file.
	     - nothing to complain about; skip it and continue. */
	  if (fp == NULL) 
	    continue;
	  
	  /* read user-online info */
	  fread(&mynodeinfo, 1, sizeof(struct DayDream_NodeInfo), fp);
	  fclose(fp);

	  /* no user online */
	  if (mynodeinfo.ddn_userslot == -1)
	    continue;

	  /* scan activity field for strings which can indicate that a transfer is
	     going on */
	  for (int x = 0; !transferstr[x].IsEmpty(); x++)
	    {
	      /* we've found a activity that matches our search strings */
	      if (strncasecmp(mynodeinfo.ddn_activity, transferstr[x], transferstr[x].GetLength()) == 0)
		{
		  /* now we want to scan dd's transfer log /tmp/dd/dszlog.%d 
		     (node), to find out what files the user has transfered. 
		     if we can't find the file, we know that 1) the transfer 
                     has not started yet 2) false alarm */
		  tempStr.Format("%sdszlog.%d", (char *)ddtmp, node);

		  /* false alarm */
		  if ((fp = fopen(tempStr, "r")) == NULL)
		    break;
		  
		  /* print seperatorline if this isn't the first transfer
		     activity */
		  if (totactivities > 0)
		    PrintSeperator('+');
		  
		  /* ahh. one more transfer activity. */
		  totactivities++;
		  
		  /* read user account from userbase.dat so we can get the
		     user_name and user_handle. a slotnumber wont tell very
		     much, right?  see DDDIUserbase class for more info */
		  DDDIUserbase myuser(mynodeinfo.ddn_userslot);
		  
		  struct TransferLogEntry myentry;
		  
		  /* read file entries from dszlog */
		  while (ReadEntryFromTransferLog(fp, &myentry) != -1)
		    {
		      PrintStatusLine(node, myuser.GetHandle(), myentry.type, \
				      myentry.filename, myentry.filesize, myentry.cps);
		      /* add totals */
		      switch (myentry.type)
			{
			case STAT_UPLOADOK:
			case STAT_HYDRAUPLOADOK:
			  mTotalUlBytes+=myentry.filesize;
			  mTotalUlFiles++;
			  break;
			case STAT_DOWNLOADOK:
			case STAT_HYDRADOWNLOADOK:
			  mTotalDlBytes+=myentry.filesize;
			  mTotalDlFiles++;
			  break;
			}
		    }
		  fclose(fp);

		  /* now we want to show the files that are being transfered 
		     right now. the only place we can get that kind of info 
		     is from the activityfield. upload-> "ul: %s" 
		     download-> "dl: %s". if it is a hydra transfer, we
		     won't know anything about the current file. */
		  
		  if (strncasecmp(mynodeinfo.ddn_activity, dduploadingstr, dduploadingstr.GetLength()) == 0)		  
		    {
		      PrintStatusLine(node, myuser.GetHandle(), \
				      STAT_UPLOADING, &mynodeinfo.ddn_activity[4], 0L, mynodeinfo.ddn_bpsrate);
		    }
		  if (strncasecmp(mynodeinfo.ddn_activity, dddownloadingstr, dddownloadingstr.GetLength()) == 0)
		    {
		      PrintStatusLine(node, myuser.GetHandle(), \
				      STAT_DOWNLOADING, &mynodeinfo.ddn_activity[4], 0L, mynodeinfo.ddn_bpsrate);
		    }
		}
	    }
	}
    }
  closedir(mydir);

  if (totactivities <= 0)
    SendString("|           \e[32m- NO TRANSFER ACTIVITIES -\e[34m                                     |\n");

  /* print stats before exiting */
  PrintStats();
}

/*
 * analyze arguments supplied to WHAT
 */
void what::AnalyzeArguments(int argc, String* argv)
{
  /* loop through commandlist */
  for (int i=0; i < argc; i++)
    {
      /* skip empty arrays */
      if (!argv[i])
	continue;

      /* check for commands */
      if (argv[i][0] == '-')
	{
	  /* a command found; make sure it's valid */
	  if (argv[i][1] == 0)
	    continue;

	  switch (argv[i][1])
	    {
	    case 'f':
	      /*
	      if (argv[i][2])
		{
		  int first = atoi((char *)&argv[i][2]);
		  if (first > 0)
		    mFirstCnt = first;
		}
		*/
	      break;
	    case 'l':
	      /*
	      if (argv[i][2])
		{
		  int last = atoi((char *)&argv[i][2]);
		  if (last > 0)
		    mLastCnt = last;
		}
		*/
	      break;
	    case 'u':
	      /* show upload activities only */
	      mShowUploadsOnly = true;
	      break;
	    case 'd':
	      /* show download activities only */
	      mShowDownloadsOnly = true;
	      break;
	    case 'e':
	      /* show extended filenames */
	      mShowExtended = true;
	      break;
	    case 'n':
	      /* nonstop; no pauses */
	      if (argv[i][2] == 's')
		mShowNonstop = true;
	      break;
	      /*
	    default:
	      SendString("command %s unsupported by this version\n", (char *)argv[i]);
	      break;
	      */
	    }
	}
      else
	{
	  /* should be the node-argument or we've got a invalid commandline */
	  int node = atoi(argv[i]);
	  if (node > 0)
	    mShowNode = node;
	}
    }
}

void what::PrintHeader()
{
  SendString("\e[34m.--------------------------------------------------------------------------.\n");
  SendString("\e[34m| \e[33mWHAT: Transfer Activities v%.1d.%2d Coding by bObO/mYSTiC (WHAT -h for help) \e[34m|\n", DOOR_VER, DOOR_REV);
  
  // WHAT: Transfer Activities v0.80 (DayDream) Copyright (c)1997 bObO/mYSTiC
  // WHAT: Transfer Activities v0.81 Coding by bObO/mYSTiC (WHAT -h for help)

  PrintSeperator('v');

  SendString("\e[34m|\e[0mND\e[34m| \e[0mUsername (Handle)    \e[34m| \e[0mStatus      \e[34m| \e[0mFilename     \e[34m|  \e[0mFilesize \e[34m|   \e[0mCPS \e[34m|\n");

  PrintSeperator('+');
}

void what::PrintHelp()
{
  SendString("\n");
  SendString(" WHAT v%.1d.%.2d (DDDI v%.1d.%.2d): (commandline override defaults)\n", DOOR_VER, DOOR_REV, DDDI_VER, DDDI_REV);
  SendString("   Enter 'WHAT <node>' to view only one node.\n");
  SendString("   Enter 'WHAT -f<num>' to view only the first <num> transfers.\n");
  SendString("   Enter 'WHAT -l<num>' to view only the last <num> transfers.\n");
  SendString("   Enter 'WHAT -u' to view only upload activities.\n");
  SendString("   Enter 'WHAT -d' to view only download activities.\n");
  SendString("   Enter 'WHAT -e' to view extended filenames (i.e. filenames>12).\n");
  SendString("   Enter 'WHAT -ns' to view nonstop without pauses.\n\n");
  SendString(" Example:\n");
  SendString("   'WHAT -l10 -u -e' (last 10 transfers, upload transfers only, extended filenames)\n\n");
  SendString(" Default:\n");
  SendString("   'WHAT %s' (%s)\n\n", "n/a", "n/a");
}

void what::PrintStatusLine(int node, const char* username, int status, const char* file, int size, int cps)
{
  const char* nodestatus[] = {
    /* STAT_STARTUL          */ "BeginningUL",
    /* STAT_STARTDL          */ "BeginningDL",
    /* STAT_UPLOADING        */ "Uploading",
    /* STAT_DOWNLOADING      */ "Downloading",
    /* STAT_UPLOADOK         */ "Upload - OK",
    /* STAT_DOWNLOADOK       */ "Download OK",
    /* STAT_HYDRAUPLOADING   */ "Hydra Uling",
    /* STAT_HYDRADOWNLOADING */ "Hydra Dling",
    /* STAT_HYDRAUPLOADOK    */ "Hydra UL OK",
    /* STAT_HYDRADOWNLOADOK  */ "Hydra DL OK",
    /* STAT_NA               */ "N/A" };

  char ind;

  switch (status)
    {
    case STAT_UPLOADING:
    case STAT_DOWNLOADING:
    case STAT_HYDRAUPLOADING:
    case STAT_HYDRADOWNLOADING:
      ind = '*';
      break;
    case STAT_STARTUL:
    case STAT_STARTDL:
      ind = '#';
      break;
    default:
      ind = ' ';
      break;
    }

  SendString("\e[34m|\e[32m%.2d\e[34m| \e[0m%-20s\e[34m |\e[33m%c\e[35m%-11s\e[34m | \e[36m%-12.12s\e[34m | \e[0m%9d\e[34m | \e[32m%5d\e[34m |\n", node, username, ind, nodestatus[status], file, size, cps);
}

void what::PrintSeperator(char sep)
{
  SendString("\e[34m|--%c----------------------%c-------------%c--------------%c-----------%c-------|\n", sep, sep, sep, sep, sep);
}

void what::PrintStats()
{
  PrintSeperator('^');

  SendString("\e[34m| \e[0m  UPLOAD ACTIVITIES  ->  \e[33mTotal bytes\e[32m: [ \e[0m%11d\e[32m ]   \e[33mfiles\e[32m: [ \e[0m%5d\e[32m ]\e[34m |\n", mTotalUlBytes, mTotalUlFiles);
  SendString("\e[34m| \e[0mDOWNLOAD ACTIVITIES  ->  \e[33mTotal bytes\e[32m: [ \e[0m%11d\e[32m ]   \e[33mfiles\e[32m: [ \e[0m%5d\e[32m ]\e[34m |\n", mTotalDlBytes, mTotalDlFiles);

  PrintSeperator('-');

  SendString("\e[34m| \e[32m[\e[33m#\e[32m] \e[0mFile Transfer not started yet (node) \e[32m[\e[33m*\e[32m] \e[0mTransfer in progress (file) \e[34m|\n");
  SendString("\e[34m`--------------------------------------------------------------------------'\e[0m\n");
}

int what::ReadEntryFromTransferLog(FILE* fp, struct TransferLogEntry* tlf)
{
/* 
   /tmp/dd/dszlog.1
   01234567890123456789012345678901234567890123456789012345678901234567890
   S 1483999     0 bps 3693 cps   0 errors     0 1024 blhwws01.zip 0
   S 1461011     0 bps 3698 cps   0 errors     0 1024 blhwws02.zip 0

   FIRST WORD "S" ->
   's' = download
   'S' = upload
   'H' = hydra download
   'R' = hydra upload
*/

#define DSZLOG_TRANSFERTYPE_POS 1
#define DSZLOG_FILESIZE_POS     2
#define DSZLOG_BPS_POS          3
#define DSZLOG_CPS_POS          5
#define DSZLOG_ERRORS_POS       7
#define DSZLOG_BLOCKSIZE_POS   10
#define DSZLOG_FILENAME_POS    11

  /* make sure our file-pointer is valid */
  if (fp == NULL)
    return -1;

  String temp;

  /* buffer for file-reading (200 bytes / line) */
  String buf(200);

  /* read a line from dszlog */
  if (fgets(buf, buf.GetLength(), fp) != NULL)
    {
      /* does not work!! ->
	 note: does not work because our stringclass returns
	 the number of bytes allocated, and not the stringlength
	 in this case -> */
      if (buf.GetLength() < 60)
	return -1;

      temp = buf.GetWord(DSZLOG_TRANSFERTYPE_POS);
      /* make sure we actually have a word in this line; otherwise things are probably
	 kind of screwed up */
      if (!temp)
	return -1;

      /* we need to find out what type of transfer this is */
      switch (temp[0])
	{
	case 's':
	  /* download */
	  tlf->type = STAT_DOWNLOADOK;
	  break;
	case 'S':
	  /* upload */
	  tlf->type = STAT_UPLOADOK;
	  break;
	case 'R':
	  /* hydra upload */
	  tlf->type = STAT_HYDRAUPLOADOK;
	  break;
	case 'H':
	  /* hydra download */
	  tlf->type = STAT_HYDRADOWNLOADOK;
	  break;
	}

      temp = buf.GetWord(DSZLOG_FILESIZE_POS);
      tlf->filesize = atoi(temp);

      temp = buf.GetWord(DSZLOG_BPS_POS);
      tlf->bps = atoi(temp);

      temp = buf.GetWord(DSZLOG_CPS_POS);
      tlf->cps = atoi(temp);

      temp = buf.GetWord(DSZLOG_ERRORS_POS);
      tlf->errors = atoi(temp);

      temp = buf.GetWord(DSZLOG_BLOCKSIZE_POS);
      tlf->blocksize = atoi(temp);

      tlf->filename = buf.GetWord(DSZLOG_FILENAME_POS);

      return 1;
    }

  return -1;
}

/*
  .--------------------------------------------------------------------------.
  | WHAT: Transfer Activities v0.89 (DayDream) Copyright (c)1997 bObO/mYSTiC |
  |--v----------------------v-------------v--------------v-----------v-------.
  |ND| Username (Handle)    | Status      | Filename     |      Size |   CPS |
  |--+----------------------+-------------+--------------+-----------+-------|
  |01| bObO/mYSTiC          | UPLOAD-CRC  | 12345678.123 | 2.312.312 | 23456 |
  |12| 12345678901234567890 | 12345678901 | 123456789012 | 123456789 | 12345
  |01| bObO/mYSTiC          | DOWNLOAD-OK | 12345678.123 | 2.345.678 | 23456 |
  |01|*bObO/mYSTiC          | HYDRA UL    | 12345678.123 | 2.345.678 | 23456 |
  |--^----------------------^-------------^--------------^-----------^-------|
  |   UPLOAD ACTIVITIES  ->  Total bytes: [ 112.312.312 ]   files: [ 1.234 ] |
  | DOWNLOAD ACTIVITIES  ->  Total bytes: [ 123.123.123 ]   files: [ 1.123 ] |
  |--------------------------------------------------------------------------|
*/
