/*
 * This file is part of OpenTTD.
 * OpenTTD is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, version 2.
 * OpenTTD is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with OpenTTD. If not, see <http://www.gnu.org/licenses/>.
 */

/** @file table/townname.h Namepart tables for the town name generator */

#include "../core/enum_type.hpp"

static const char * const _name_original_english_1[] = {
	"Great ",
	"Little ",
	"New ",
	"Fort ",
};

static const char * const _name_original_english_2[] = {
	"Wr",
	"B",
	"C",
	"Ch",
	"Br",
	"D",
	"Dr",
	"F",
	"Fr",
	"Fl",
	"G",
	"Gr",
	"H",
	"L",
	"M",
	"N",
	"P",
	"Pr",
	"Pl",
	"R",
	"S",
	"S",
	"Sl",
	"T",
	"Tr",
	"W",
};

static const char * const _name_original_english_3[] = {
	"ar",
	"a",
	"e",
	"in",
	"on",
	"u",
	"un",
	"en",
};

static const char * const _name_original_english_4[] = {
	"n",
	"ning",
	"ding",
	"d",
	"",
	"t",
	"fing",
};

static const char * const _name_original_english_5[] = {
	"ville",
	"ham",
	"field",
	"ton",
	"town",
	"bridge",
	"bury",
	"wood",
	"ford",
	"hall",
	"ston",
	"way",
	"stone",
	"borough",
	"ley",
	"head",
	"bourne",
	"pool",
	"worth",
	"hill",
	"well",
	"hattan",
	"burg",
};

static const char * const _name_original_english_6[] = {
	"-on-sea",
	" Bay",
	" Market",
	" Cross",
	" Bridge",
	" Falls",
	" City",
	" Ridge",
	" Springs",
};

static const char * const _name_additional_english_prefix[] = {
	"Great ",
	"Little ",
	"New ",
	"Fort ",
	"St. ",
	"Old ",
};

static const char * const _name_additional_english_1a[] = {
	"Pen",
	"Lough",
	"Stam",
	"Aber",
	"Acc",
	"Ex",
	"Ax",
	"Bre",
	"Cum",
	"Dun",
	"Fin",
	"Inver",
	"Kin",
	"Mon",
	"Nan",
	"Nant",
	"Pit",
	"Pol",
	"Pont",
	"Strath",
	"Tre",
	"Tilly",
	"Beck",
	"Canter",
	"Bath",
	"Liver",
	"Mal",
	"Ox",
	"Bletch",
	"Maccles",
	"Grim",
	"Wind",
	"Sher",
	"Gates",
	"Orp",
	"Brom",
	"Lewis",
	"Whit",
	"White",
	"Worm",
	"Tyne",
	"Avon",
	"Stan",
};

static const char * const _name_additional_english_1b1[] = {
	"Wr",
	"B",
	"C",
	"Ch",
	"Br",
	"D",
	"Dr",
	"F",
	"Fr",
	"Fl",
	"G",
	"Gr",
	"H",
	"L",
	"M",
	"N",
	"P",
	"Pr",
	"Pl",
	"R",
	"S",
	"S",
	"Sl",
	"T",
	"Tr",
	"W",
};

static const char * const _name_additional_english_1b2[] = {
	"ar",
	"a",
	"e",
	"in",
	"on",
	"u",
	"o",
	"ee",
	"es",
	"ea",
	"un",
	"en",
};

static const char * const _name_additional_english_1b3a[] = {
	"n",
	"d",
	"",
	"t",
	"",
	"",
};

static const char * const _name_additional_english_1b3b[] = {
	"ning",
	"ding",
	"fing",
};

static const char * const _name_additional_english_2[] = {
	"ville",
	"ham",
	"field",
	"ton",
	"town",
	"borough",
	"bridge",
	"bury",
	"wood",
	"ditch",
	"ford",
	"hall",
	"dean",
	"leigh",
	"dore",
	"ston",
	"stow",
	"church",
	"wich",
	"low",
	"way",
	"stone",
	"minster",
	"ley",
	"head",
	"bourne",
	"pool",
	"worth",
	"hill",
	"well",
	"hattan",
	"burg",
	"berg",
	"burgh",
	"port",
	"stoke",
	"haven",
	"stable",
	"stock",
	"side",
	"brook",
	"don",
	"den",
	"down",
	"nor",
	"grove",
	"combe",
	"by",
	"say",
	"ney",
	"chester",
	"dale",
	"ness",
	"shaw",
	"thwaite",
};

static const char * const _name_additional_english_3[] = {
	"-on-sea",
	" Bay",
	" Market",
	" Beeches",
	" Common",
	" Park",
	" Heath",
	" Marsh",
	" Green",
	" Castle",
	" End",
	" Rivers",
	" Cross",
	" Bridge",
	" Falls",
	" City",
	" Ridge",
	" Springs",
};

static const char * const _name_austrian_a1[] = {
	"Bad ",
	"Deutsch ",
	"Gross ",
	"Klein ",
	"Markt ",
	"Maria ",
};

static const char * const _name_austrian_a2[] = {
	"Aus",
	"Alten",
	"Braun",
	u8"V\u00f6sl",
	"Mittern",
	"Nuss",
	"Neu",
	"Walters",
	"Breiten",
	"Eisen",
	"Feld",
	"Mittern",
	"Gall",
	"Obern",
	"Grat",
	"Heiligen",
	"Hof",
	"Holla",
	"Stein",
	"Eber",
	"Eggen",
	"Enzers",
	"Frauen",
	"Herren",
	"Hof",
	u8"H\u00fctt",
	"Kaisers",
	u8"K\u00f6nigs",
	"Knittel",
	"Lang",
	"Ober",
	"Ollers",
	"Pfaffen",
	"Potten",
	"Salz",
	"Schwarz",
	"Stocker",
	"Unter",
	"Utten",
	u8"V\u00f6sen",
	"Vill",
	"Weissen",
};

static const char * const _name_austrian_a3[] = {
	"see",
	"bach",
	"dorf",
	"ach",
	"stein",
	"hofen",
	"au",
	"ach",
	"kirch",
	"kirchen",
	"kreuz",
	"brunn",
	"siedl",
	"markt",
	"wang",
	"haag",
};

static const char * const _name_austrian_a4[] = {
	"Bruck",
	"Brunn",
	"Gams",
	"Grein",
	"Ried",
	"Faak",
	"Zell",
	"Spital",
	"Kirchberg",
	"Saal",
	"Taferl",
	"Wald",
};

static const char * const _name_austrian_a5[] = {
	"St. ",
	"Sankt ",
};

static const char * const _name_austrian_a6[] = {
	"Aegyd",
	u8"Andr\u00e4",
	"Georgen",
	"Jakob",
	"Johann",
	"Leonhard",
	"Marein",
	"Lorenzen",
	"Margarethen",
	"Martin",
	"Michael",
	"Nikolai",
	"Oswald",
	"Peter",
	u8"P\u00f6lten",
	"Stefan",
	"Stephan",
	"Thomas",
	"Veit",
	"Wolfgang",
};

static const char * const _name_austrian_f1[] = {
	" an der ",
	" ob der ",
};

static const char * const _name_austrian_f2[] = {
	"Donau",
	"Steyr",
	"Lafnitz",
	"Leitha",
	"Thaya",
	"Gail",
	"Drau",
	"Salzach",
	"Ybbs",
	"Traisen",
	"Enns",
	"Mur",
	"Ill",
};

static const char * const _name_austrian_b1[] = {
	" am ",
};

static const char * const _name_austrian_b2[] = {
	"Brenner",
	"Dachstein",
	"Gebirge",
	"Grossglockner",
	"Hausruck",
	"Semmering",
	"Wagram",
	"Wechsel",
	"Wilden Kaiser",
	"Ziller",
};

static const char * const _name_german_real[] = {
	"Berlin",
	"Bonn",
	"Bremen",
	"Cottbus",
	"Chemnitz",
	"Dortmund",
	"Dresden",
	"Erfurt",
	"Erlangen",
	"Essen",
	"Fulda",
	"Gera",
	"Kassel",
	"Kiel",
	u8"K\u00f6ln",
	u8"L\u00fcbeck",
	"Magdeburg",
	u8"M\u00fcnchen",
	"Potsdam",
	"Stuttgart",
	"Wiesbaden",
};

static const char * const _name_german_pre[] = {
	"Bad ",
	"Klein ",
	"Neu ",
};

static const char * const _name_german_1[] = {
	"Alb",
	"Als",
	"Ander",
	"Arns",
	"Bruns",
	"Bam",
	"Biele",
	"Cloppen",
	"Co",
	"Duis",
	u8"D\u00fcssel",
	"Dannen",
	"Elb",
	"Els",
	"Elster",
	"Eichen",
	"Ems",
	"Fahr",
	"Falken",
	"Flens",
	"Frank",
	"Frei",
	"Freuden",
	"Fried",
	u8"F\u00fcrsten",
	"Hahn",
	"Ham",
	"Harz",
	"Heidel",
	"Hers",
	"Herz",
	"Holz",
	"Hildes",
	"Inns",
	"Ilsen",
	"Ingols",
	"Kel",
	"Kies",
	"Korn",
	"Kor",
	"Kreuz",
	"Kulm",
	"Langen",
	"Lim",
	"Lohr",
	u8"L\u00fcne",
	"Mel",
	"Michels",
	u8"M\u00fchl",
	"Naum",
	"Nest",
	"Nord",
	"Nort",
	"Nien",
	"Nidda",
	"Nieder",
	u8"N\u00fcrn",
	"Ober",
	"Offen",
	"Osna",
	"Olden",
	"Ols",
	"Oranien",
	"Pader",
	"Quedlin",
	"Quer",
	"Ravens",
	"Regens",
	"Rott",
	"Ros",
	u8"R\u00fcssels",
	"Saal",
	"Saar",
	"Salz",
	u8"Sch\u00f6ne",
	"Schwein",
	"Sonder",
	"Sonnen",
	"Stein",
	"Strals",
	"Straus",
	u8"S\u00fcd",
	"Ton",
	"Unter",
	"Ur",
	"Vor",
	"Wald",
	"War",
	"Wert",
	"Wester",
	"Witten",
	"Wolfs",
	u8"W\u00fcrz",
};

static const char * const _name_german_2[] = {
	"bach",
	"berg",
	u8"br\u00fcck",
	u8"br\u00fccken",
	"burg",
	"dorf",
	"feld",
	"furt",
	"hausen",
	"haven",
	"heim",
	"horst",
	"mund",
	u8"m\u00fcnster",
	"stadt",
	"wald",
};

static const char * const _name_german_3_an_der[] = {
	" an der ",
};

static const char * const _name_german_3_am[] = {
	" am ",
};

static const char * const _name_german_4_an_der[] = {
	"Oder",
	"Spree",
	"Donau",
	"Saale",
	"Elbe",
};

static const char * const _name_german_4_am[] = {
	"Main",
};

static const char * const _name_spanish_real[] = {
	"Caracas",
	"Maracay",
	"Maracaibo",
	"Valencia",
	"El Dorado",
	"Morrocoy",
	"Cata",
	"Cataito",
	u8"Ciudad Bol\u00edvar",
	"Barquisimeto",
	u8"M\u00e9rida",
	"Puerto Ordaz",
	"Santa Elena",
	"San Juan",
	"San Luis",
	"San Rafael",
	"Santiago",
	"Barcelona",
	"Barinas",
	u8"San Crist\u00f3bal",
	"San Francisco",
	u8"San Mart\u00edn",
	"Guayana",
	"San Carlos",
	u8"El Lim\u00f3n",
	"Coro",
	"Coro Coro",
	"Puerto Ayacucho",
	"Elorza",
	"Arismendi",
	"Trujillo",
	u8"Car\u00fapano",
	"Anaco",
	"Lima",
	"Cuzco",
	"Iquitos",
	"Callao",
	"Huacho",
	u8"Caman\u00e1",
	"Puerto Chala",
	"Santa Cruz",
	"Quito",
	"Cuenca",
	u8"Medell\u00edn",
	u8"Tulc\u00e1n",
	"Esmeraldas",
	"Ibarra",
	"San Lorenzo",
	"Macas",
	u8"Mora\u00f1a",
	"Machala",
	"Zamora",
	"Latacunga",
	"Tena",
	"Cochabamba",
	u8"Ascensi\u00f3n",
	"Magdalena",
	"Santa Ana",
	"Manoa",
	"Sucre",
	"Oruro",
	"Uyuni",
	u8"Potos\u00ed",
	"Tupiza",
	"La Quiaca",
	u8"Yacu\u00edba",
	"San Borja",
	"Fuerte Olimpo",
	u8"Fort\u00edn Esteros",
	"Campo Grande",
	u8"Bogot\u00e1",
	"El Banco",
	"Zaragoza",
	"Neiva",
	"Mariano",
	"Cali",
	"La Palma",
	"Andoas",
	"Barranca",
	"Montevideo",
	"Valdivia",
	"Arica",
	"Temuco",
	"Tocopilla",
	"Mendoza",
	"Santa Rosa",
};

static const char * const _name_french_real[] = {
	"Agincourt",
	"Lille",
	"Dinan",
	"Aubusson",
	"Rodez",
	"Bergerac",
	"Bordeaux",
	"Bayonne",
	"Montpellier",
	u8"Mont\u00e9limar",
	"Valence",
	"Digne",
	"Nice",
	"Cannes",
	"St. Tropez",
	"Marseille",
	"Narbonne",
	u8"S\u00e8te",
	"Aurillac",
	u8"Gu\u00e9ret",
	"Le Creusot",
	"Nevers",
	"Auxerre",
	"Versailles",
	"Meaux",
	u8"Ch\u00e2lons",
	u8"Compi\u00e8gne",
	"Metz",
	"Chaumont",
	"Langres",
	"Bourg",
	"Lyon",
	"Vienne",
	"Grenoble",
	"Toulon",
	"Rennes",
	"Le Mans",
	"Angers",
	"Nantes",
	u8"Ch\u00e2teauroux",
	u8"Orl\u00e9ans",
	"Lisieux",
	"Cherbourg",
	"Morlaix",
	"Cognac",
	"Agen",
	"Tulle",
	"Blois",
	"Troyes",
	"Charolles",
	"Toulouse",
	u8"Chamb\u00e9ry",
	"Tours",
	"St. Brieuc",
	"St. Malo",
	"La Rochelle",
	"St. Flour",
	"Le Puy",
	"Vichy",
	"St. Valery",
	"Beaujolais",
	"Reims",
	"Albi",
	"Paris",
	"Biarritz",
	u8"B\u00e9ziers",
	u8"N\u00eemes",
	"Chamonix",
	u8"Angoul\u00e8me",
	u8"Alen\u00e7on",
};

static const char * const _name_silly_1[] = {
	"Binky",
	"Blubber",
	"Bumble",
	"Crinkle",
	"Crusty",
	"Dangle",
	"Dribble",
	"Flippety",
	"Google",
	"Muffin",
	"Nosey",
	"Pinker",
	"Quack",
	"Rumble",
	"Sleepy",
	"Sliggles",
	"Snooze",
	"Teddy",
	"Tinkle",
	"Twister",
	"Pinker",
	"Hippo",
	"Itchy",
	"Jelly",
	"Jingle",
	"Jolly",
	"Kipper",
	"Lazy",
	"Frogs",
	"Mouse",
	"Quack",
	"Cheeky",
	"Lumpy",
	"Grumpy",
	"Mangle",
	"Fiddle",
	"Slugs",
	"Noodles",
	"Poodles",
	"Shiver",
	"Rumble",
	"Pixie",
	"Puddle",
	"Riddle",
	"Rattle",
	"Rickety",
	"Waffle",
	"Sagging",
	"Sausage",
	"Egg",
	"Sleepy",
	"Scatter",
	"Scramble",
	"Silly",
	"Simple",
	"Trickle",
	"Slippery",
	"Slimey",
	"Slumber",
	"Soggy",
	"Sliggles",
	"Splutter",
	"Sulky",
	"Swindle",
	"Swivel",
	"Tasty",
	"Tangle",
	"Toggle",
	"Trotting",
	"Tumble",
	"Snooze",
	"Water",
	"Windy",
	"Amble",
	"Bubble",
	"Cheery",
	"Cheese",
	"Cockle",
	"Cracker",
	"Crumple",
	"Teddy",
	"Evil",
	"Fairy",
	"Falling",
	"Fishy",
	"Fizzle",
	"Frosty",
	"Griddle",
};

static const char * const _name_silly_2[] = {
	"ton",
	"bury",
	"bottom",
	"ville",
	"well",
	"weed",
	"worth",
	"wig",
	"wick",
	"wood",
	"pool",
	"head",
	"burg",
	"gate",
	"bridge",
};

static const char * const _name_swedish_1[] = {
	"Gamla ",
	"Lilla ",
	"Nya ",
	"Stora ",
};

static const char * const _name_swedish_2[] = {
	"Boll",
	"Bor",
	"Ed",
	"En",
	"Erik",
	"Es",
	"Fin",
	"Fisk",
	u8"Gr\u00f6n",
	"Hag",
	"Halm",
	"Karl",
	"Kram",
	"Kung",
	"Land",
	"Lid",
	"Lin",
	"Mal",
	"Malm",
	"Marie",
	"Ner",
	"Norr",
	"Oskar",
	"Sand",
	"Skog",
	"Stock",
	"Stor",
	u8"Str\u00f6m",
	"Sund",
	u8"S\u00f6der",
	"Tall",
	"Tratt",
	"Troll",
	"Upp",
	"Var",
	u8"V\u00e4ster",
	u8"\u00c4ngel",
	u8"\u00d6ster",
};

static const char * const _name_swedish_2a[] = {
	"B",
	"Br",
	"D",
	"Dr",
	"Dv",
	"F",
	"Fj",
	"Fl",
	"Fr",
	"G",
	"Gl",
	"Gn",
	"Gr",
	"H",
	"J",
	"K",
	"Kl",
	"Kn",
	"Kr",
	"Kv",
	"L",
	"M",
	"N",
	"P",
	"Pl",
	"Pr",
	"R",
	"S",
	"Sk",
	"Skr",
	"Sl",
	"Sn",
	"Sp",
	"Spr",
	"St",
	"Str",
	"Sv",
	"T",
	"Tr",
	"Tv",
	"V",
	"Vr",
};

static const char * const _name_swedish_2b[] = {
	"a",
	"e",
	"i",
	"o",
	"u",
	"y",
	u8"\u00e5",
	u8"\u00e4",
	u8"\u00f6",
};

static const char * const _name_swedish_2c[] = {
	"ck",
	"d",
	"dd",
	"g",
	"gg",
	"l",
	"ld",
	"m",
	"n",
	"nd",
	"ng",
	"nn",
	"p",
	"pp",
	"r",
	"rd",
	"rk",
	"rp",
	"rr",
	"rt",
	"s",
	"sk",
	"st",
	"t",
	"tt",
	"v",
};

static const char * const _name_swedish_3[] = {
	"arp",
	"berg",
	"boda",
	"borg",
	"bro",
	"bukten",
	"by",
	"byn",
	"fors",
	"hammar",
	"hamn",
	"holm",
	"hus",
	u8"h\u00e4ttan",
	"kulle",
	u8"k\u00f6ping",
	"lund",
	u8"l\u00f6v",
	"sala",
	"skrona",
	u8"sl\u00e4tt",
	u8"sp\u00e5ng",
	"stad",
	"sund",
	"svall",
	"svik",
	u8"s\u00e5ker",
	"udde",
	"valla",
	"viken",
	u8"\u00e4lv",
	u8"\u00e5s",
};

static const char * const _name_dutch_1[] = {
	"Nieuw ",
	"Oud ",
	"Groot ",
	"Zuid ",
	"Noord ",
	"Oost ",
	"West ",
	"Klein ",
};

static const char * const _name_dutch_2[] = {
	"Hoog",
	"Laag",
	"Zuider",
	"Zuid",
	"Ooster",
	"Oost",
	"Wester",
	"West",
	"Hoofd",
	"Midden",
	"Eind",
	"Amster",
	"Amstel",
	"Dord",
	"Rotter",
	"Haar",
	"Til",
	"Enk",
	"Dok",
	"Veen",
	"Leidsch",
	"Lely",
	"En",
	"Kaats",
	"U",
	"Maas",
	"Mar",
	"Bla",
	"Al",
	"Alk",
	"Eer",
	"Drie",
	"Ter",
	"Groes",
	"Goes",
	"Soest",
	"Coe",
	"Uit",
	"Zwaag",
	"Hellen",
	"Slie",
	"IJ",
	"Grubben",
	"Groen",
	"Lek",
	"Ridder",
	"Schie",
	"Olde",
	"Roose",
	"Haar",
	"Til",
	"Loos",
	"Hil",
};

static const char * const _name_dutch_3[] = {
	"Drog",
	"Nat",
	"Valk",
	"Bob",
	"Dedem",
	"Kollum",
	"Best",
	"Hoend",
	"Leeuw",
	"Graaf",
	"Uithuis",
	"Purm",
	"Hard",
	"Hell",
	"Werk",
	"Spijk",
	"Vink",
	"Wams",
	"Heerhug",
	"Koning",
};

static const char * const _name_dutch_4[] = {
	"e",
	"er",
	"el",
	"en",
	"o",
	"s",
};

static const char * const _name_dutch_5[] = {
	"stad",
	"vorst",
	"dorp",
	"dam",
	"beek",
	"doorn",
	"zijl",
	"zijlen",
	"lo",
	"muiden",
	"meden",
	"vliet",
	"nisse",
	"daal",
	"vorden",
	"vaart",
	"mond",
	"zaal",
	"water",
	"duinen",
	"heuvel",
	"geest",
	"kerk",
	"meer",
	"maar",
	"hoorn",
	"rade",
	"wijk",
	"berg",
	"heim",
	"sum",
	"richt",
	"burg",
	"recht",
	"drecht",
	"trecht",
	"tricht",
	"dricht",
	"lum",
	"rum",
	"halen",
	"oever",
	"wolde",
	"veen",
	"hoven",
	"gast",
	"kum",
	"hage",
	"dijk",
	"zwaag",
	"pomp",
	"huizen",
	"bergen",
	"schede",
	"mere",
	"end",
};

static const char * const _name_finnish_real[] = {
	"Aijala",
	"Kisko",
	"Espoo",
	"Helsinki",
	"Tapiola",
	u8"J\u00e4rvel\u00e4",
	"Lahti",
	"Kotka",
	"Hamina",
	"Loviisa",
	"Kouvola",
	"Tampere",
	"Oulu",
	"Salo",
	"Malmi",
	"Pelto",
	"Koski",
	"Iisalmi",
	"Raisio",
	"Taavetti",
	"Joensuu",
	"Imatra",
	"Tapanila",
	"Pasila",
	"Turku",
	"Kupittaa",
	"Vaasa",
	"Pori",
	"Rauma",
	"Kolari",
	"Lieksa",
};

static const char * const _name_finnish_1[] = {
	"Hiekka",
	"Haapa",
	"Mylly",
	"Sauna",
	"Uusi",
	"Vanha",
	u8"Kes\u00e4",
	"Kuusi",
	"Pelto",
	"Tuomi",
	"Terva",
	"Olki",
	u8"Hein\u00e4",
	u8"Sein\u00e4",
	"Rova",
	"Koivu",
	"Kokko",
	u8"M\u00e4nty",
	"Pihlaja",
	u8"Pet\u00e4j\u00e4",
	"Kielo",
	"Kauha",
	"Viita",
	"Kivi",
	"Riihi",
	u8"\u00c4\u00e4ne",
	"Niini",
};

static const char * const _name_finnish_2[] = {
	"Lappeen",
	"Lohjan",
	"Savon",
	"Lapin",
	u8"Pit\u00e4j\u00e4n",
	"Martin",
	"Kuusan",
	"Kemi",
	"Keri",
	u8"H\u00e4meen",
	"Kangas",
};

static const char * const _name_finnish_3[] = {
	"harju",
	"linna",
	u8"j\u00e4rvi",
	"kallio",
	u8"m\u00e4ki",
	"nummi",
	"joki",
	u8"kyl\u00e4",
	"lampi",
	"lahti",
	u8"mets\u00e4",
	"suo",
	"laakso",
	"niitty",
	"luoto",
	"hovi",
	"ranta",
	"koski",
	"salo",
};

static const char * const _name_polish_1_m[] = {
	"Wielki ",
	u8"Ma\u0142y ",
	u8"Z\u0142y ",
	"Dobry ",
	"Nowy ",
	"Stary ",
	u8"Z\u0142oty ",
	"Zielony ",
	u8"Bia\u0142y ",
	"Modry ",
	u8"D\u0119bowy ",
};

static const char * const _name_polish_1_f[] = {
	"Wielka ",
	u8"Ma\u0142a ",
	u8"Z\u0142a ",
	"Dobra ",
	"Nowa ",
	"Stara ",
	u8"Z\u0142ota ",
	"Zielona ",
	u8"Bia\u0142a ",
	"Modra ",
	u8"D\u0119bowa ",
};

static const char * const _name_polish_1_n[] = {
	"Wielkie ",
	u8"Ma\u0142e ",
	u8"Z\u0142e ",
	"Dobre ",
	"Nowe ",
	"Stare ",
	u8"Z\u0142ote ",
	"Zielone ",
	u8"Bia\u0142e ",
	"Modre ",
	u8"D\u0119bowe ",
};

static const char * const _name_polish_2_o[] = {
	"Frombork",
	"Gniezno",
	"Olsztyn",
	u8"Toru\u0144",
	"Bydgoszcz",
	"Terespol",
	u8"Krak\u00f3w",
	u8"Pozna\u0144",
	u8"Wroc\u0142aw",
	"Katowice",
	"Cieszyn",
	"Bytom",
	"Frombork",
	"Hel",
	"Konin",
	"Lublin",
	"Malbork",
	"Sopot",
	"Sosnowiec",
	u8"Gda\u0144sk",
	"Gdynia",
	"Sieradz",
	"Sandomierz",
	"Szczyrk",
	"Szczytno",
	"Szczecin",
	"Zakopane",
	u8"Szklarska Por\u0119ba",
	"Bochnia",
	"Golub-Dobrzyn",
	"Chojnice",
	"Ostrowiec",
	"Otwock",
	"Wolsztyn",
};

static const char * const _name_polish_2_m[] = {
	"Jarocin",
	"Gogolin",
	u8"Tomasz\u00f3w",
	u8"Piotrk\u00f3w",
	"Lidzbark",
	"Rypin",
	"Radzymin",
	u8"Wo\u0142omin",
	u8"Pruszk\u00f3w",
	"Olsztynek",
	"Rypin",
	"Cisek",
	"Krotoszyn",
	"Stoczek",
	"Lubin",
	"Lubicz",
	"Milicz",
	"Targ",
	u8"Ostr\u00f3w",
	"Ozimek",
	"Puck",
	"Rzepin",
	"Siewierz",
	"Stargard",
	"Starogard",
	"Turek",
	"Tymbark",
	"Wolsztyn",
	"Strzepcz",
	"Strzebielin",
	"Sochaczew",
	u8"Gr\u0119bocin",
	"Gniew",
	"Lubliniec",
	"Lubasz",
	"Lutomiersk",
	"Niemodlin",
	"Przeworsk",
	"Ursus",
	"Tyczyn",
	"Sztum",
	"Szczebrzeszyn",
	"Wolin",
	"Wrzeszcz",
	"Zgierz",
	"Zieleniec",
	"Drobin",
	"Garwolin",
};

static const char * const _name_polish_2_f[] = {
	"Szprotawa",
	"Pogorzelica",
	u8"Mot\u0142awa",
	"Lubawa",
	"Nidzica",
	"Kruszwica",
	"Bierawa",
	"Brodnica",
	"Chojna",
	"Krzepica",
	"Ruda",
	"Rumia",
	"Tuchola",
	"Trzebinia",
	"Ustka",
	"Warszawa",
	"Bobowa",
	"Dukla",
	"Krynica",
	"Murowana",
	"Niemcza",
	"Zaspa",
	"Zawoja",
	"Wola",
	"Limanowa",
	"Rabka",
	"Skawina",
	"Pilawa",
};

static const char * const _name_polish_2_n[] = {
	"Lipsko",
	"Pilzno",
	"Przodkowo",
	"Strzelno",
	"Susz",
	"Jaworzno",
	"Choszczno",
	"Mogilno",
	"Luzino",
	"Miasto",
	"Dziadowo",
	"Kowalewo",
	"Legionowo",
	"Miastko",
	"Zabrze",
	"Zawiercie",
	"Kochanowo",
	"Miechucino",
	"Mirachowo",
	"Robakowo",
	"Kosakowo",
	"Borne",
	"Braniewo",
	"Sulinowo",
	"Chmielno",
	u8"Jastrz\u0119bie",
	"Gryfino",
	"Koronowo",
	"Lubichowo",
	"Opoczno",
};

static const char * const _name_polish_3_m[] = {
	" Wybudowanie",
	u8" \u015awi\u0119tokrzyski",
	u8" G\u00f3rski",
	" Morski",
	u8" Zdr\u00f3j",
	" Wody",
	" Bajoro",
	u8" Kraje\u0144ski",
	u8" \u015al\u0105ski",
	" Mazowiecki",
	" Pomorski",
	" Wielki",
	" Maly",
	u8" Warmi\u0144ski",
	" Mazurski",
	" Mniejszy",
	u8" Wi\u0119kszy",
	u8" G\u00f3rny",
	" Dolny",
	" Wielki",
	" Stary",
	" Nowy",
	" Wielkopolski",
	u8" Wzg\u00f3rze",
	" Mosty",
	" Kujawski",
	u8" Ma\u0142opolski",
	" Podlaski",
	" Lesny",
};

static const char * const _name_polish_3_f[] = {
	" Wybudowanie",
	u8" \u015awi\u0119tokrzyska",
	u8" G\u00f3rska",
	" Morska",
	u8" Zdr\u00f3j",
	" Woda",
	" Bajoro",
	u8" Kraje\u0144ska",
	u8" \u015al\u0105ska",
	" Mazowiecka",
	" Pomorska",
	" Wielka",
	u8" Ma\u0142a",
	u8" Warmi\u0144ska",
	" Mazurska",
	" Mniejsza",
	u8" Wi\u0119ksza",
	u8" G\u00f3rna",
	" Dolna",
	" Wielka",
	" Stara",
	" Nowa",
	" Wielkopolska",
	u8" Wzg\u00f3rza",
	" Mosty",
	" Kujawska",
	" Malopolska",
	" Podlaska",
	u8" Le\u015bna",
};

static const char * const _name_polish_3_n[] = {
	" Wybudowanie",
	u8" \u015awietokrzyskie",
	u8" G\u00f3rskie",
	" Morskie",
	u8" Zdr\u00f3j",
	" Wody",
	" Bajoro",
	u8" Kraje\u0144skie",
	u8" \u015al\u0105skie",
	" Mazowieckie",
	" Pomorskie",
	" Wielkie",
	u8" Ma\u0142e",
	u8" Warmi\u0144skie ",
	" Mazurskie ",
	" Mniejsze",
	u8" Wi\u0119ksze",
	u8" G\u00f3rne",
	" Dolne",
	" Wielkie",
	" Stare",
	" Nowe",
	" Wielkopolskie",
	u8" Wzg\u00f3rze",
	" Mosty",
	" Kujawskie",
	u8" Ma\u0142opolskie",
	" Podlaskie",
	u8" Le\u015bne",
};

static const char * const _name_czech_real[] = {
	u8"A\u0161",
	u8"Bene\u0161ov",
	"Beroun",
	u8"Bezdru\u017eice",
	"Blansko",
	u8"B\u0159eclav",
	"Brno",
	u8"Brunt\u00e1l",
	u8"\u010cesk\u00e1 L\u00edpa",
	u8"\u010cesk\u00e9 Bud\u011bjovice",
	u8"\u010cesk\u00fd Krumlov",
	u8"D\u011b\u010d\u00edn",
	u8"Doma\u017elice",
	u8"Dub\u00ed",
	u8"Fr\u00fddek-M\u00edstek",
	u8"Havl\u00ed\u010dk\u016fv Brod",
	u8"Hodon\u00edn",
	u8"Hradec Kr\u00e1lov\u00e9",
	"Humpolec",
	"Cheb",
	"Chomutov",
	"Chrudim",
	"Jablonec nad Nisou",
	u8"Jesen\u00edk",
	u8"Ji\u010d\u00edn",
	"Jihlava",
	u8"Jind\u0159ich\u016fv Hradec",
	"Karlovy Vary",
	u8"Karvin\u00e1",
	"Kladno",
	"Klatovy",
	u8"Kol\u00edn",
	"Kosmonosy",
	u8"Krom\u011b\u0159\u00ed\u017e",
	u8"Kutn\u00e1 Hora",
	"Liberec",
	u8"Litom\u011b\u0159ice",
	"Louny",
	u8"Man\u011bt\u00edn",
	u8"M\u011bln\u00edk",
	u8"Mlad\u00e1 Boleslav",
	"Most",
	u8"N\u00e1chod",
	u8"Nov\u00fd Ji\u010d\u00edn",
	"Nymburk",
	"Olomouc",
	"Opava",
	u8"Or\u00e1\u010dov",
	"Ostrava",
	"Pardubice",
	u8"Pelh\u0159imov",
	u8"Pol\u017eice",
	u8"P\u00edsek",
	u8"Plze\u0148",
	"Praha",
	"Prachatice",
	u8"P\u0159erov",
	u8"P\u0159\u00edbram",
	u8"Prost\u011bjov",
	u8"Rakovn\u00edk",
	"Rokycany",
	u8"Rudn\u00e1",
	u8"Rychnov nad Kn\u011b\u017enou",
	"Semily",
	"Sokolov",
	"Strakonice",
	u8"St\u0159edokluky",
	u8"\u0160umperk",
	"Svitavy",
	u8"T\u00e1bor",
	"Tachov",
	"Teplice",
	u8"T\u0159eb\u00ed\u010d",
	"Trutnov",
	u8"Uhersk\u00e9 Hradi\u0161t\u011b",
	u8"\u00dast\u00ed nad Labem",
	u8"\u00dast\u00ed nad Orlic\u00ed",
	u8"Vset\u00edn",
	u8"Vy\u0161kov",
	u8"\u017d\u010f\u00e1r nad S\u00e1zavou",
	u8"Zl\u00edn",
	"Znojmo",
};


/* The advanced hyperintelligent Czech town names generator!
 * The tables and MakeCzechTownName() is (c) Petr Baudis 2005 (GPL'd)
 * Feel free to ask me about anything unclear or if you need help
 * with cloning this for your own language. */

/* Sing., pl. */
enum CzechGender {
	CZG_SMASC,
	CZG_SFEM,
	CZG_SNEUT,
	CZG_PMASC,
	CZG_PFEM,
	CZG_PNEUT,
	/* Special for substantive stems - the ending chooses the gender. */
	CZG_FREE,
	/* Like CZG_FREE, but disallow CZG_SNEUT. */
	CZG_NFREE
};

enum CzechPattern {
	CZP_JARNI,
	CZP_MLADY,
	CZP_PRIVL
};

/* [CzechGender][CzechPattern] - replaces the last character of the adjective
 * by this.
 * XXX: [CZG_SMASC][CZP_PRIVL] needs special handling: -ovX -> -uv. */
static const char * const _name_czech_patmod[][3] = {
	/* CZG_SMASC */ { u8"\u00ed", u8"\u00fd", "X" },
	/* CZG_SFEM */  { u8"\u00ed", u8"\u00e1", "a" },
	/* CZG_SNEUT */ { u8"\u00ed", u8"\u00e9", "o" },
	/* CZG_PMASC */ { u8"\u00ed", u8"\u00e9", "y" },
	/* CZG_PFEM */  { u8"\u00ed", u8"\u00e9", "y" },
	/* CZG_PNEUT */ { u8"\u00ed", u8"\u00e1", "a" }
};

/* This way the substantive can choose only some adjectives/endings:
 * At least one of these flags must be satisfied: */
enum CzechAllow {
	CZA_SHORT = 1,
	CZA_MIDDLE = 2,
	CZA_LONG = 4,
	CZA_ALL = ~0
};

DECLARE_ENUM_AS_BIT_SET(CzechAllow)

/* All these flags must be satisfied (in the stem->others direction): */
enum CzechChoose {
	CZC_NONE = 0, // No requirements.
	CZC_COLOR = 1,
	CZC_POSTFIX = 2, // Matched if postfix was inserted.
	CZC_NOPOSTFIX = 4, // Matched if no postfix was inserted.
	CZC_ANY = ~0
};

DECLARE_ENUM_AS_BIT_SET(CzechChoose)

struct CzechNameSubst {
	CzechGender gender;
	CzechAllow allow;
	CzechChoose choose;
	const char *name;
};

struct CzechNameAdj {
	CzechPattern pattern;
	CzechChoose choose;
	const char *name;
};

/* Some of items which should be common are doubled. */
static const CzechNameAdj _name_czech_adj[] = {
	{ CZP_JARNI, CZC_ANY, u8"Horn\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"Horn\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"Doln\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"Doln\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"P\u0159edn\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"Zadn\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"Kosteln\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"Havran\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"\u0158\u00ed\u010dn\u00ed" },
	{ CZP_JARNI, CZC_ANY, u8"Jezern\u00ed" },
	{ CZP_MLADY, CZC_ANY, u8"Velk\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Velk\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Mal\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Mal\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Vysok\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"\u010cesk\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Moravsk\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Slov\u00e1ck\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Slezsk\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Uhersk\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Star\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Star\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Nov\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Nov\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Mlad\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Kr\u00e1lovsk\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Kamenn\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Cihlov\u00fd" },
	{ CZP_MLADY, CZC_ANY, u8"Divn\u00fd" },
	{ CZP_MLADY, CZC_COLOR, u8"\u010cerven\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"\u010cerven\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"\u010cerven\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"Zelen\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"\u017dlut\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"Siv\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"\u0160ed\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"B\u00edl\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"B\u00edl\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"Modr\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"R\u016f\u017eov\u00e1" },
	{ CZP_MLADY, CZC_COLOR, u8"\u010cern\u00e1" },
	{ CZP_PRIVL, CZC_ANY, u8"Kr\u00e1lova" },
	{ CZP_PRIVL, CZC_ANY, "Janova" },
	{ CZP_PRIVL, CZC_ANY, "Karlova" },
	{ CZP_PRIVL, CZC_ANY, u8"Kry\u0161tofova" },
	{ CZP_PRIVL, CZC_ANY, u8"Ji\u0159\u00edkova" },
	{ CZP_PRIVL, CZC_ANY, "Petrova" },
	{ CZP_PRIVL, CZC_ANY, "Sudovo" },
};

/* Considered a stem for choose/allow matching purposes. */
static const CzechNameSubst _name_czech_subst_full[] = {
	{ CZG_SMASC, CZA_ALL, CZC_COLOR, "Sedlec" },
	{ CZG_SMASC, CZA_ALL, CZC_COLOR, "Brod" },
	{ CZG_SMASC, CZA_ALL, CZC_COLOR, "Brod" },
	{ CZG_SMASC, CZA_ALL, CZC_NONE, u8"\u00daval" },
	{ CZG_SMASC, CZA_ALL, CZC_COLOR, u8"\u017d\u010f\u00e1r" },
	{ CZG_SMASC, CZA_ALL, CZC_COLOR, "Smrk" },
	{ CZG_SFEM,  CZA_ALL, CZC_COLOR, "Hora" },
	{ CZG_SFEM,  CZA_ALL, CZC_COLOR, "Lhota" },
	{ CZG_SFEM,  CZA_ALL, CZC_COLOR, "Lhota" },
	{ CZG_SFEM,  CZA_ALL, CZC_COLOR, "Hlava" },
	{ CZG_SFEM,  CZA_ALL, CZC_COLOR, u8"L\u00edpa" },
	{ CZG_SNEUT, CZA_ALL, CZC_COLOR, "Pole" },
	{ CZG_SNEUT, CZA_ALL, CZC_COLOR, u8"\u00dadol\u00ed" },
	{ CZG_PMASC, CZA_ALL, CZC_NONE, u8"\u00davaly" },
	{ CZG_PFEM,  CZA_ALL, CZC_COLOR, "Luka" },
	{ CZG_PNEUT, CZA_ALL, CZC_COLOR, "Pole" },
};

/* TODO: More stems needed. --pasky */
static const CzechNameSubst _name_czech_subst_stem[] = {
	{ CZG_SMASC,             CZA_MIDDLE,            CZC_COLOR, "Kostel" },
	{ CZG_SMASC,             CZA_MIDDLE,            CZC_COLOR, u8"Kl\u00e1\u0161ter" },
	{ CZG_SMASC, CZA_SHORT,                         CZC_COLOR, "Lhot" },
	{ CZG_SFEM,  CZA_SHORT,                         CZC_COLOR, "Lhot" },
	{ CZG_SFEM,  CZA_SHORT,                         CZC_COLOR, "Hur" },
	{ CZG_FREE,              CZA_MIDDLE | CZA_LONG, CZC_NONE, "Sedl" },
	{ CZG_FREE,  CZA_SHORT | CZA_MIDDLE | CZA_LONG, CZC_COLOR, "Hrad" },
	{ CZG_NFREE,             CZA_MIDDLE,            CZC_NONE, "Pras" },
	{ CZG_NFREE,             CZA_MIDDLE,            CZC_NONE, u8"Ba\u017e" },
	{ CZG_NFREE,             CZA_MIDDLE,            CZC_NONE, "Tes" },
	{ CZG_NFREE,             CZA_MIDDLE,            CZC_NONE, u8"U\u017e" },
	{ CZG_NFREE,             CZA_MIDDLE | CZA_LONG, CZC_POSTFIX, u8"B\u0159" },
	{ CZG_NFREE,             CZA_MIDDLE | CZA_LONG, CZC_NONE, "Vod" },
	{ CZG_NFREE,             CZA_MIDDLE | CZA_LONG, CZC_NONE, "Jan" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Prach" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Kunr" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Strak" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, u8"V\u00edt" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, u8"Vy\u0161" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, u8"\u017dat" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, u8"\u017der" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, u8"St\u0159ed" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Harv" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Pruh" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Tach" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, u8"P\u00edsn" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Jin" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Jes" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Jar" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Sok" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Hod" },
	{ CZG_NFREE,                          CZA_LONG, CZC_NONE, "Net" },
	{ CZG_FREE,                           CZA_LONG, CZC_NONE, u8"Pra\u017e" },
	{ CZG_FREE,                           CZA_LONG, CZC_NONE, "Nerat" },
	{ CZG_FREE,                           CZA_LONG, CZC_NONE, "Kral" },
	{ CZG_FREE,                           CZA_LONG, CZC_NONE, "Hut" },
	{ CZG_FREE,                           CZA_LONG, CZC_NOPOSTFIX, "Pan" },
	{ CZG_FREE,  CZA_SHORT | CZA_MIDDLE | CZA_LONG, CZC_NOPOSTFIX, u8"Odst\u0159ed" },
	{ CZG_FREE,  CZA_SHORT | CZA_MIDDLE | CZA_LONG, CZC_COLOR, "Mrat" },
	{ CZG_FREE,                           CZA_LONG, CZC_COLOR, "Hlav" },
	{ CZG_FREE,  CZA_SHORT | CZA_MIDDLE,            CZC_NONE, u8"M\u011b\u0159" },
	{ CZG_FREE,              CZA_MIDDLE | CZA_LONG, CZC_NONE, "Lip" },
};

/* Optional postfix inserted between stem and ending. */
static const char * const _name_czech_subst_postfix[] = {
	"av", "an", "at",
	"ov", "on", "ot",
	"ev", "en", "et",
};

/* This array must have the both neutral genders at the end! */
static const CzechNameSubst _name_czech_subst_ending[] = {
	{ CZG_SMASC, CZA_SHORT | CZA_MIDDLE,            CZC_ANY, "ec" },
	{ CZG_SMASC, CZA_SHORT | CZA_MIDDLE,            CZC_ANY, u8"\u00edn" },
	{ CZG_SMASC, CZA_SHORT | CZA_MIDDLE | CZA_LONG, CZC_ANY, "ov" },
	{ CZG_SMASC, CZA_SHORT       |        CZA_LONG, CZC_ANY, "kov" },
	{ CZG_SMASC,                          CZA_LONG, CZC_POSTFIX, u8"\u00edn" },
	{ CZG_SMASC,                          CZA_LONG, CZC_POSTFIX, u8"n\u00edk" },
	{ CZG_SMASC,                          CZA_LONG, CZC_ANY, "burk" },
	{ CZG_SFEM,  CZA_SHORT,                         CZC_ANY, "ka" },
	{ CZG_SFEM,              CZA_MIDDLE,            CZC_ANY, "inka" },
	{ CZG_SFEM,              CZA_MIDDLE,            CZC_ANY, u8"n\u00e1" },
	{ CZG_SFEM,                           CZA_LONG, CZC_ANY, "ava" },
	{ CZG_PMASC,                          CZA_LONG, CZC_POSTFIX, u8"\u00edky" },
	{ CZG_PMASC,                          CZA_LONG, CZC_ANY, "upy" },
	{ CZG_PMASC,                          CZA_LONG, CZC_ANY, "olupy" },
	{ CZG_PFEM,                           CZA_LONG, CZC_ANY, "avy" },
	{ CZG_PFEM,  CZA_SHORT | CZA_MIDDLE | CZA_LONG, CZC_ANY, "ice" },
	{ CZG_PFEM,  CZA_SHORT | CZA_MIDDLE | CZA_LONG, CZC_ANY, u8"i\u010dky" },
	{ CZG_PNEUT, CZA_SHORT | CZA_MIDDLE,            CZC_ANY, "na" },
	{ CZG_SNEUT, CZA_SHORT | CZA_MIDDLE,            CZC_ANY, "no" },
	{ CZG_SNEUT,                          CZA_LONG, CZC_ANY, u8"i\u0161t\u011b" },
};

static const char * const _name_czech_suffix[] = {
	"nad Cidlinou",
	u8"nad Dyj\u00ed",
	"nad Jihlavou",
	"nad Labem",
	"nad Lesy",
	"nad Moravou",
	"nad Nisou",
	"nad Odrou",
	u8"nad Ostravic\u00ed",
	u8"nad S\u00e1zavou",
	"nad Vltavou",
	u8"pod Prad\u011bdem",
	u8"pod Radho\u0161t\u011bm",
	u8"pod \u0158\u00edpem",
	u8"pod Sn\u011b\u017ekou",
	u8"pod \u0160pi\u010d\u00e1kem",
	"pod Sedlem",
	u8"v \u010cech\u00e1ch",
	u8"na Morav\u011b",
};



static const char * const _name_romanian_real[] = {
	"Adjud",
	"Alba Iulia",
	"Alexandria",
	"Babadag",
	u8"Bac\u0103u",
	"Baia Mare",
	u8"B\u0103ile Herculane",
	u8"B\u0103ilesti",
	u8"B\u00e2rlad",
	"Bicaz",
	u8"Bistri\u021ba",
	"Blaj",
	"Borsec",
	u8"Boto\u0219ani",
	u8"Br\u0103ila",
	u8"Bra\u021bov",
	u8"Bucure\u0219ti",
	"Buftea",
	u8"Buz\u0103u",
	u8"C\u0103l\u0103rasi",
	u8"Caransebe\u0219",
	u8"Cernavod\u0103",
	"Cluj-Napoca",
	u8"Constan\u021ba",
	"Covasna",
	"Craiova",
	"Dej",
	"Deva",
	"Dorohoi",
	"Drobeta-Turnu Severin",
	u8"Dr\u0103g\u0103sani",
	u8"F\u0103g\u0103ras",
	u8"F\u0103lticeni",
	u8"Fete\u0219ti",
	u8"Foc\u0219ani",
	u8"Gala\u0219i",
	"Gheorgheni",
	"Giurgiu",
	u8"H\u00e2r\u0219ova",
	"Hunedoara",
	u8"Hu\u0219i",
	u8"Ia\u0219i",
	"Isaccea",
	"Lugoj",
	u8"M\u0103cin",
	"Mangalia",
	"Medgidia",
	u8"Media\u0219",
	"Miercurea Ciuc",
	"Mizil",
	"Motru",
	u8"N\u0103s\u0103ud",
	u8"N\u0103vodari",
	u8"Odobe\u0219ti",
	u8"Olteni\u021ba",
	u8"One\u0219ti",
	"Oradea",
	u8"Or\u0219ova",
	u8"Petro\u0219ani",
	u8"Piatra Neam\u021b",
	u8"Pite\u0219ti",
	u8"Ploie\u0219ti",
	"Predeal",
	u8"R\u00e2mnicu V\u00e2lcea",
	"Reghin",
	u8"Re\u0219i\u021ba",
	"Roman",
	u8"Ro\u0219iorii de Vede",
	"Satu Mare",
	u8"Sebe\u0219",
	u8"Sf\u00e2ntu Gheorghe",
	"Sibiu",
	u8"Sighi\u0219oara",
	"Sinaia",
	"Slatina",
	"Slobozia",
	"Sovata",
	"Suceava",
	"Sulina",
	u8"T\u0103nd\u0103rei",
	u8"T\u00e2rgoviste",
	u8"T\u00e2rgu Jiu",
	u8"T\u00e2rgu Mures",
	"Tecuci",
	u8"Timi\u0219oara",
	"Tulcea",
	"Turda",
	u8"Turnu M\u0103gurele",
	"Urziceni",
	"Vaslui",
	"Vatra Dornei",
	"Victoria",
	"Videle",
	u8"Zal\u0103u",
};

static const char * const _name_slovak_real[] = {
	"Bratislava",
	u8"B\u00e1novce nad Bebravou",
	u8"Bansk\u00e1 Bystrica",
	u8"Bansk\u00e1 \u0160tiavnica",
	"Bardejov",
	"Brezno",
	"Brezova pod Bradlom",
	u8"Byt\u010da",
	u8"\u010cadca",
	u8"\u010cierna nad Tisou",
	"Detva",
	u8"Pre\u0161ov",
	"Dolny Kubin",
	u8"Spi\u0161sk\u00e1 Nov\u00e1 Ves",
	u8"Dunajsk\u00e1 Streda",
	u8"Gab\u010d\u00edikovo",
	"Galanta",
	"Gbely",
	"Gelnica",
	u8"Handlov\u00e1",
	"Hlohovec",
	u8"Hol\u00ed\u010d",
	u8"Humenn\u00e9",
	"Hurbanovo",
	"Kezmarok",
	u8"Kom\u00e1rno",
	u8"Ko\u0161ice",
	"Kremnica",
	"Krompachy",
	"Kuty",
	"Leopoldov",
	u8"Levo\u010da",
	u8"Liptovsk\u00fd Mikul\u00e1\u0161",
	u8"Lu\u010denec",
	"Malacky",
	"Martin",
	"Medzilaborce",
	"Michalovce",
	"Modra",
	"Myjava",
	u8"N\u00e1mestovo",
	"Nitra",
	u8"Nov\u00e1 Ba\u0148a",
	u8"Nov\u00e9 Mesto nad V\u00e1hom",
	u8"Nov\u00e9 Z\u00e1mky",
	u8"Partiz\u00e1nske",
	"Pezinok",
	u8"Pie\u0161\u0165any",
	u8"Polt\u00e1r",
	"Poprad",
	u8"Pova\u017esk\u00e1 Bystrica",
	"Prievidza",
	u8"P\u00fachov",
	u8"Rev\u00faca",
	u8"Rimavsk\u00e1 Sobota",
	u8"Ro\u017e\u0148ava",
	u8"Ru\u017eomberok",
	"Sabinov",
	u8"\u0160a\u013ea",
	"Senec",
	"Senica",
	u8"Sere\u010f",
	"Skalica",
	u8"Sl\u00e1dkovi\u00cdovo",
	"Smolenice",
	"Snina",
	u8"Star\u00e1 \u013dubov\u0148a",
	u8"Star\u00e1 Tur\u00e1",
	u8"Str\u00e1\u017eske",
	"Stropkov",
	"Stupava",
	u8"\u0160t\u00farovo",
	u8"\u0120ulekovo",
	u8"Topo\u013e\u010dany",
	u8"Trebi\u0161ov",
	u8"Tren\u010d\u00edn",
	"Trnava",
	u8"Tur\u010dianske Teplice",
	u8"Tvrdo\u0161\u00edn",
	u8"Vr\u00e1ble",
	u8"Vranov nad Top\u013eou",
	u8"Z\u00e1horsk\u00e1 Bystrica",
	u8"\u017ddiar",
	u8"\u013diar nad Hronom",
	u8"\u017dilina",
	u8"Zlat\u00e9 Moravce",
	"Zvolen",
};

static const char * const _name_norwegian_1[] = {
	"Arna",
	"Aust",
	u8"Bj\u00f8rk",
	u8"Bj\u00f8rn",
	"Brand",
	u8"B\u00f8ver",
	"Drag",
	u8"Dr\u00f8",
	"Eids",
	"Egge",
	"Fager",
	"Finns",
	"Flat",
	"Foll",
	"Foss",
	"Fugle",
	"Furu",
	"Gaus",
	"Galte",
	"Geir",
	u8"Gl\u00f8s",
	"Gran",
	"Grind",
	"Grims",
	u8"Gr\u00f8n",
	u8"Gr\u00f8t",
	"Gulle",
	"Haka",
	"Hammer",
	"Haug",
	"Hol",
	"Hon",
	"Hop",
	"Hov",
	"Jess",
	"Kabel",
	"Kjerns",
	"Kjerring",
	"Knatte",
	"Krok",
	u8"K\u00f8y",
	"Lang",
	"Lauv",
	"Leir",
	"Lund",
	"Logn",
	"Lo",
	"Lyng",
	u8"L\u00f8n",
	"Mesna",
	"Mel",
	"Mo",
	"Nar",
	"Nitte",
	"Nord",
	"Odd",
	"Ola",
	"Otte",
	"Ran",
	"Rev",
	"Rog",
	"Roms",
	"Rosen",
	"Sand",
	"Sau",
	"Sel",
	"Sol",
	"Sjur",
	u8"Sk\u00e5r",
	u8"Sl\u00e5tt",
	u8"Stj\u00f8r",
	"Stor",
	"Svart",
	"Svens",
	"Svin",
	"Sylte",
	"Syn",
	"Tran",
	"Vass",
	"Ved",
	"Vest",
	"Vesle",
	"Vik",
	u8"V\u00e5g",
};

static const char * const _name_norwegian_2[] = {
	"aker",
	"anger",
	"bakken",
	"bekk",
	"berg",
	"botn",
	"breen",
	"bu",
	"bugen",
	"by",
	"bygd",
	u8"b\u00f8",
	"dal",
	"egga",
	"eid",
	"elv",
	"enga",
	"foss",
	"fjell",
	"fjord",
	"foten",
	"gard",
	"grend",
	"hammer",
	"haug",
	"havn",
	"heim",
	"hella",
	"hovda",
	u8"h\u00f8a",
	u8"h\u00f8gda",
	"kampen",
	u8"kj\u00f8len",
	"kollen",
	"kroken",
	"land",
	"lia",
	"mark",
	"moen",
	"myr",
	"nes",
	"nuten",
	"osen",
	"rike",
	"rud",
	"sand",
	"set",
	u8"sj\u00f8en",
	"skogen",
	"slette",
	"snipa",
	"stad",
	"stua",
	"stulen",
	"sund",
	"svingen",
	u8"s\u00e6tra",
	"tinden",
	"tun",
	"vang",
	"vatn",
	"veid",
	"vik",
	"voll",
	u8"v\u00e5g",
	"um",
	u8"\u00e5sen",
};

static const char * const _name_norwegian_real[] = {
	"Alta",
	"Arendal",
	"Askim",
	"Bergen",
	u8"Bod\u00f8",
	"Brevik",
	"Bryne",
	u8"Br\u00f8nn\u00f8ysund",
	"Drammen",
	u8"Dr\u00f8bak",
	"Egersund",
	"Elverum",
	"Farsund",
	"Fauske",
	"Finnsnes",
	"Flekkefjord",
	"Flora",
	u8"Fosnav\u00e5g",
	"Fredrikstad",
	u8"F\u00f8rde",
	u8"Gj\u00f8vik",
	"Grimstad",
	"Halden",
	"Hamar",
	"Hammerfest",
	"Harstad",
	"Haugesund",
	"Holmestrand",
	"Horten",
	u8"J\u00f8rpeland",
	"Kirkenes",
	"Kolvereid",
	"Kongsberg",
	"Kongsvinger",
	"Kopervik",
	u8"Krager\u00f8",
	"Kristiansand",
	"Kristiansund",
	"Langesund",
	"Larvik",
	"Leirvik",
	"Leknes",
	"Levanger",
	"Lillehammer",
	"Lillesand",
	u8"Lillestr\u00f8m",
	"Lyngdal",
	u8"L\u00f8renskog",
	"Mandal",
	"Mo i Rana",
	"Molde",
	u8"Mosj\u00f8en",
	"Moss",
	"Mysen",
	u8"M\u00e5l\u00f8y",
	"Namsos",
	"Narvik",
	"Notodden",
	"Odda",
	"Oslo",
	"Otta",
	"Porsgrunn",
	"Ringerike",
	u8"Ris\u00f8r",
	"Rjukan",
	"Sandefjord",
	"Sandnes",
	u8"Sandnessj\u00f8en",
	"Sandvika",
	"Sarpsborg",
	"Sauda",
	"Ski",
	"Skien",
	"Skudeneshavn",
	"Sortland",
	"Stathelle",
	"Stavanger",
	"Steinkjer",
	u8"Stj\u00f8rdal",
	"Stokmarknes",
	"Stord",
	"Svelvik",
	u8"Svolv\u00e6r",
	u8"Troms\u00f8",
	"Trondheim",
	"Tvedestrand",
	u8"T\u00f8nsberg",
	"Ulsteinvik",
	u8"Vads\u00f8",
	u8"Vard\u00f8",
	u8"Verdals\u00f8ra",
	u8"\u00c5krehamn",
	u8"\u00c5lesund",
	u8"\u00c5ndalsnes",
};

static const char * const _name_hungarian_1[] = {
	"Nagy-",
	"Kis-",
	u8"Fels\u0151-",
	u8"Als\u00f3-",
	u8"\u00daj-",
};

static const char * const _name_hungarian_2[] = {
	"Bodrog",
	u8"Dr\u00e1va",
	"Duna",
	u8"Hej\u0151",
	u8"Hern\u00e1d",
	u8"R\u00e1ba",
	u8"Saj\u00f3",
	"Szamos",
	"Tisza",
	"Zala",
	"Balaton",
	u8"Fert\u0151",
	"Bakony",
	u8"Cserh\u00e1t",
	"Bihar",
	u8"Hajd\u00fa",
	u8"J\u00e1sz",
	"Kun",
	"Magyar",
	u8"N\u00f3gr\u00e1d",
	u8"Ny\u00edr",
	"Somogy",
	u8"Sz\u00e9kely",
	"Buda",
	u8"Gy\u0151r",
	"Pest",
	u8"Feh\u00e9r",
	u8"Cser\u00e9p",
	u8"Erd\u0151",
	"Hegy",
	"Homok",
	u8"Mez\u0151",
	"Puszta",
	u8"S\u00e1r",
	u8"Cs\u00e1sz\u00e1r",
	"Herceg",
	u8"Kir\u00e1ly",
	"Nemes",
	u8"P\u00fcsp\u00f6k",
	"Szent",
	u8"Alm\u00e1s",
	u8"Szilv\u00e1s",
	"Agg",
	"Aranyos",
	u8"B\u00e9k\u00e9s",
	u8"Egyh\u00e1zas",
	"Gagy",
	"Heves",
	"Kapos",
	u8"T\u00e1pi\u00f3",
	"Torna",
	"Vas",
	u8"V\u00e1mos",
	u8"V\u00e1s\u00e1ros",
};

static const char * const _name_hungarian_3[] = {
	u8"ap\u00e1ti",
	u8"b\u00e1ba",
	"bikk",
	"dob",
	"fa",
	u8"f\u00f6ld",
	"hegyes",
	"kak",
	"kereszt",
	u8"k\u00fcrt",
	u8"lad\u00e1ny",
	u8"m\u00e9rges",
	"szalonta",
	"telek",
	"vas",
	u8"v\u00f6lgy",
};

static const char * const _name_hungarian_4[] = {
	"alja",
	u8"egyh\u00e1za",
	u8"h\u00e1za",
	u8"\u00far",
	u8"v\u00e1r",
};

static const char * const _name_hungarian_real[] = {
	"Ajka",
	u8"Asz\u00f3d",
	"Badacsony",
	"Baja",
	"Budapest",
	"Debrecen",
	"Eger",
	u8"Fony\u00f3d",
	u8"G\u00f6d\u00f6ll\u0151",
	u8"Gy\u0151r",
	"Gyula",
	"Karcag",
	u8"Kecskem\u00e9t",
	"Keszthely",
	u8"Kisk\u00f6re",
	"Kocsord",
	u8"Kom\u00e1rom",
	u8"K\u0151szeg",
	u8"Mak\u00f3",
	u8"Moh\u00e1cs",
	"Miskolc",
	u8"\u00d3zd",
	"Paks",
	u8"P\u00e1pa",
	u8"P\u00e9cs",
	u8"Polg\u00e1r",
	"Sarkad",
	u8"Si\u00f3fok",
	"Szeged",
	"Szentes",
	"Szolnok",
	"Tihany",
	"Tokaj",
	u8"V\u00e1c",
	u8"Z\u00e1hony",
	"Zirc",
};

static const char * const _name_swiss_real[] = {
	"Aarau",
	"Aesch",
	"Altdorf",
	"Arosa",
	"Appenzell",
	"Arbon",
	u8"Altst\u00e4tten",
	"Baar",
	"Baden",
	"Bellinzona",
	"Brig-Glis",
	"Bienne",
	"Bulle",
	"Binningen",
	"Burgdorf",
	"Bern",
	"Basel",
	u8"B\u00fclach",
	"Carouge",
	"Cham",
	"Chiasso",
	"Chur",
	"Davos",
	u8"Del\u00e9mont",
	"Dietikon",
	u8"D\u00fcbendorf",
	"Emmen",
	u8"Freienbach-Pf\u00e4ffikon",
	"Fribourg",
	"Frauenfeld",
	u8"Gen\u00e8ve",
	"Glarus",
	"Gossau",
	"Grenchen",
	"Herisau",
	"Horgen",
	"Horw",
	"Illnau-Effretikon",
	"Ittigen",
	"Jona",
	"Kriens",
	"Kloten",
	u8"K\u00f6niz",
	"Kreuzlingen",
	u8"K\u00fcsnacht",
	"Agen",
	"Lancy",
	"La Chaux-de-Fonds",
	"Lenzburg",
	"Lugano",
	"Langenthal",
	"Littau",
	"Le Locle",
	"La Neuveville",
	"Locarno",
	"Liestal",
	"La Tour-de-Peilz",
	"Lausanne",
	"Lyss",
	"Luzern",
	"Martigny",
	u8"M\u00fcnchenstein",
	"Meyrin",
	"Montreux",
	"Monthey",
	"Morges",
	"Murten",
	"Moutier",
	"Muttenz",
	u8"Neuch\u00e2tel",
	"Neuhausen am Rheinfall",
	"Nyon",
	"Olten",
	"Onex",
	"Opfikon",
	"Ostermundigen",
	"Payerne",
	"Peseux",
	"Prilly",
	"Pully",
	"Rapperswil",
	"Richterswil",
	"Regensdorf",
	"Rheinfelden",
	"Riehen",
	"Renens",
	"Romanshorn",
	"Rorschach",
	"Stans",
	"Schaffhausen",
	"Steffisburg",
	"St. Gallen",
	"Schlieren",
	"Sierre",
	"Solothurn",
	"St. Moritz",
	"Sion",
	"Spiez",
	u8"St\u00e4fa",
	"Sursee",
	"Schwyz",
	"Thalwil",
	u8"Th\u00f4nex",
	"Thun",
	"Uster",
	"Uzwil",
	"Vernier",
	"Volketswil",
	"Versoix",
	"Vevey",
	u8"W\u00e4denswil",
	"Wettingen",
	"Wil",
	"Wallisellen",
	"Winterthur",
	"Wohlen",
	"Worb",
	"Wetzikon",
	"Yverdon-les-Bains",
	"Zollikon",
	"Zofingen",
	u8"Z\u00fcrich",
	"Zug",
};

static const char * const _name_danish_1[] = {
	"Gamle ",
	"Lille ",
	"Nye ",
	"Store ",
	"Kirke ",
	u8"N\u00f8rre ",
	"Vester ",
	u8"S\u00f8nder ",
	u8"\u00d8ster ",
	"Hvide ",
	u8"H\u00f8je ",
	"Kongens ",
};

static const char * const _name_danish_2[] = {
	"Ager",
	"Alle",
	"Aske",
	"Balle",
	"Bede",
	"Birke",
	"Bjerring",
	u8"Bj\u00e6ver",
	"Blommens",
	"Blok",
	"Bolder",
	"Bred",
	"Charlotten",
	"Christians",
	"Danne",
	"Diana",
	"Es",
	"Fredens",
	"Frederiks",
	"Fugle",
	u8"F\u00e5re",
	"Gille",
	"Gis",
	"Givs",
	"Glams",
	"Glo",
	"Guld",
	"Had",
	"Haralds",
	"Hassel",
	"Hede",
	"Helle",
	"Hessel",
	"Hjorts",
	"Hol",
	"Horn",
	"Humle",
	u8"H\u00f8j",
	u8"H\u00f8r",
	"Is",
	"Jyde",
	u8"J\u00e6gers",
	"Karls",
	"Klov",
	"Kokke",
	"Kvist",
	"Lang",
	"Lange",
	"Mari",
	"Nord",
	"Ny",
	"Oks",
	"Ring",
	u8"R\u00f8de",
	"Rung",
	u8"R\u00f8r",
	"Rud",
	"Saks",
	"Salt",
	"Skam",
	"Silke",
	"Skod",
	u8"Sk\u00e6l",
	u8"Sk\u00e6r",
	"Sol",
	"Svend",
	"Svine",
	"Strand",
	"Stubbe",
	"Ting",
	u8"Tj\u00e6re",
	"Tore",
	"Uger",
	"Ulf",
	"Val",
	"Vand",
	"Vej",
	"Vor",
	u8"V\u00e6r",
	u8"\u00d8r",
	u8"\u00c5l",
};

static const char * const _name_danish_3[] = {
	"basse",
	"borg",
	"berg",
	"bro",
	"by",
	"havn",
	"strup",
	"holm",
	"hus",
	u8"k\u00f8bing",
	"lund",
	"lunde",
	"sund",
	"ovre",
	u8"h\u00f8j",
	"dal",
	"sted",
	"sten",
	u8"l\u00f8se",
	u8"r\u00f8d",
	"magle",
	u8"s\u00f8",
	"bjerg",
	u8"b\u00e6k",
	"drup",
	"lev",
	"bo",
	"lyst",
	"feld",
	"skov",
};

static const char * const _name_turkish_prefix[] = {
	u8"Ak\u00e7a",
	u8"Alt\u0131n",
	u8"Bah\u00e7e",
	"Boz",
	u8"B\u00fcy\u00fck",
	u8"\u00c7ay",
	u8"Do\u011fu",
	"Eski",
	u8"G\u00fczel",
	u8"K\u0131z\u0131l",
	u8"K\u00fc\u00e7\u00fck",
	"Orta",
	u8"Sar\u0131",
	"Sultan",
	"Ulu",
	"Yeni",
};

static const char * const _name_turkish_middle[] = {
	u8"aga\u00e7",
	"ayva",
	u8"\u00e7am",
	"elma",
	"kurt",
	"pazar",
	u8"yal\u0131",
};

static const char * const _name_turkish_suffix[] = {
	"dere",
	"hisar",
	"kale",
	"kaya",
	"kent",
	u8"k\u00f6y",
	"ova",
	u8"\u00f6z\u00fc",
	u8"\u00f6ren",
	"pazar",
	"saray",
	"tepe",
	"yer",
	"yurt",
};

static const char * const _name_turkish_real[] = {
	"Adana",
	u8"Ad\u0131yaman",
	"Afyon",
	u8"A\u011fr\u0131",
	"Amasya",
	"Antalya",
	"Artvin",
	u8"Bal\u0131kesir",
	"Bilecik",
	"Bitlis",
	"Bolu",
	"Burdur",
	"Bursa",
	u8"\u00c7anakkale",
	u8"\u00c7ank\u0131r\u0131",
	"Denizli",
	u8"Diyarbak\u0131r",
	"Edirne",
	u8"Elaz\u0131\u011f",
	"Erzurum",
	u8"Eski\u015fehir",
	"Giresun",
	u8"G\u00fcm\u00fc\u015fhane",
	"Hatay",
	"Isparta",
	u8"\u0130\u00e7el",
	u8"\u0130stanbul",
	u8"\u0130zmir",
	"Kars",
	"Kastamonu",
	"Kayseri",
	"Kirklareli",
	"Kocaeli",
	"Konya",
	u8"K\u00fctahya",
	"Malatya",
	"Manisa",
	u8"Kahramanmara\u015f",
	"Mardin",
	u8"Mu\u011fla",
	u8"Mu\u015f",
	u8"Nev\u015fehir",
	u8"Ni\u011fde",
	"Rize",
	"Sakarya",
	"Samsun",
	"Siirt",
	"Sinop",
	"Sivas",
	"Trabzon",
	u8"\u015eanl\u0131urfa",
	"Van",
	"Yozgat",
	"Zonguldak",
	"Aksaray",
	"Bayburt",
	"Karaman",
	u8"\u015e\u0131rnak",
	u8"Bart\u0131n",
	"Ardahan",
	u8"I\u011fd\u0131r",
	"Yalova",
	u8"Karab\u00fck",
	"Osmaniye",
	u8"D\u00fczce",
};

static const char * const _name_italian_real[] = {
	"Roma",
	"Milano",
	"Napoli",
	"Torino",
	"Venezia",
	"Firenze",
	"Palermo",
	"Genova",
	"Parma",
	"Bologna",
	"Bari",
	"Cagliari",
	"Sassari",
	"Pisa",
	"Aosta",
	"Brescia",
	"Verona",
	"Bolzano",
	"Padova",
	"Udine",
	"Trieste",
	"Livorno",
	"Ancona",
	"Perugia",
	"Pescara",
	"L'Aquila",
	"Campobasso",
	"Potenza",
	"Cosenza",
	"Reggio Calabria",
	"Catania",
	"Caltanisetta",
	"Agrigento",
	"La Spezia",
	"Modena",
	"Vicenza",
	"Mantova",
	"Cremona",
	"Piacenza",
	"Reggio Emilia",
	"Foggia",
	"Benevento",
	"Salerno",
	"Catanzaro",
	"Lecce",
	"Como",
	"Lecco",
	"Sondrio",
	"Trento",
	"Desenzano",
	"Cuneo",
	"Asti",
	"Lodi",
	"Novara",
	"Biella",
	"Vercelli",
	"Rieti",
	"Nuoro",
	"Oristano",
	"Matera",
	"Taranto",
	"Varese",
	"Bergamo",
	"Pavia",
	"Caserta",
	"Frosinone",
	"Latina",
	"Enna",
	"Ragusa",
	"Siracusa",
	"Pordenone",
	"Imperia",
	"Verbania",
	"Alessandria",
	"Messina",
	"Siena",
	"Arezzo",
	"Grosseto",
};

static const char * const _name_italian_pref[] = {
	"Alpe ",
	"Borgo ",
	"Cascina ",
	"Castel ",
	"Fonte ",
	"Forte ",
	"Malga ",
	"Pieve ",
	"Poggio ",
	"Rocca ",
	"Villa ",
	"Villar ",
};

static const char * const _name_italian_1m[] = {
	"Bel",
	"Borgo",
	"Bosco",
	"Campo",
	"Capo",
	"Casal",
	"Castel",
	"Colle",
	"Fiume",
	"Fonte",
	"Lago",
	"Mezzo",
	"Monte",
	"Mon",
	"Orto",
	"Passo",
	"Prato",
	"Poggio",
	"Ponte",
	"Pozzo",
	"Sasso",
	"Tra",
	"Tre",
	"Ver",
	"Vico",
};

static const char * const _name_italian_1f[] = {
	"Acqua",
	"Bra",
	"Cala",
	"Casa",
	"Chiesa",
	"Citta",
	"Civita",
	"Corte",
	"Costa",
	"Croce",
	"Fontana",
	"Grotta",
	"Guardia",
	"Mezza",
	"Palma",
	"Pietra",
	"Ripa",
	"Rocca",
	"Serra",
	"Torre",
	"Val",
	"Valle",
	"Villa",
};

static const char * const _name_italian_2[] = {
	"bell",
	"bianc",
	"cald",
	"chiar",
	"cort",
	"ferrat",
	"fier",
	"fredd",
	"gioios",
	"grec",
	"guzz",
	"lung",
	"long",
	"migli",
	"negr",
	"ner",
	"nov",
	"nuov",
	"ross",
	"rotond",
	"scur",
	"secc",
	"sett",
	"vecchi",
	"ventos",
	"vers",
	"viv",
};

static const char * const _name_italian_2i[] = {
	"",
	"breve",
	"brevi",
	"chiari",
	"ferro",
	"fieschi",
	"fiore",
	"fonte",
	"forte",
	"gate",
	"leone",
	"maggiore",
	"minore",
	"mole",
	"monte",
	"poli",
	"scuri",
	"terra",
	"te",
	"torrione",
	"vento",
	"verde",
	"versiere",
};


static const char * const _name_italian_3[] = {
	" Marittimo",
	" Marittima",
	" del Capo",
	" del Monte",
	" di Sopra",
	" di Sotto",
	" sui Monti",
	" dei Marmi",
	" dei Sassi",
	" delle Fonti",
	" sui Prati",
	" a Mare",
	" Superiore",
	" Inferiore",
	" Terme",
	" Alta",
	" Bassa",
	" Brianza",
	" Vesuviano",
	" Scrivia",
	" Ticino",
};

static const char * const _name_italian_river1[] = {
	" del",
	" sul",
	" al",
	" nel",
};

static const char * const _name_italian_river2[] = {
	"l'Adda",
	"l'Adige",
	"le Alpi",
	"l'Arno",
	" Bormida",
	" Brenta",
	"la Dora Baltea",
	" Lambro",
	" Mincio",
	" Naviglio",
	"l'Oglio",
	"l'Olona",
	"l'Ombrone",
	" Panaro",
	" Piave",
	" Po",
	" Reno",
	" Scrivia",
	" Secchia",
	" Serio",
	" Tagliamento",
	" Tanaro",
	" Taro",
	" Ticino",
	" Tevere",
};

static const char * const _name_catalan_real[] = {
	"Barcelona",
	"L'Hospitalet de Llobregat",
	"Cerdanyola",
	"Martorell",
	"Badalona",
	"Tarragona",
	"Lleida",
	"Girona",
	"Sabadell",
	"Terrassa",
	"Reus",
	"Valls",
	"Vic",
	"Vielha e Mijaran",
	"Amposta",
	"Tortosa",
	"Berga",
	"Olot",
	"Mollerussa",
	"Banyoles",
	"Figueres",
	"Balaguer",
	u8"Vilafranca del Pened\u00e8s",
	"La Seu d'Urgell",
	"El Pont de Suert",
	"Igualada",
	"Manresa",
	"Solsona",
	"Les Borges Blanques",
	"Tremp",
	"Sort",
	"Colera",
	"Portbou",
	"El Vendrell",
	"Falset",
	"Ripoll",
	"Cervera",
	"Gandesa",
	u8"Matar\u00f3",
	"Montblanc",
	u8"Vilanova i la Geltr\u00fa",
	u8"T\u00e0rrega",
	"Camprodon",
	u8"Campdev\u00e0nol",
	"Cambrils",
	"Begur",
	"Setcases",
	"Palafrugell",
	"Begues",
	"El Bruc",
	u8"Cadaqu\u00e9s",
	u8"Collbat\u00f3",
	u8"Cervell\u00f3",
	"Esparreguera",
	"Abrera",
	"Alp",
	"Das",
	"Cercs",
	"Manlleu",
	"El Masnou",
	"Molins de Rei",
	"Monistrol",
	"Rocallaura",
	u8"Rub\u00ed",
	"Ripollet",
	"Sitges",
	"Roses",
};

static const char * const _name_catalan_pref[] = {
	"El Pont de ",
	"Parets de ",
	"Canet de ",
	"Castellar de ",
	"Corbera de ",
	"Arenys de ",
	"Calella de ",
	"La Seu de ",
	"La Bisbal de ",
	"Torroella de ",
	"Port de ",
	"Vilafranca de ",
	"Vilanova de ",
	"Caldes de ",
	"La Conca de ",
	"Olesa de ",
	"La Roca de ",
	"Sant Esteve de ",
	"Sant Andreu de ",
	"Sant Jordi de ",
	"Sant Joan de ",
	"Sant Feliu de ",
	"Sant Quirze de ",
	u8"Sant Sadurn\u00ed de ",
	"Santa Coloma de ",
	"Santa Margarida de ",
	"Santa Maria de ",
	u8"Sant Mart\u00ed de ",
	"Sant Pere de ",
	u8"Sant Juli\u00e0 de ",
	u8"Sant Vicen\u00e7 de ",

};

static const char * const _name_catalan_1m[] = {
	"Torrent",
	"Cami",
	"Mont",
	"Bell",
	"Puig",
	"Riu",
};

static const char * const _name_catalan_1f[] = {
	"Pala",
	"Selva",
	"Vall",
	"Serra",
	"Torre",
	"Riba",
	"Cova",
	"Terra",
};

static const char * const _name_catalan_2m[] = {
	"alt",
	"baix",
	"fosc",
	"pelat",
	u8"vent\u00f3s",
	"negre",
	"roig",
	u8"gr\u00eds",
};

static const char * const _name_catalan_2f[] = {
	"baixa",
	"alta",
	"fosca",
	"clara",
	"negra",
	"roja",
	"grisa",
	"freda",
};

static const char * const _name_catalan_3[] = {
	u8" Desp\u00ed",
	" Desvern",
	u8" del Cam\u00ed",
	" de Mar",
	" de Dalt",
	" de Baix",
	u8" del Vall\u00e8s",
	u8" de Bergued\u00e0",
	" de Conflent",
	" de la Plana",
};

static const char * const _name_catalan_river1[] = {
	" d'Anoia",
	" de Ter",
	" de Llobregat",
	" d'Ebre",
	" de Segre",
	u8" de Francol\u00ed",
};
